package tauzaman.timestamp;

import tauzaman.timestamp.domain.*;
import tauzaman.temporaldatatypes.*;

/**
* The <code>Semantics</code> interface contains all the operations, such as
* arithmetic and comparison operations, that are available on 
* Anchorable and Unanchorable objects.
* This is an interface because there are lots of different semantics
* for even standard operations such as precedes, such as left-operand
* semantics where the operation is performed at the granularity of the left 
* operand.
* Below is an example of how one might use a semantics (assume that
* LeftOperandSemantics implements Semantics).
*  <pre>
*     LeftOperandSemantics ops = new LeftOperandSemantics();
*     Instant j = new Instant(new Granularity("months", 32));
*     Instant i = new Instant(new Granularity("days", 32));
*     Instant k = ops.add(i,j); // k is in the granularity of "days"
*  </pre>
* Just a few operations are currently represented to give an idea of the
* eventual interface.  All the operations are on Sets rather than 
* individual entities because an individual is just a singleton set.
*
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 10/10/02
* @see     tauzaman.temporaldatatypes.Anchorable
* @see     tauzaman.temporaldatatypes.Unanchorable
**/
public interface Semantics {
  /** 
  * Does one Anchorable precede another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @param result - ExtendedBoolean 
  * @return ExtendedBoolean 
  **/
  public ExtendedBoolean precedes(Anchorable alpha, Anchorable beta);

  /** 
  * Does one Anchorable overlap another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean overlaps(Anchorable alpha, Anchorable beta);

  /** 
  * Does one Anchorable meet another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean meets(Anchorable alpha, Anchorable beta);

  /** 
  * Does one Anchorable contain another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean contains(Anchorable alpha, Anchorable beta);

  /** 
  * Is one Anchorable equalTo another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean equalTo(Anchorable alpha, Anchorable beta);

  /** 
  * Displace (add) an Anchorable with an Unanchorable.  
  * @param alpha - displacee
  * @param beta - displacor
  * @return - (alpha + beta)
  **/
  public Anchorable add(Anchorable alpha, Unanchorable beta);

  /** 
  * Displace (add) an Unanchorable with an Anchorable.  
  * @param alpha - displacor
  * @param beta - displacee
  * @return - (beta + alpha)
  **/
  public Anchorable add(Unanchorable alpha, Anchorable beta);

  /** 
  * Displace (subtract) an Anchorable with an Unanchorable.  
  * @param alpha - displacee
  * @param beta - displacor
  * @return - (alpha - beta)
  **/
  public Anchorable subtract(Anchorable alpha, Unanchorable beta);

  /** 
  * Displace (subtract) an Unanchorable with an Anchorable.  
  * @param alpha - displacor
  * @param beta - displacee
  * @return - (beta - alpha)
  **/
  public Anchorable subtract(Unanchorable alpha, Anchorable beta);

  /** 
  * Negate an Unanchorable. Note this will mutate 
  * the result parameter.
  * @param alpha - Unanchorable to negate
  * @return - -alpha
  **/
  public Unanchorable negate(Unanchorable alpha);

  /** 
  * Multiply an Anchorable by a TimeValue.  
  * @param alpha - Anchorable to multiply
  * @param n - how many 
  * @return result - (alpha * n)
  **/
  public Unanchorable multiply(Unanchorable alpha, TimeValue n);

  /** 
  * Divide an Unanchorable by an Unanchorable. 
  * @param alpha - divisee
  * @param beta - divisor
  * @param result - (TimeValue) alpha/beta
  **/
  public TimeValue[] divide(Unanchorable alpha, Unanchorable beta);

  /** 
  * Divide an Unanchorable by a TimeValue. 
  * @param alpha - divisee
  * @param n - divisor
  * @param result - alpha/n
  **/
  public Unanchorable divide(Unanchorable alpha, TimeValue beta);

  ///** 
  //* Grab the first Anchorable in an Anchorable, and make a new Anchorable 
  //* with it.
  //* Note, this is not a mutator, it will create a new Anchorable.
  //* @param alpha - Anchorable to retrieve from
  //* @return first Anchorable in alpha
  //**/
  //public Anchorable first(Anchorable alpha);

  ///** 
  //* Grab the last Anchor in an Anchorable, and make a new Anchorable with it.
  //* Note, this is not a mutator, it will create a new Anchorable.
  //* @param alpha - Anchorable to retrieve from
  //* @return last Anchorable in alpha
  //**/
  //public Anchorable last(Anchorable alpha);

}
