package tauadt.timestamp.domain;

import java.lang.*;
import java.math.*;
import tauadt.timestamp.*;
import tauadt.timestamp.internal.*;
import tauadt.calendricsystem.*;

/**
* <code>TimeValue</code> class implements the arithmetic and 
* comparison operations
* for an underlying domain of now-relative, indeterminate, and
* determinate times that are bounded by "MIN" and "MAX" values.
* MIN (MAX) are before (after) the smallest (largest) times.  
* Operations on TimeValues
* usually construct a new TimeValue.
*
* For example, assuming a discrete domain of Granules implemented
* using BigIntegers (see the Granule class).
* <pre>
*    Granule oneGranule = new Granule(new BigInteger("1"));
*    Granule threeGranule = new Granule(new BigInteger("3"));
*    TimeValue max = new TimeValue(TimeValue.MAX);
*    TimeValue min = new TimeValue(TimeValue.MIN);
*    TimeValue one = new TimeValue(TimeValue.NORMAL, oneGranule);
*    TimeValue three = new TimeValue(TimeValue.NORMAL, threeGranule);
*    TimeValue result = max.add(min);
*    System.out.println(max.image());  // Prints [TimeValue MAX]
*    System.out.println(min.image()); // Prints [TimeValue MIN]
*    System.out.println(result.image()); // prints [TimeValue NORMAL 0]
*    result = one.add(three); 
*    System.out.println(one.image()); // Prints [TimeValue NORMAL 1]
*    System.out.println(three.image()); // Prints [TimeValue NORMAL 3]
*    System.out.println(result.image()); //Prints [TimeValue NORMAL 4]
* </pre>
*
* @author  Curtis Dyreson 
* @version 0.9, Jan/31/2003
* @see     tauadt.timestamp.domain.Granule
* @status  Implemented only for NORMAL, MAX, and MIN
**/
public class TimeValue {

  public  static final int MAX = 0;
  public  static final int MIN = 1;
  public  static final int NORMAL = 2;
  public  static final int NOW_RELATIVE = 3;
  public  static final int INDETERMINATE = 4;

  private static final TimeValue NOW = new TimeValue(NOW_RELATIVE);
  private static final TimeValue MAX_TIMEVALUE = new TimeValue(MAX);
  private static final TimeValue MIN_TIMEVALUE = new TimeValue(MIN);
  private static final TimeValue ZERO_TIMEVALUE = 
    new TimeValue(NORMAL, new Granule(new BigInteger("0")));
  private static final TimeValue ONE_TIMEVALUE = 
    new TimeValue(NORMAL, new Granule(new BigInteger("1")));
  private static final TimeValue MINUS_ONE_TIMEVALUE = 
    new TimeValue(NORMAL, new Granule(new BigInteger("-1")));

  private int kind;
  private Granule granules;

  /**
  * Construct a NOW_RELATIVE or NORMAL time value. 
  * @param kind - NORMAL or NOW_RELATIVE only
  * @param p - which Granule or time does this timeValue represent
  * @throws BadTimeValue - tried to construct something wierd
  **/ 
  public TimeValue(int kind, Granule p) {
    if (kind == NORMAL) {
      this.kind = kind;
      this.granules = p;
      }
    elsif (kind == NOW_RELATIVE) {
      this.kind = kind;
      this.granules = p;
      }
    else {
      // Bad kind passed
      Internal.Error("TimeValue Constructor, Can't use MAX or MIN");
      }
    }

  /**
  * Construct an INDETERMINATE time value with an unknown PMF.
  * @param lower - Granule that is the lower bound
  * @param upper - Granule that is the lower bound
  * @throws BadTimeValue - tried to construct something wierd
  **/ 
  public TimeValue(Granule lower, Granule upper) {
    }

  /**
  * Construct an INDETERMINATE time value with the desired PMF.
  * @param lower - Granule that is the lower bound
  * @param upper - Granule that is the lower bound
  * @param pmf - desired probability mass function
  * @throws BadTimeValue - tried to construct something wierd
  **/ 
  public TimeValue(Granule lower, Granule upper, ProbabilityMassFunction) {
    }

  /**
  * A special TimeValue only needs a kind 
  * @param kind - MAX or MIN only
  * @throws BadTimeValue - tried to construct something wierd
  **/ 
  public TimeValue(int kind) {
    if (kind == MAX) OR (kind == MIN) {
      this.kind = kind;
      this.granules = null;
      }
    // else throw bad time value
    }

  /**
  * Create a nice string image of a time value
  * @return String image of time value
  **/
  public String image() {
    switch (kind) {
      case MAX: return "[TimeValue MAX]";
      case MIN: return "[TimeValue MIN]";
      case NORMAL: return "[TimeValue NORMAL " + granules.image() + "]";
      default: Internal.Error("unknown TimeValue in TimeValue image");
      }
    return ""; //dummy return
    }

  /**
  * Accessor function to get the granules represented by this TimeValue.
  * @return the granules
  **/ 
  public Granule granules() {
    switch (kind) {
      case NORMAL: return granules;
      }
    Internal.Error("Can't get Granules for a Bound TimeValue");
    return null; //dummy return since compiler is brain-dead
    }

  /**
  * Scale from one granularity to another.  A TimeValue is a granularity-less
  * value.  Hence we need to know the from and to granularities for scaling
  * the value.  Furthermore, the scale may produce two TimeValues (an 
  * upper and lower bound).  In the implementation, a GranularityMapping
  * is constructed from the fromGranularity to the toGranularity.  Since
  * it may be expensive to construct the mapping, and typically mappings
  * are reused, the mapping object is cached for further use internally 
  * by this function.  Periodically, the cached mapping table may become large,
  * so we need to set a maximum size to destroy and recreate the table.
  * Next, the actual value is scaled by evaluating the mapping.  Note that
  * only NORMAL values are scaled, MIN and MAX values just result in MIN
  * and MAX.
  * NEED TO IMPLEMENT
  * @param fromGranularity  
  * @param toGranularity 
  * @return the scaled TimeValue
  **/
  public TimeValue[] scale(Granularity fromGranularity, Granularity toGranularity) {
    // First build a mapping, then scale
    return null;
    }

  /**
  * Cast from one granularity to another.  A TimeValue is a granularity-less
  * value.  Hence we need to know the from and to granularities for scaling
  * the value.  A cast produces only one TimeValue (the 
  * upper or lower bound?).  In the implementation, a GranularityMapping
  * is constructed from the fromGranularity to the toGranularity.  Since
  * it may be expensive to construct the mapping, and typically mappings
  * are reused, the mapping object is cached for further use internally 
  * by this function.  Periodically, the cached mapping table may become large,
  * so we need to set a maximum size to destroy and recreate the table.
  * Next, the actual value is scaled by evaluating the mapping.  Note that
  * only NORMAL values are scaled, MIN and MAX values just result in MIN
  * and MAX.
  * NEED TO IMPLEMENT
  * @param fromGranularity  
  * @param toGranularity 
  * @return the cast TimeValue
  **/
  public TimeValue cast(Granularity fromGranularity, Granularity toGranularity) {
    // First build a mapping, then scale
    return null;
    }

  /**
  * Get the hashCode for this TimeValue.
  * @return the hashCode
  **/ 
  public int hashCode(TimeValue other) {
    switch (kind) {
      case MAX: return MAX;
      case MIN: return MIN;
      case NORMAL: return granules.hashCode();
      }
    Internal.Error("unknown TimeValue in TimeValue equals");
    return 0; //dummy return since compiler is brain-dead
    }

  /**
  * Selector - retrieve the kind of the TimeValue
  * @return the kind
  **/ 
  public int kind() {
    return this.kind;
    }

  /**
  * The equality tester for Hashtables
  * @param other - The TimeValue to test for equality
  * @return does this == other?
  **/ 
  public boolean equals(TimeValue other) {
    if (other.kind() != kind) return false;
    switch (kind) {
      case MAX: return true;
      case MIN: return true;
      case NORMAL: return granules.equals(other.granules);
      }
    Internal.Error("unknown TimeValue in TimeValue equals");
    return false; //dummy return since compiler is brain-dead
    }

  public boolean equalTo(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return true;
        case MIN: return false;
        case NORMAL: return false;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return false;
        case MIN: return true;
        case NORMAL: return false;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return false;
        case MIN: return false;
        case NORMAL: return granules.equalTo(other.granules());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue equalTo");
    return false; //dummy return since compiler is brain-dead
    }

  public boolean lessThan(TimeValue other) {
    switch (kind) {
      case MAX: return false;
      case MIN: switch (other.kind()) {
        case MAX: return true;
        case MIN: return false;
        case NORMAL: return true;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return true;
        case MIN: return false;
        case NORMAL: return granules.lessThan(other.granules());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue lessThan");
    return false; //dummy return since compiler is brain-dead
    }

  public boolean lessThanOrEqualTo(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return true;
        case MIN: return false;
        case NORMAL: return false;
        } break;
      case MIN: return true;
      case NORMAL: switch (other.kind()) {
        case MAX: return true;
        case MIN: return false;
        case NORMAL: return granules.lessThanOrEqualTo(other.granules());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue lessThanOrEqualTo");
    return false; //dummy return since compiler is brain-dead
    }

  public boolean greaterThan(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return false;
        case MIN: return true;
        case NORMAL: return true;
        } break;
      case MIN: return false;
      case NORMAL: switch (other.kind()) {
        case MAX: return false;
        case MIN: return true;
        case NORMAL: return granules.greaterThan(other.granules());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue greaterThan");
    return false; //dummy return since compiler is brain-dead
    }

  public boolean greaterThanOrEqualTo(TimeValue other) {
    switch (kind) {
      case MAX: return true;
      case MIN: switch (other.kind()) {
        case MAX: return false;
        case MIN: return true;
        case NORMAL: return false;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return false;
        case MIN: return true;
        case NORMAL: return granules.greaterThanOrEqualTo(other.granules());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue greaterThanOrEqualTo");
    return false; //dummy return since compiler is brain-dead
    }

  public TimeValue subtract(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return ZERO_TIMEVALUE;
        case MIN: return MAX_TIMEVALUE;
        case NORMAL: return MAX_TIMEVALUE;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return MIN_TIMEVALUE;
        case MIN: return ZERO_TIMEVALUE;
        case NORMAL: return MIN_TIMEVALUE;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return MIN_TIMEVALUE;
        case MIN: return MAX_TIMEVALUE;
        case NORMAL: 
          return new TimeValue(NORMAL,granules.subtract(other.granules()));
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue subtract");
    return null;
    }

  public TimeValue add(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return MAX_TIMEVALUE;
        case MIN: return ZERO_TIMEVALUE;
        case NORMAL: return MAX_TIMEVALUE;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return ZERO_TIMEVALUE;
        case MIN: return MIN_TIMEVALUE;
        case NORMAL: return MIN_TIMEVALUE;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return MAX_TIMEVALUE;
        case MIN: return MIN_TIMEVALUE;
        case NORMAL: 
          return new TimeValue(NORMAL,granules.add(other.granules()));
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue add");
    return null;
    }

  public TimeValue negate() {
    switch (kind) {
      case MAX: return MIN_TIMEVALUE;
      case MIN: return MAX_TIMEVALUE;
      case NORMAL: 
        return new TimeValue(NORMAL, granules.negate());
      }
    Internal.Error("unknown TimeValue in TimeValue negate");
    return null;
    }

// Removed by Curt Jan-28-03, readd if we want multiplication
//  public TimeValue multiply(TimeValue other) {
//    switch (kind) {
//      case MAX: switch (other.kind()) {
//        case MAX: return MAX_TIMEVALUE;
//        case MIN: return MIN_TIMEVALUE;
//        case NORMAL: return MAX_TIMEVALUE;
//        } break;
//      case MIN: switch (other.kind()) {
//        case MAX: return MIN_TIMEVALUE;
//        case MIN: return MAX_TIMEVALUE;
//        case NORMAL: return MIN_TIMEVALUE;
//        } break;
//      case NORMAL: switch (other.kind()) {
//        case MAX: return MAX_TIMEVALUE;
//        case MIN: return MIN_TIMEVALUE;
//        case NORMAL:
//          return new TimeValue(NORMAL,granules.multiply(other.granules()));
//        }
//      }
//    Internal.Error("unknown TimeValue in TimeValue multiply");
//    return null;
//    }

// Removed by Curt Jan-28-03, readd if we want division
//  public TimeValue divide(TimeValue other) {
//    switch (kind) {
//      case MAX: switch (other.kind()) {
//        case MAX: return ONE_TIMEVALUE;
//        case MIN: return MINUS_ONE_TIMEVALUE;
//        case NORMAL: return ZERO_TIMEVALUE; // Should be MaxInt
//        } break;
//      case MIN: switch (other.kind()) {
//        case MAX: return MINUS_ONE_TIMEVALUE;
//        case MIN: return ONE_TIMEVALUE;
//        case NORMAL: return ZERO_TIMEVALUE; // Should be MinInt
//        } break;
//      case NORMAL: switch (other.kind()) {
//        case MAX: return ZERO_TIMEVALUE;
//        case MIN: return ZERO_TIMEVALUE;
//        case NORMAL: 
//          return new TimeValue(NORMAL, granules.divide(other.granules()));
//        } 
//      }
//    Internal.Error("unknown TimeValue in TimeValue divide");
//    return null;
//    }

}
