package tauzaman.timestamp;

import java.lang.*;
import tauzaman.timestamp.*;
import tauzaman.*;
import tauzaman.calendricsystem.Granularity;
import java.io.*;

/**
* A <code>Granule</code> is a pairing of a TimeValue and a
* Granularity, independent of a TauZamanService.  This class implements
* operations on Granules.  
* Granules can be used for both unanchored and anchored times.  
* If the time is anchored, the Granule represents the distance from the 
* granularity anchor point.  
* If the time is unanchored, the Granule just represents
* a distance.  Granules can be further classified as determinate or 
* indeterminate.  A determinate granule has a known, fixed distance.
* An indeterminate granule is some distance between a lower and upper 
* bound.  A probability mass function determines the probability of 
* each indeterminate alternative.
*
* @author  Curtis Dyreson
* @version 1.0, Mar/6/2003
* @see     tauzaman.timestamp.TimeValue
* @see     tauzaman.timestamp.ExtendedBoolean
**/
public class Granule implements Serializable{
  // Kind of this granule
  protected int kind;

  public final static int DETERMINATE = 0;
  public final static int INDETERMINATE = 1;
  public final static int NOWRELATIVE = 2;

  // Some useful ExtendedBoolean values 
  protected final static ExtendedBoolean FALSE_EB = ExtendedBoolean.FalseEB;
  protected final static ExtendedBoolean TRUE_EB = ExtendedBoolean.TrueEB;
  protected final static ExtendedBoolean MAYBE_EB = ExtendedBoolean.MaybeEB;

  // Granularity of this granule
  protected Granularity granularity = null;

  // A determinate granule is represented as a TimeValue distance (in granules)
  // An indeterminate granule is represented by lower and upper bounds.
  protected TimeValue lower;
  protected TimeValue upper;

  // An indeterminate granule also has a probability mass function.
  protected ProbabilityMassFunction pmf = null;

  /**
  * Create a determinate granule at the default Granularity
  * @param p - count of granules
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(TimeValue p) /* throws TauZamanException */{
    this.lower = p;
    this.upper = p;
    this.kind = DETERMINATE;
//    granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create a determinate granule at the specified Granularity
  * @param g - desired Granularity
  * @param p - count of granules
  **/ 
  public Granule(Granularity g, TimeValue p) {
    this.lower = p;
    this.upper = p;
    this.kind = DETERMINATE;
    this.granularity = g;
    }

  /**
  * Create a determinate granule at the default Granularity
  * @param p - count of granules
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(long p) /* throws TauZamanException */{
    this.lower = new TimeValue(p);
    this.upper = new TimeValue(p);
    this.kind = DETERMINATE;
    //granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create a determinate granule at the specified Granularity
  * @param g - desired Granularity
  * @param p - count of granules
  **/ 
  public Granule(Granularity g, long p) {
    this.lower = new TimeValue(p);
    this.upper = new TimeValue(p);
    this.granularity = g;
    this.kind = DETERMINATE;
    }

  /**
  * Create an indeterminate granule with an unknown PMF at the default Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(TimeValue lower, TimeValue upper)/* throws TauZamanException */{
    this.lower = lower;
    this.upper = upper;
    this.kind = INDETERMINATE;
    //granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create an indeterminate granule with an unknown PMF at the default Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(long lower, long upper) /* throws TauZamanException */{
    this.lower = new TimeValue(lower);
    this.upper = new TimeValue(upper);
    this.kind = INDETERMINATE;
    //granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create an indeterminate granule with an unknown PMF at the specified Granularity
  * @param g - desired Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  **/ 
  public Granule(Granularity g, TimeValue lower, TimeValue upper) {
    this.granularity = g;
    this.lower = lower;
    this.upper = upper;
    this.kind = INDETERMINATE;
    }

  /**
  * Create an indeterminate granule with an unknown PMF at the specified Granularity
  * @param g - desired Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  **/ 
  public Granule(Granularity g, long lower, long upper) {
    this.granularity = g;
    this.lower = new TimeValue(lower);
    this.upper = new TimeValue(upper);
    this.kind = INDETERMINATE;
    }

  /**
  * Create an indeterminate granule with the specified PMF at the default Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  * @param pmf - probability mass function
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(TimeValue lower, TimeValue upper, ProbabilityMassFunction pmf) /*throws TauZamanException */{
    this.lower = lower;
    this.upper = upper;
    this.kind = INDETERMINATE;
    this.pmf = pmf;
    //granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create an indeterminate granule with the specified PMF at the default Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  * @param pmf - probability mass function
  *
  * @throws TauZamanException if any abnormal condition occurs when 
  * setting this Granule's default Granularity
  **/ 
  public Granule(long lower, long upper, ProbabilityMassFunction pmf) /*throws TauZamanException*/{
    this.lower = new TimeValue(lower);
    this.upper = new TimeValue(upper);
    this.kind = INDETERMINATE;
    this.pmf = pmf;
    //granularity = TauZamanSystem.getActiveService().getActiveCalendricSystemDefaultGranularity();
    }

  /**
  * Create an indeterminate granule with the specified PMF at the specified Granularity
  * @param g - desired Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  * @param pmf - probability mass function
  **/ 
  public Granule(Granularity g, TimeValue lower, TimeValue upper, ProbabilityMassFunction pmf) {
    this.granularity = g;
    this.lower = lower;
    this.upper = upper;
    this.kind = INDETERMINATE;
    this.pmf = pmf;
    }

  /**
  * Create an indeterminate granule with the specified PMF at the specified Granularity
  * @param g - desired Granularity
  * @param lower - count of granules
  * @param upper - count of granules
  * @param pmf - probability mass function
  **/ 
  public Granule(Granularity g, long lower, long upper, ProbabilityMassFunction pmf) {
    this.granularity = g;
    this.lower = new TimeValue(lower);
    this.upper = new TimeValue(upper);
    this.kind = INDETERMINATE;
    this.pmf = pmf;
    }

  /**
  * Accessor - retrieve the granularity of this granule
  * @return the granularity
  **/
  public Granularity getGranularity() {
    return this.granularity;
    }

  /**
  * Accessor - retrieve the kind of this granule, INDETERMINATE or DETERMINATE
  * @return the kind
  **/
  public int getKind() {
    return kind;
    }

  /**
  * Accessor - retrieve the granules from a determinate granule (defaults
  * to lower bound for indeterminate granule)
  * @return TimeValue object that records the granules
  **/
  public TimeValue getGranule() {
    return lower;
    }

  /**
  * Accessor - retrieve the lower bound from an indeterminate granule (defaults
  * to granules for determinate granule)
  * @return TimeValue object that records the lower bound
  **/
  public TimeValue getLower() {
    return lower;
    }

  /**
  * Accessor - retrieve the upper bound from an indeterminate granule (defaults
  * to granules for determinate granule)
  * @return TimeValue object that records the lower bound
  **/
  public TimeValue getUpper() {
    switch (kind) {
      // The NOWRELATIVE case is handled by the NowRelativeGranule class
      case INDETERMINATE: return upper;
      case DETERMINATE: default: return lower;
      }
    }

  /**
  * Create a nice string image of a granule, alternative to toString()
  * @return String image of a granule
  **/
  public String image() {
    switch (kind) {
      case INDETERMINATE: 
        return "[Indeterminate " + lower.image() + "~" + upper.image() + "]";
// The NOWRELATIVE case is handled by the NowRelativeGranule class
//      case NOWRELATIVE: 
//        return "[NowRelative " + lower.image() + "]";
      case DETERMINATE: default:
        return "[Determinate " + lower.image() + "]";
      }
    }

  /**
  * Construct a Granule that has each bound decremented by 1
  * @return this - 1
  **/ 
  public Granule decrement() {
    switch (kind) {
      case INDETERMINATE: 
        return new Granule(this.granularity, lower.decrement(), upper.decrement());
// The NOWRELATIVE case is handled by the NowRelativeGranule class
//      case NOWRELATIVE: 
//        return new NowRelativeGranule(this.granularity, lower.decrement());
      case DETERMINATE: default:
        return new Granule(this.granularity, lower.decrement());
      }
    }

  /**
  * Construct a Granule that has each bound incremented by 1
  * @return this + 1
  **/ 
  public Granule increment() {
    switch (kind) {
      case INDETERMINATE: 
        return new Granule(this.granularity, lower.increment(), upper.increment());
// The NOWRELATIVE case is handled by the NowRelativeGranule class
//      case NOWRELATIVE: 
//        return new NowRelativeGranule(this.granularity, lower.increment());
      case DETERMINATE: default:
        return new Granule(this.granularity, lower.increment());
      }
    }

  /**
  * Negate the granule.  This method constructs a new granule that has both
  * both bounds negated.  Since the granule represents a distance, negation
  * is well-defined.  The PMF and granularity are unchanged.
  * @return a new Granule with the lower and upper bounds negated
  **/ 
  public Granule negate() {
    switch (kind) {
      case INDETERMINATE: 
        return new Granule(this.granularity, upper.negate(), lower.negate(), pmf); 
// The NOWRELATIVE case is handled by the NowRelativeGranule class
//      case NOWRELATIVE: 
//      return new NowRelativeGranule(lower.negate()); 
      case DETERMINATE: default:
        return new Granule(this.granularity, lower.negate()); 
      }
    }

  /**
  * Is this == other?  This returns an ExtendedBoolean value to
  * capture the comparison for indeterminate granules.  For determinate
  * granules, two granules are equal if they represent the same granule.
  * For indeterminate granules, the probability of their being equal must
  * exceed the plausibility.
  * @param other - The Granule to compare
  * @return does this == other?
  **/
  public ExtendedBoolean equalTo(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
        case NOWRELATIVE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) == 0) return TRUE_EB;
          return FALSE_EB;
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case NOWRELATIVE:
        case INDETERMINATE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) == 0) return TRUE_EB;
          return FALSE_EB;
        }
      }
    }


  /** 
  * Is this < other?  This returns an ExtendedBoolean value to
  * capture the comparison for indeterminate granules.  For determinate
  * granules, a granule is < if the count is no smaller.
  * For indeterminate granules, the probability of their being < must
  * exceed the plausibility.
  * @param other - The Granule to compare
  * @return does this < other?
  **/
  public ExtendedBoolean lessThan(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
        case NOWRELATIVE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) == -1) return TRUE_EB;
          return FALSE_EB;
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case NOWRELATIVE:
        case INDETERMINATE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) == -1) return TRUE_EB;
          return FALSE_EB;
        }
      }
    }

  /**
  * Is this <= other?  This returns an ExtendedBoolean value to
  * capture the comparison for indeterminate granules.  For determinate
  * granules, a granule is <= if the count is no smaller.
  * For indeterminate granules, the probability of their being <= must
  * exceed the plausibility.
  * @param other - The Granule to compare
  * @return does this <= other?
  **/
  public ExtendedBoolean lessThanOrEqualTo(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
        case NOWRELATIVE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) <= 0) return TRUE_EB;
          return FALSE_EB;
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case NOWRELATIVE:
        case INDETERMINATE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) <= 0) return TRUE_EB;
          return FALSE_EB;
        }
      }
    }

  /**
  * Is this > other?  This returns an ExtendedBoolean value to
  * capture the comparison for indeterminate granules.  For determinate
  * granules, a granule is > if the count is no smaller.
  * For indeterminate granules, the probability of their being > must
  * exceed the plausibility.
  * @param other - The Granule to compare
  * @return does this > other?
  **/
  public ExtendedBoolean greaterThan(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
        case NOWRELATIVE:
        case DETERMINATE: default: 
          if (this.lower.compareTo(other.getLower()) > 0) return TRUE_EB;
          return FALSE_EB; 
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case NOWRELATIVE:
        case INDETERMINATE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) > 0) return TRUE_EB;
          return FALSE_EB;
        }
      }
    }

  /**
  * Is this >= other?  This returns an ExtendedBoolean value to
  * capture the comparison for indeterminate granules.  For determinate
  * granules, a granule is >= if the count is no smaller.
  * For indeterminate granules, the probability of their being >= must
  * exceed the plausibility.
  * @param other - The Granule to compare
  * @return does this >= other?
  **/
  public ExtendedBoolean greaterThanOrEqualTo(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
        case NOWRELATIVE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) >= 0) return TRUE_EB;
          return FALSE_EB;
        }
      case DETERMINATE: default: switch (other.getKind()) { 
        case NOWRELATIVE:
        case INDETERMINATE:
        case DETERMINATE: default:
          if (this.lower.compareTo(other.getLower()) >= 0) return TRUE_EB;
          return FALSE_EB;
        }
      }
    }

  /**
  * Construct a new Granule that is this + other.  The addition is
  * similar to that in interval mathematics, i.e., [a,b] + [c,d] = [a+c, b+d],
  * where a and c are lower bounds and b and d are upper bounds of the
  * respective granules.  The granules must be at the same granularity.
  * The addition discards the PMF, that is, the PMF of the new granule
  * is (usually) unknown since the needed PMF is the convolution of
  * the two PMFs which is difficult/expensive to construct on the fly.
  * @param other - The Granule to add
  * return this + other
  **/
  public Granule add(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
          return new Granule(this.granularity,
                             lower.add(other.getLower()),
                             upper.add(other.getUpper())
                             );
        case NOWRELATIVE:
        case DETERMINATE: default:
          return new Granule(this.granularity,
                             lower.add(other.getLower()),
                             upper.add(other.getLower())
                             );
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case INDETERMINATE:
          return new Granule(this.granularity,
                             lower.add(other.getLower()),
                             lower.add(other.getUpper())
                             );
        case NOWRELATIVE:
        case DETERMINATE: default:
          return new Granule(this.granularity,
                             lower.add(other.getLower())
                            );
        }
      }
    }

  /**
  * Construct a new Granule that is this - other.  The addition is
  * similar to that in interval mathematics, i.e., [a,b] - [c,d] = [a-c, b-d],
  * where a and c are lower bounds and b and d are upper bounds of the
  * respective granules.  The granules must be at the same granularity.
  * The subtraction discards the PMF, that is, the PMF of the new granule
  * is (usually) unknown since the needed PMF is the convolution of
  * the two PMFs which is difficult/expensive to construct on the fly.
  * @param other - The Granule to subtract
  * return this - other
  **/
  public Granule subtract(Granule other) {
    switch (kind) {
      case INDETERMINATE: switch (other.getKind()) {
        case INDETERMINATE:
          return new Granule(this.granularity,
                             lower.subtract(other.getLower()),
                             upper.subtract(other.getUpper())
                             );
        case NOWRELATIVE:
        case DETERMINATE: default:
          return new Granule(this.granularity,
                             lower.subtract(other.getLower()),
                             upper.subtract(other.getLower())
                             );
        }
      case DETERMINATE: default: switch (other.getKind()) {
        case INDETERMINATE:
          return new Granule(this.granularity,
                             lower.subtract(other.getLower()),
                             lower.subtract(other.getUpper())
                             );
        case NOWRELATIVE:
        case DETERMINATE: default:
          return new Granule(this.granularity,
                             lower.subtract(other.getLower())
                            );
        }
      }
    }


  /**
  * Multiply by a constant.  This constructs a new Granule that is the
  * the multiplication of the existing granule by a constant.  Both bounds
  * are multiplied, but the PMF and granularity remain the same.
  * @param n - The divisor
  * @return a new Granule with the lower and upper bounds multiplied by n
  **/
  public Granule multiply(int n) {
    switch (kind) {
      case INDETERMINATE: 
        return new Granule(this.granularity,
                           lower.multiply(n),
                           upper.multiply(n),
                           this.pmf
                           );
      case NOWRELATIVE: case DETERMINATE: default: 
        return new Granule(this.granularity,
                           lower.multiply(n)
                          );
      }
    }

  /**
  * Divide by a constant.  This constructs a new Granule that is the
  * the division of the existing granule by a constant. Both bounds
  * are divided, but the PMF and granularity remain the same.
  * @param n - The divisor
  * @return a new Granule with the lower and upper bounds divided by n
  **/
  public Granule divide(int n) {
    switch (kind) {
      case INDETERMINATE:
        return new Granule(this.granularity,
                           lower.divide(n),
                           upper.divide(n),
                           this.pmf
                           );
      case NOWRELATIVE: case DETERMINATE: default:
        return new Granule(this.granularity,
                           lower.divide(n)
                          );
      }
    }
  /**
  * A simple test for the class.
  **/ 
  public static void main(String argv[]) throws Exception{
    TimeValue minTV = TimeValue.BEGINNING_OF_TIME;
    TimeValue maxTV = TimeValue.END_OF_TIME;
    TimeValue oneTV = new TimeValue(1);
    TimeValue threeTV = new TimeValue(3);

    Granule min = new Granule(minTV);
    Granule max = new Granule(maxTV);
    Granule one = new Granule(oneTV);
    Granule three = new Granule(threeTV);
    Granule oneToThree = new Granule(oneTV, threeTV);
    Granule minToMax = new Granule(minTV, maxTV);
    Granule result = null;

    System.out.println("max is " + max.image());  
    System.out.println("min is " + min.image()); 
    System.out.println("3 is " + three.image()); 
    System.out.println("1~3 is " + oneToThree.image()); 
    System.out.println("min~max is " + minToMax.image()); 

    // Negate tests
    System.out.println("Testing negate --------------------------------"); 

    System.out.print("negate min = ");
    result = min.negate();  System.out.println(result.image());

    System.out.print("negate max = ");
    result = max.negate();  System.out.println(result.image());

    System.out.print("negate 3 = ");
    result = three.negate();  System.out.println(result.image());

    System.out.print("negate 1~3 = ");
    result = oneToThree.negate();  System.out.println(result.image());

    System.out.print("negate min~max = ");
    result = minToMax.negate();  System.out.println(result.image());

    // Increment tests
    System.out.println("Testing increment --------------------------------"); 

    System.out.print("min++ = ");
    result = min.increment();  System.out.println(result.image());

    System.out.print("max++ = ");
    result = max.increment();  System.out.println(result.image());

    System.out.print("3++ = ");
    result = three.increment();  System.out.println(result.image());

    System.out.print("1~3++ = ");
    result = oneToThree.increment();  System.out.println(result.image());

    System.out.print("min~max++ = ");
    result = minToMax.increment();  System.out.println(result.image());

    // Decrement tests
    System.out.println("Testing decrement --------------------------------"); 

    System.out.print("min-- = ");
    result = min.decrement();  System.out.println(result.image());

    System.out.print("max-- = ");
    result = max.decrement();  System.out.println(result.image());

    System.out.print("3-- = ");
    result = three.decrement();  System.out.println(result.image());

    System.out.print("1~3-- = ");
    result = oneToThree.decrement();  System.out.println(result.image());

    System.out.print("min~max-- = ");
    result = minToMax.decrement();  System.out.println(result.image());

    // Addition tests
    System.out.println("Testing add --------------------------------"); 

    System.out.print("1 + 3 = ");
    result = one.add(three); System.out.println(result.image());

    System.out.print("3 + 1 = ");
    result = three.add(one); System.out.println(result.image());

    System.out.print("3 + 3 = ");
    result = three.add(three); System.out.println(result.image());

    System.out.println("Min as an interval means -max, i.e., negative all of time.");
    System.out.print("3 + min => 3 + (-max) => 3 - max = ");
    result = three.add(min); System.out.println(result.image());

    System.out.print("three + max = ");
    result = three.add(max); System.out.println(result.image());

    System.out.print("min + min => min + (-max) => min - max = ");
    result = min.add(min); System.out.println(result.image());

    System.out.print("min + max = ");
    result = min.add(max); System.out.println(result.image());

    System.out.print("max + min => max + (-max) => max - max =  ");
    result = max.add(min); System.out.println(result.image());

    System.out.print("max + max = ");
    result = max.add(max); System.out.println(result.image());

    // Subtraction tests
    System.out.println("Testing subtract --------------------------------"); 

    System.out.print("1 - 3 = ");
    result = one.subtract(three); System.out.println(result.image());

    System.out.print("3 - 1 = ");
    result = three.subtract(one); System.out.println(result.image());

    System.out.print("3 - 3 = ");
    result = three.subtract(three); System.out.println(result.image());

    System.out.print("3 - min => 3 - (-max) => 3 + max = ");
    result = three.subtract(min); System.out.println(result.image());

    System.out.print("3 - max = ");
    result = three.subtract(max); System.out.println(result.image());

    System.out.print("min - three = ");
    result = min.subtract(three); System.out.println(result.image());

    System.out.print("min - min => min - (-max) => min + max = ");
    result = min.subtract(min); System.out.println(result.image());

    System.out.print("min - max = ");
    result = min.subtract(max); System.out.println(result.image());

    System.out.print("max - 3 = ");
    result = max.subtract(three); System.out.println(result.image());

    System.out.print("max - min => max - (-max) => max + max = ");
    result = max.subtract(min); System.out.println(result.image());

    System.out.print("max - max = ");
    result = max.subtract(max); System.out.println(result.image());

    // Multiply tests
    System.out.println("Testing * --------------------------------"); 

    System.out.print("1 * 3 = ");
    result = one.multiply(3); System.out.println(result.image());

    System.out.print("3 * 1 = ");
    result = three.multiply(1); System.out.println(result.image());

    System.out.print("3 * 3 = ");
    result = three.multiply(3); System.out.println(result.image());

    System.out.print("min * 3 = ");
    result = min.multiply(3); System.out.println(result.image());

    System.out.print("max * 3 = ");
    result = max.multiply(3); System.out.println(result.image());

    // Divide tests
    System.out.println("Testing * --------------------------------"); 

    System.out.print("1 / 3 = ");
    result = one.divide(3); System.out.println(result.image());

    System.out.print("3 / 1 = ");
    result = three.divide(1); System.out.println(result.image());

    System.out.print("3 / 3 = ");
    result = three.divide(3); System.out.println(result.image());

    System.out.print("min / 3 = ");
    result = min.divide(3); System.out.println(result.image());

    System.out.print("max / 3 = ");
    result = max.divide(3); System.out.println(result.image());

    // Less than tests
    System.out.println("Testing < --------------------------------"); 

    System.out.print("Is 1 < max? "); 
    if (one.lessThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 1 < 3? "); 
    if (one.lessThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 < 3? "); 
    if (three.lessThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max < 3? "); 
    if (max.lessThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min < 3? "); 
    if (min.lessThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 < min? "); 
    if (three.lessThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 < max? "); 
    if (three.lessThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min < min? "); 
    if (min.lessThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max < min? "); 
    if (max.lessThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min < max? "); 
    if (min.lessThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max < max? "); 
    if (max.lessThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.println("Testing == --------------------------------"); 

    System.out.print("Is 1 == max? "); 
    if (one.equalTo(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 1 == 3? "); 
    if (one.equalTo(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 == 3? "); 
    if (three.equalTo(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max == 3? "); 
    if (max.equalTo(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min == 3? "); 
    if (min.equalTo(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 == min? "); 
    if (three.equalTo(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 == max? "); 
    if (three.equalTo(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min == min? "); 
    if (min.equalTo(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max == min? "); 
    if (max.equalTo(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min == max? "); 
    if (min.equalTo(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max == max? "); 
    if (max.equalTo(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.println("Testing > --------------------------------"); 

    System.out.print("Is 1 > max? "); 
    if (one.greaterThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 1 > 3? "); 
    if (one.greaterThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 > 3? "); 
    if (three.greaterThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max > 3? "); 
    if (max.greaterThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min > 3? "); 
    if (min.greaterThan(three).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 > min? "); 
    if (three.greaterThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is 3 > max? "); 
    if (three.greaterThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min > min? "); 
    if (min.greaterThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max > min? "); 
    if (max.greaterThan(min).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is min > max? "); 
    if (min.greaterThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 

    System.out.print("Is max > max? "); 
    if (max.greaterThan(max).satisfied()) System.out.println("Yes"); 
    else System.out.println("No"); 
    }

} // End of class
