package tauzaman.timestamp;

import tauzaman.timestamp.*;
import java.io.*;

/**
* <code>ExtendedBoolean</code> class implements the boolean algebra for 
* a Three-Valued Logic.
* For example:
* <pre>
*    ExtendedBoolean trueValue = new ExtendedBoolean(true);
*    ExtendedBoolean falseValue = new ExtendedBoolean(false);
*    // Maybe values are currently unsupported, but eventually...
*    ExtendedBoolean maybeValue = new ExtendedBoolean();
*    if ((trueValue.or(falseValue)).satisfied()) {
*       // works, should be satisfied
*       }
*    if (!maybeValue.satisfied()) {
*       // works, maybe is not satisfied
*       }
*    if (maybeValue.unsatisfied()) {
*       // works, maybe is not unsatisfied either!
*       }
* </pre>
*
* @author  Curtis Dyreson 
* @version 1.0, Mar/3/2003
**/
public class ExtendedBoolean implements Serializable{
  private final static int SATISFIED = 1;
  private final static int UNSATISFIED = 0;
  private final static int MAYBE = 2;
  private int status;
  public static ExtendedBoolean TrueEB = new ExtendedBoolean(true);
  public static ExtendedBoolean FalseEB = new ExtendedBoolean(false);
  public static ExtendedBoolean MaybeEB = new ExtendedBoolean();
  
  /**
  * Build a TRUE or FALSE value in the ExtendedBoolean logic.
  * @param satisfied - truth condition of the true part
  **/
  public ExtendedBoolean(boolean satisfied) {
    if (satisfied) this.status = SATISFIED;
    else this.status = UNSATISFIED;
    }
  
  /**
  * Build a MAYBE value in the ExtendedBoolean logic.
  **/
  public ExtendedBoolean() {
    this.status = MAYBE;
    }
  
  /**
  * And this one with another.
  * @param other - the other one
  * @return this && other
  **/
  public ExtendedBoolean and(ExtendedBoolean other) {
    switch (status) {
      case SATISFIED: 
        if (other.satisfied()) return TrueEB;
        if (other.unsatisfied()) return FalseEB;
        return MaybeEB;
      case UNSATISFIED: 
        if (other.satisfied()) return FalseEB;
        if (other.unsatisfied()) return FalseEB;
        return MaybeEB;
      case MAYBE: default:
        return this;
      }
    }

  /**
  * Or this one with another.
  * @param other - the other one
  * @return this || other
  **/
  public ExtendedBoolean or(ExtendedBoolean other) {
    switch (status) {
      case SATISFIED:
        return TrueEB;
      case UNSATISFIED:
        if (other.satisfied()) return TrueEB;
        if (other.unsatisfied()) return FalseEB;
        return MaybeEB;
      case MAYBE: default:
        if (other.satisfied()) return TrueEB;
        return this;
      }
    }

  /**
  * Not.
  * @return !this
  **/
  public ExtendedBoolean not(ExtendedBoolean other) {
    switch (status) {
      case SATISFIED: return FalseEB;
      case UNSATISFIED: return TrueEB;
      case MAYBE: default: return this;
      }
    }

  /**
  * Satisfied?
  * @return true if truth condition is represented
  **/
  public boolean satisfied() {
    return status == SATISFIED;
    }

  /**
  * Unsatisfied?
  * @return true if false condition is represented
  **/
  public boolean unsatisfied() {
    return status == UNSATISFIED;
    }

  /**
  * Maybe?
  * @return true if this is a maybe value
  **/
  public boolean maybe() {
    return status == MAYBE;
    }

} 
