package tauzaman.temporaldatatypes;

import java.util.*;
import java.math.*;

import tauzaman.timestamp.*;
import tauzaman.timestamp.domain.*;
import tauzaman.timestamp.internal.*;
import tauzaman.calendricsystem.*;

/**
* The <code>IndeterminateInstant</code> class provides a 
* set of constructor and selector operations on indeterminate instants.
* (Arithmetic and comparison operations are implemented in an
* Indeterminate Semantics.)
* An indeterminate instant is an time that is anchored sometime between
* two points on a time-line, e.g., "July 3, 2002 ~ July 14, 2002".
* The <em>lower bound</em> of the instant is "July 3, 2002".  
* The <em>upper bound</em> is "July 14, 2002".  
* The <em>period of indeterminacy</em> is the time between the lower and 
* upper bounds.
*
* To perform comparisons and arithmetic operations on IndeterminateInstants, 
* it is first necessary to create a @see IndeterminateSemantics interface.
* An IndeterminateSemantics implements common comparison and arithmetic 
* operations on temporal entities.  
* For example a user would write code like the following
*  <pre>
*     // Indeterminate left operand semantics scales operands in binary 
*     // operations to the granularity of the left operand, and then performs
*     // the desired operation. 
*     Ops = new IndeterminateLeftOperandSemantics();
*     // Operations in an indeterminate semantics are performed at a 
*     // plausibility and credibility.
*     Ops.setCredibility = 100;
*     Ops.setPlausibility = 50;
*     // Create two Instants, in this case, both are at the granularity of days
*     IndeterminateInstant i = new IndeterminateInstant("July 3, 2002 ~ July 14, 2002");
*     IndeterminateInstant j = new IndeterminateInstant("July 10, 2002 ~ July 13, 2002");
*     // Is i earlier on the time-line than j?
*     if (Ops.precedes(i,j).satisfied()) ...
*  </pre>
* We employ this strategy so that indeterminate instant operations can be
* performed under varying IndeterminateSemantics.
*
* @author  Curtis Dyreson 
* @version 0.9, Jan/31/2003
* @see     tauzaman.timestamp.IndeterminateSemantics
* @see     tauzaman.temporaldatatypes.IndeterminateAnchoredDataType
* @status  design complete, NOT IMPLEMENTED
**/
public class IndeterminateInstant extends IndeterminateAnchoredDataType {
  // The UNKNOWN_TIME indeterminate instant constant represents an instant that
  // is the sometime between the start of the time-line and the end.
  // It is a granularity-less constant.
  public static final Instant UNKNOWN_TIME = 
      new Instant(null, new TimeValue(TimeValue.MIN), new TimeValue(TimeValue.MAX));

  // IndeterminateInstants have an upper bound, a lower bound, and a pmf.
  private TimeValue lower;
  private TimeValue upper;
  private ProbabilityMassFunction pmf;

  // This is the granularity of the TimeValues.  
  // It replicates the granularity that is in both Granules.
  private Granularity g;

  /**
  * Construct an IndeterminateInstant with an unknown PMF from two 
  * Granules.  Scale the IndeterminateInstant to the desired granularity.
  * We check to ensure that the lower <= upper.
  * @param g - granularity
  * @param lower - lower bound, distance in granules from the anchor 
  * @param upper - upper bound, distance in granules from the anchor
  * @status NOT IMPLEMENTED, should throw a bad instant
  **/
  public Instant(Granularity g, Granule lower, Granule upper) {
    this.g = g;
    this.pmf = null;
    this.lower = new TimeValue(TimeValue.NORMAL, lower);
    this.upper = new TimeValue(TimeValue.NORMAL, upper);
    if (lower.greaterThan(upper)) { // die };
    }

  /**
  * Construct an IndeterminateInstant with an desired PMF from two 
  * Granules.  Scale the IndeterminateInstant to the desired granularity.
  * We check to ensure that the lower <= upper.
  * @param g - granularity
  * @param lower - lower bound, distance in granules from the anchor 
  * @param upper - upper bound, distance in granules from the anchor
  * @param pmf - probability mass function
  * @status NOT IMPLEMENTED, should throw a bad instant
  **/
  public Instant(Granularity g, Granule lower, Granule upper, ProbabilityMassFunction pmf) {
    this.g = g;
    this.pmf = pmf;
    this.lower = new TimeValue(TimeValue.NORMAL, lower);
    this.upper = new TimeValue(TimeValue.NORMAL, upper);
    if (lower.greaterThan(upper)) { // die };
    }

  /**
  * Construct an IndeterminateInstant by parsing a string, using the default granularity.
  * @param literal - a string representation of the indeterminate instant
  * @status NOT IMPLMENTED
  **/
  public static IndeterminateInstant(String literal) {
    this.g = g;
    }

  /**
  * Construct an IndeterminateInstant by parsing a string.  The resulting IndeterminateInstant will
  * be <em>scaled</em> to the desired Granularity.
  * @param literal - a string representation of the indeterminate instant
  * @param g - granularity
  * @status NOT IMPLMENTED
  **/
  public static IndeterminateInstant(String literal, Granularity g) {
    this.g = g;
    }

  /**
  * Construct an Indeterminate Instant with an unknown PMF 
  * from a pair of TimeValues.  
  * Scale the bounds to the desired granularity.
  * @param g - granularity
  * @param lower - TimeValue representing the lower bound
  * @param upper - TimeValue representing the upper bound
  * @status NOT IMPLEMENTED
  **/
  public Instant(Granularity g, TimeValue lower, TimeValue upper) {
    this.g = g;
    this.lower = lower;
    this.upper = upper;
    }

  /**
  * Construct an Indeterminate Instant with an unknown PMF 
  * from a pair of TimeValues.  
  * Scale the bounds to the default granularity.
  * @param lower - TimeValue representing the lower bound
  * @param upper - TimeValue representing the upper bound
  * @status NOT IMPLEMENTED
  **/
  public Instant(TimeValue lower, TimeValue upper) {
    this.g = null;
    this.lower = lower;
    this.upper = upper;
    }

  /**
  * Construct an Indeterminate Instant with an known PMF 
  * from a pair of TimeValues.  
  * Scale the bounds to the desired granularity.
  * @param g - granularity
  * @param lower - TimeValue representing the lower bound
  * @param upper - TimeValue representing the upper bound
  * @param pmf - probability mass function
  * @status NOT IMPLEMENTED
  **/
  public Instant(Granularity g, TimeValue lower, TimeValue upper, ProbabilityMassFunction pmf) {
    this.g = g;
    this.lower = lower;
    this.upper = upper;
    }

  /**
  * Construct an Indeterminate Instant with an known PMF 
  * from a pair of TimeValues.  
  * Scale the bounds to the default granularity.
  * @param lower - TimeValue representing the lower bound
  * @param upper - TimeValue representing the upper bound
  * @param pmf - probability mass function
  * @status NOT IMPLEMENTED
  **/
  public Instant(TimeValue lower, TimeValue upper, ProbabilityMassFunction pmf) {
    this.lower = lower;
    this.upper = upper;
    }

  /** 
  * Build a nice string image of an Indeterminate Instant, 
  * for debugging mostly.  Provides
  * an alternative to toString().
  * @return String image 
  * @status NOT IMPLEMENTED
  **/
  public String image() {
    return "[IndeterminateInstant lower " + lower.image() + " granularity " + 
             this.g.toString() + "]";
    }

  /** 
  * Generate the hash code value, needed for supporting Hashtables.
  * @return int 
  **/
  public int hashCode() {
    return lower.hashCode() + upper.hashCode() + pmf.hashCode();
    }

  /** 
  * Test for object equality, needed only for supporting Hashtables, use
  * equalTo for comparing Instants directly.  This method is used for
  * Instant/Instant comparisons, the default method for Objects is used
  * for Instant/Object comparisons.
  * @param other - Instant to compare
  * @return true or false 
  **/
  public boolean equals(IndeterminateInstant other) {
    return (this.hashCode() == other.hashCode());
    }

  /**
  * Returns the earliest possible TimeValue in the period of indeterminacy
  * @return TimeValue - the earliest possible TimeValue
  **/
  public TimeValue earliest() {
    return this;
    }

  /**
  * Returns the latest (only!) TimeValue in the Instant.
  * @return TimeValue - the latest TimevValue
  **/
  public TimeValue latest() {
    return this;
    }

// Removed Curt Jan-28-03
//  /**
//  * Create an Enumeration of Instants on the Anchorable.
//  **/
//  public Enumeration enumerate() {
//    return (Enumeration) new InstantEnumeration(this);
//    }

  /**
  * Cast this Instant to the indicated granularity.
  * Note that it is not a mutator, it will create a new Anchorable.
  * @param g - desired @see Granularity
  * @status NOT IMPLEMENTED
  **/
  public Instant cast(Granularity g) {
    return this;
    }

  /**
  * Scale this Instant to the indicated granularity.
  * Note that it is not a mutator, it will create a new Instant.
  * @param g - desired @see Granularity
  * @status NOT IMPLEMENTED
  **/
  public Instant scale(Granularity g) {
    return this;
    }

  /**
  * Retrieve the array of TimeVAlues (containing only one TimeValue) from
  * this Instant.
  * @return array of TimeValues with only one TimeValue in it.
  **/
  public TimeValue[] timeValueArray() {
    TimeValue[] temp = new TimeValue[1];
    temp[0] = this.tv;
    return temp;
    }

  /**
  * Retrieve the Granularity from this Instant
  * @return Granularity object
  **/
  public Granularity granularity() {
    return g;
    }

  /**
  * Retrieve the TimeValue from this Instant
  * @return TimeValue
  **/
  public TimeValue timeValue() {
    return tv;
    }

  /**
  * Construct a new instance of an Instant from an array of TimeValues.
  * This method may not seem useful but operations on AnchoredDataTypes
  * will produce arrays of TimeValues, from which the first TimeValue is 
  * selected to "create" a new Instant.
  * @param - i an array of TimeValues
  * @return - new Anchorable
  * @status NOT IMPLEMENTED
  **/
  public static Instant newInstance(TimeValue[] i) {
    // Sanity check
    if (i.length != 1) Internal.Error("More than one TimeValue in array");
    return new Instant(i[0].granularity(), i[0]);
    }
 
  /** 
  * Does this precede an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "precedes" beta
  **/
  public ExtendedBoolean precedes(Anchorable beta) {
    return TauZamanManager.Semantics.precedes(this, beta);
    }

  /** 
  * Does this precede an Instant?
  * @param beta - Instant to compare
  * @return this "precedes" beta
  **/
  public ExtendedBoolean precedes(Instant beta) {
    return TauZamanManager.Semantics.precedes(this, beta);
    }

  /** 
  * Does this overlap an Anchorable?  The overlap test evaluates whether
  * the instant touches any portion of the Anchorable.
  * @param beta - Anchorable to compare
  * @return this "overlaps" beta
  **/
  public ExtendedBoolean overlaps(Anchorable beta) {
    return TauZamanManager.Semantics.overlaps(this, beta);
    }

  /** 
  * Does this meet an Anchorable?  Meet is a predicate that evaluates 
  * whether the Instant 'touches' an Anchorable endpoint.
  * @param beta - Anchorable to compare
  * @return this "meets" beta
  **/
  public ExtendedBoolean meets(Anchorable beta) {
    return TauZamanManager.Semantics.meets(this, beta);
    }

  /** 
  * Does this contains an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "contains" beta
  **/
  public ExtendedBoolean contains(Anchorable beta) {
    return TauZamanManager.Semantics.contains(this, beta);
    }

  /** 
  * Does this equalTo an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "equalTo" beta
  **/
  public ExtendedBoolean equalTo(Anchorable beta) {
    return TauZamanManager.Semantics.equalTo(this, beta);
    }

  /** 
  * Displace (add) this instant with an Unanchorable
  * @param beta - displacor
  * @return Semantics(self + beta)
  **/
  public Instant add(Unanchorable beta) {
    return (Instant)TauZamanManager.Semantics.add(this, beta);
    }

  /**
  * Compute the distance between this Instant and another
  * @param beta - other Instant
  * @return Semantics(self - beta)
  **/
  public Interval subtract(Anchorable beta) {
    return (Interval)TauZamanManager.Semantics.subtract(this, beta);
    }

  /**
  * Displace (subtract) an Unanchroable from this instant
  * @param beta - displacor
  * @return Semantics(self - beta)
  **/
  public Instant subtract(Unanchorable beta) {
    return (Instant)TauZamanManager.Semantics.subtract(this, beta);
    }

}
