package tauadt.temporaldatatypes;

/**
* An <code>AnchoredTemporalDataType</code> provides a set of basic 
* operations available on Anchored temporal entities such as instants and periods.
* An Anchored entity is one that is anchored to specific points
* on the underlying time-line.  For example, an Instant (e.g., Dec. 23, 2002) 
* is a kind of Anchored entity since it is tethered to a particular point on 
* the time-line.  In contrast an Interval (e.g., 4 days)
* is an <em>unanchored</em> duration of time.  It would be implemented by the
* <code>UnanchoredTemporalDataType</code> class.
**/
public abstract class AnchoredTemporalDataType extends TemporalDataType {

  public AnchoredTemporalDatatype() {
    super(null);
    }

  public AnchoredTemporalDatatype(Granularity g) {
    super(g);
    }

  /**
  * Returns the earliest Instant in the Anchorable.
  * @return Instant - the earliest Instant
  **/
  public abstract Instant earliest();

  /**
  * Returns the latest Instant in the Anchorable.
  * @return Instant - the latest Instant
  **/
  public abstract Instant latest();

// Removed Curt Jan-03-28
//  /**
//  * Create an Enumeration on the Anchorable.
//  **/
//  public abstract Enumeration enumerate() {}
 
  /**
  * Retrieve the Instants from this Anchorable.
  * @return - array of all the Instants in the Anchorable
  **/
  public abstract Instant[] instantArray() {}

  /**
  * Returns the earliest (only!) Instant in the Anchored value.
  **/
  public abstract Instant earliest() {}

  /**
  * Returns the latest (only!) Instant in the Anchored value.
  * @return Instant 
  **/
  public abstract Instant latest() {}

  /**
  * Cast this Anchored to the indicated granularity.
  * Note that it is not a mutator, it will create a new Anchored.
  **/
  public abstract AnchoredDataType cast(Granularity g) { }

  /**
  * Cast this Anchored to the default granularity.
  * Note that it is not a mutator, it will create a new Anchored.
  **/
  public abstract AnchoredDataType cast(Granularity g) { }

  /**
  * Scale this Anchored to the indicated granularity.
  * Note that it is not a mutator, it will create a new Anchored.
  **/
  public abstract Anchored scale(Granularity g) {}

  /**
  * Scale this Anchored to the default granularity.
  * Note that it is not a mutator, it will create a new Anchored.
  **/
  public abstract Anchored scale() {}

  /** 
  * Is this Anchored EqualTo another?
  * @param beta - Anchored to compare
  * @return this == beta
  **/
  public ExtendedBoolean equalTo(Instant beta) {
    return new ExtendedBoolean(tv.equalTo(beta.tv));
    }

  /** 
  * Is this Instant lessThan another Instant?
  * @param beta - Instant to compare
  * @return this < beta
  **/
  public ExtendedBoolean lessThan(Instant beta) {
    return new ExtendedBoolean(tv.lessThan(beta.tv));
    }

  /** 
  * Is this Instant lessThanOrEqualTo another?
  * @param beta - Instant to compare
  * @return this <= beta
  **/
  public ExtendedBoolean lessThanOrEqualTo(Instant beta) {
    return new ExtendedBoolean(tv.lessThanOrEqualTo(beta.tv));
    }

  /** 
  * Is this Instant greaterThan another?
  * @param beta - Instant to compare
  * @return this > beta
  **/
  public ExtendedBoolean greaterThan(Instant beta) {
    return new ExtendedBoolean(tv.greaterThan(beta.tv));
    }

  /** 
  * Is this Instant greaterThanOrEqualTo another?
  * @param beta - Instant to compare
  * @return this >= beta
  **/
  public ExtendedBoolean greaterThanOrEqualTo(Instant beta) {
    return new ExtendedBoolean(tv.greaterThanOrEqualTo(beta.tv));
    }

  /**
  * Add an Interval to this Instant
  * @param beta - Interval to add
  * @return Instant - new Instant
  **/
  public Instant add(Interval beta) {
    return new Instant(g, tv.add(beta.timeValue()));
    }

  /**
  * Subtract an Interval from this Instant
  * @param beta - Interval to subtract
  * @return Instant - new Instant
  **/
  public Instant subtract(Interval beta) {
    return new Instant(g, tv.subtract(beta.timeValue()));
    }

  /**
  * Returns the first Instant out of two Instants
  * @param beta - Instant to compare with
  * @return the earliest Instant
  **/
  public Instant first(Instant beta) {
    if (tv.lessThanOrEqualTo(beta.tv)) return this;
    else return beta;
    }

  /**
  * Returns the last Instant out of two Instants
  * @param beta - Instant to compare with
  * @return the earliest Instant
  **/
  public Instant last(Instant beta) {
    if (tv.greaterThanOrEqualTo(beta.tv)) return this;
    else return beta;
    }

  /**
  * Subtract an Instant from this Instant
  * @param beta - Instant to subtract
  * @return resulting Interval
  **/
  public Interval subtract(Instant beta) {
    return new Interval(g, tv.subtract(beta.tv));
    }

  /**
  * Retrieve the array of Instants (containing only one Instant) from
  * this Instant.
  * @return array of Instants with one Instant in it.
  **/
  public Instant[] instantArray() {
    Instant[] temp = new Instant[1];
    temp[0] = this;
    return temp;
    }

  /**
  * Retrieve the Granularity from this Instant
  * @return Granularity object
  **/
  public Granularity granularity() {
    return g;
    }

  /**
  * Return the name of this class.
  * @return String
  **/
  public String kind() {
    return "Instant";
    }

  /**
  * Retrieve the TimeValue from this Instant
  * @return TimeValue
  **/
  public TimeValue timeValue() {
    return tv;
    }

  /**
  * Construct a new instance of an Instant, Interval combination
  * @return - new Anchorable
  **/
  public Anchorable newInstance(String beta, Instant[] i) {
    if (i.length > 1) Internal.Error("More than one Instant in array");
    if (beta.compareTo("Interval") != 0) Internal.Error("Unknown kind of Unanchorable");
    return i[0];
    }
 
  /** 
  * Does this precede an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "precedes" beta
  **/
  public ExtendedBoolean precedes(Anchorable beta) {
    return Instant.Semantics.precedes(this, beta);
    }

  /** 
  * Does this overlap an Anchorable?  The overlap test evaluates whether
  * the instant touches any portion of the Anchorable.
  * @param beta - Anchorable to compare
  * @return this "overlaps" beta
  **/
  public ExtendedBoolean overlaps(Anchorable beta) {
    return Instant.Semantics.overlaps(this, beta);
    }

  /** 
  * Does this meet an Anchorable?  Meet is a predicate that evaluates 
  * whether the Instant 'touches' an Anchorable endpoint.
  * @param beta - Anchorable to compare
  * @return this "meets" beta
  **/
  public ExtendedBoolean meets(Anchorable beta) {
    return Instant.Semantics.meets(this, beta);
    }

  /** 
  * Does this contains an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "contains" beta
  **/
  public ExtendedBoolean contains(Anchorable beta) {
    return Instant.Semantics.contains(this, beta);
    }

  /** 
  * Does this equalTo an Anchorable?
  * @param beta - Anchorable to compare
  * @return this "equalTo" beta
  **/
  public ExtendedBoolean equalTo(Anchorable beta) {
    return Instant.Semantics.equalTo(this, beta);
    }

  /** 
  * Displace (add) this instant with an Unanchorable
  * @param beta - displacor
  * @return Semantics(self + beta)
  **/
  public Instant add(Unanchorable beta) {
    return (Instant)Instant.Semantics.add(this, beta);
    }

  /**
  * Displace (subtract) an Unanchroable from this instant
  * @param beta - displacor
  * @return Semantics(self - beta)
  **/
  public Instant subtract(Unanchorable beta) {
    return (Instant)Instant.Semantics.subtract(this, beta);
    }

}

}
