package tauzaman.temporaldatatypes;

import java.util.*;
import java.rmi.*;

import tauzaman.timestamp.*;
import tauzaman.calendricsystem.*;
import tauzaman.*;
import tauzaman.io.*;

/**
* The <code>Period</code> class provides a set of constructor and accessor
* operations on periods.
* (Arithmetic and comparison operations are implemented in a Semantics.)
* A period is the time between two Instants anchored to the time-line,
* for instance, "[July 4, 2002 - July 12, 2002]".  All periods, when
* represented in a Period object, are 
* assumed to be closed, that is the endpoints are included in the
* time of the period (note however, that periods may be constructed
* from strings that represent either open or closed periods, it is just
* that the constructed period will be assumed to be closed).
* Currently there is a constraint that the Instants in the period
* must have either be granularity-less values (e.g., MIN) or have
* the same granularity.  Futhermore the starting Instant must come 
* before the ending Instant.
* This is the standard package that supports operations on periods that
* have a known, fixed position as well as those that 
* indeterminately known.
*
* To perform comparisons and arithmetic operations on Periods, 
* it is first necessary to create a Semantics interface.
* A Semantics implements common comparison and arithmetic operations on 
* temporal entities.  For example a user would write code like the following.
*  <pre>
*     // Left operand semantics casts operands in binary operations 
*     // to the granularity of the left operand, and then performs
*     // the desired operation. 
*     Ops = new LeftOperandSemantics();
*     // Create two Periods, in this case, both are at the granularity of days
*     Period i = new Period("[July 3, 2002 - July 10, 2002]");
*     Period j = new Period("[July 4, 2002 - July 5, 2002]");
*     // Is i earlier on the time-line than j?
*     if (Ops.precedes(i,j)) ...
*  </pre>
* We employ this strategy so that all the period operations can be
* performed under varying Semantics.
*
* @author  Curtis Dyreson 
* @version 0.1, 10/10/02
* @see     tauzaman.temporaldatatypes.DeterminateSemantics
* @see     tauzaman.temporaldatatypes.IndeterminateSemantics
**/
public class Period implements Cloneable {

  private Instant startingInstant;
  private Instant endingInstant;

  /** <code>TauZamanService</code> that this <code>Instant</code> is created under **/
  TauZamanService cachedTauZamanService = null;


  /**
  * Construct a Period from a pair of Instants.  It will first convert
  * each Instant to the desired granularity using the default conversion
  * semantics. It will also ensure 
  * that the starting Instant is before the ending Instant.  
  * @param g - Granularity of desired period
  * @param startingInstant - first Instant
  * @param endingInstant - second Instant
  * @exception badPeriod - possibly terminates before it starts, currently
  *  it will set the starting and terminating Instants
  *  as the same Instant.
  * @status NOT IMPLEMENTED
  **/
  public Period(Granularity g, Instant startingInstant, Instant endingInstant) {
    // Need to scale!
    this.startingInstant = startingInstant;
    this.endingInstant = endingInstant;
    ExtendedBoolean lt = this.startingInstant.getGranule().lessThanOrEqualTo(this.endingInstant.getGranule());
    if (lt.unsatisfied()) {
      this.endingInstant = this.startingInstant;
    //  throw exception "Trying to create a period that ends before it starts"
      }
    }

  /**
  * Construct a Period from a pair of Instants.  It will check to ensure
  * that both Instants are in the same granularity and that the starting
  * Instant is before the ending Instant
  * @param startingInstant - First instant
  * @param endingInstant - Second instant
  * @exception badPeriod - possibly terminates before it starts, if so 
  *  it sets the terminatingInstant to be the same as the startingInstant
  * @status NOT IMPLEMENTED
  **/
  public Period(Instant startingInstant, Instant endingInstant) {
    // Check to see if same granularity
    this.startingInstant = startingInstant;
    this.endingInstant = endingInstant;
    ExtendedBoolean lt = this.startingInstant.getGranule().lessThanOrEqualTo(this.endingInstant.getGranule());
    if (lt.unsatisfied()) {
      this.endingInstant = this.startingInstant;
    //  throw exception "Trying to create a period that ends before it starts"
      }
    }

  /**
  * Construct a <code>Period</code> object from a given String temporal constant.
  *
  * @param literal String represantion of <code>Period</code>
  *
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs during parsing of
  * string temporal constant to <code>Instant</code> timestamp
  */
  public Period(String literal) throws TemporalDataTypeFormationException{

      Granule [] granules = null;
     
      // get active TauZamanService
      TauZamanService activeTauZamanService = TauZamanSystem.getActiveService();
  
      // throws TauZamanException
      try{
          granules = activeTauZamanService.parseInput(literal, "PeriodInputFormat");
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Period data type from: " + literal, re);
      }
      catch(IOException ioe){
          throw new TemporalDataTypeFormationException("Exception when forming Period data type from: " + literal, ioe);
      }
  
      // form two Instants, we may also do some checks here...
      startingInstant = new Instant(granules[0]);
      endingInstant = new Instant(granules[1]);

      // cache the service
      cachedTauZamanService = activeTauZamanService;
  
      // cache the active CalendricSystem name  
      try{
          cachedTauZamanService.cacheCalendricSystemName();
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Period data type from: " + literal, re);
      }
      
  }
  
  /**
  * Returns string representation of this <code>Instant</code>.
  *  
  * @return String temporal constant that corresponds to this <code>Instant</code>
  *  
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs when forming    
  * string temporal constant from timestamp    
  */
  public String output() throws TemporalDataTypeFormationException{
  
       Granule granules [] = new Granule[2];
       granules[0] = startingInstant.getGranule();
       granules[1] = endingInstant.getGranule();
  
      String output = null;
       
      try{
          output = cachedTauZamanService.parseOutput(granules, "PeriodOutputFormat");
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when outputting Period data type", re);
      }
      catch(IOException ioe){
          throw new TemporalDataTypeFormationException("Exception when outputting Period data type", ioe);
      }
      
      return output;
  }  


  /** 
  * Build a nice string image of an Period, for debugging mostly.
  * Provides an alternative to toString().
  * @return String image of Period
  **/
  public String image() {
    return "[Period" + 
           " startingInstant " + startingInstant.image() + 
           " endingInstant " + endingInstant.image() + "]";
    }

  /**
  * Selector - Returns the starting Instant in the Period.
  * @return Instant - the Instant
  **/
  public Instant getStarting() {
    return startingInstant;
    }

  /**
  * Selector - Returns the ending Instant in the Period.
  * @return Instant - the Instant
  **/
  public Instant getEnding() {
    return endingInstant;
    }

  /**
  * Selector - Returns the earliest Granule in the Period.
  * @return Granule - the Granule of the Starting Instant
  **/
  public Granule getEarliest() {
    return startingInstant.getGranule();
    }

  /**
  * Selector - Returns the latest Granule in the Period.
  * @return the ending instant's granule
  **/
  public Granule getLatest() {
    return endingInstant.getGranule();
    }

  /**
  * Retrieve the granularity of the Period.
  * @return Granularity
  **/
  public Granularity getGranularity() {
    // Granularity is that of whichever Instant has a granularity
    if (startingInstant.getGranularity() != null) return startingInstant.getGranularity();
    return endingInstant.getGranularity();
    }

  /**
  * Scale the bounding Instants to the specified granularity.  Constructs
  * a new Period.
  * @param g - target Granularity 
  **/
  public Period scale(Granularity g) {
    return new Period(startingInstant.scale(g), endingInstant.scale(g));
    }

  /**
  * Cast the bounding Instants to the specified granularity.  Constructs
  * a new Period.
  * @param g - target Granularity 
  **/
  public Period cast(Granularity g) {
    return new Period(startingInstant.cast(g), endingInstant.cast(g));
    }
}
