package tauzaman.property;

import java.util.*;

import tauzaman.property.Property;



/* Structure: Each element in propertyStackList contain one stack
   corresponding to each different property. Hashtable maps a given
   property name to its stack in this list. Hence having this hashtable
   and different stacks for different properties not only increase 
   efficiency but also makes the code easier.
*/

/**
* <code>PropertyStackService</code> class implements data structures to store properties
* and provides service to <code>PropertyManager</code>.
*
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 11/15/02
* @see     tauzaman.property.PropertyManager
* @status design complete, implementation complete
*/

public class PropertyStackService{

  /** Property Stack List **/
  private Stack [] propertyStackList = null;

  /** Property Stack Hash table, provides efficient indexing of stacks **/
  private Hashtable propertyStackMap = null;
    
  /** 
  * Constructs a <code>PropertyStackService</code> object, which provides 
  * stack operations for <code>Property</code>.
  */
  public PropertyStackService(){
      
      /* currently we have 12 properties defined  */
      propertyStackList = new Stack[12];

      /* form all stacks for properties */
      for(int i = 0; i < 12; i++)
          propertyStackList[i] = new Stack();

      /* form hashtable */
      propertyStackMap = new Hashtable();
      initStackMap();
  }

  /** 
  * Initialize the hash table. This might also be done by PropertyManager, 
  * by providing a string[] of property names, however, there are this many
  * properties after all.
  * Each key is the name of the property and correspinding value 
  * is the corresponding index in the property StackList
  */
  public void initStackMap(){
      propertyStackMap.put("Locale", 				new Integer(0));
      propertyStackMap.put("InstantInputFormat", 		new Integer(1));
      propertyStackMap.put("InstantOutputFormat", 		new Integer(2));
      propertyStackMap.put("IntervalInputFormat", 		new Integer(3));
      propertyStackMap.put("IntervalOutputFormat", 		new Integer(4));
      propertyStackMap.put("NowRelativeInstantInputFormat", 	new Integer(5));
      propertyStackMap.put("NowRelativeInstantOutputFormat", 	new Integer(6));
      propertyStackMap.put("PeriodInputFormat", 		new Integer(7));
      propertyStackMap.put("PeriodOutputFormat", 		new Integer(8));
      propertyStackMap.put("IndeterminateInstantInputFormat", 	new Integer(9));
      propertyStackMap.put("IndeterminateInstantOutputFormat", 	new Integer(10));
      propertyStackMap.put("OverrideInputOrder", 		new Integer(11));
  }
 
  /** 
  * Returns the index of <code>Property</code> given its key, that is to say its name. 
  *
  * @param propertyName name of a <code>Property</code>
  *
  * @return stack, which corresponds to first parameter, propertyName,
  * in stack list
  */
  private Stack getStack(String propertyName){

      /* validness of propertyName should have already been performed */
      int index = ((Integer)propertyStackMap.get(propertyName)).intValue();

      return propertyStackList[index];
  }

  /** 
  * Pushes a <code>Property</code> to its stack.
  * @param property a <code>Property</code>
  */
  public void push(Property property){

      Stack stack = getStack(property.getName());
      /* push this Property into the stack */
      stack.push(property);
  }

  /** 
  * Pops the corresponding active <code>Property</code> from its stack. Does not
  * deactivates default <code>Property</code>.
  *
  * @param propertyName name of a <code>Property</code>
  */ 
  public void pop(String propertyName){

      Stack stack = getStack(propertyName);
      /* pop the Property, be careful of default Property */
      if(stack.size() > 1)
          stack.pop();
  }

  /** 
  * Pops all active <code>Property</code>ies from their stacks, except 
  * default ones, which are the last elements in stacks.Does not
  * deactivates default <code>Property</code>.
  */
  public void pop(){
      Stack stack;
      for(int i = 0; i < propertyStackList.length; i++){
          stack = propertyStackList[i];
          if(stack.size() > 1)
              stack.pop();
      }
  }

  /** 
  * Pops all elements from given property stack until the last elements, which
  * are the default properties.
  *
  * @param propertyName name of a <code>Property</code>
  */
  public void popToLast(String propertyName){
      Stack stack = getStack(propertyName);
      while(stack.size() > 1)
          stack.pop();
  }

  /** 
  * Pop all elements from their stack until the last elements, which
  * are the default properties. 
  */
  public void popToLast(){
      Stack stack;
      for(int i = 0; i < propertyStackList.length; i++){
          stack = propertyStackList[i];
          while(stack.size() > 1)
              stack.pop();
      }
  }

  /** 
  * Gets and returns the top element (<code>Property</code>) of the stack (without removing it).
  * If there is no <code>Property</code>, it returns null.
  *
  * @param propertyName name of a <code>Property</code>
  *
  * @return a <code>Property</code>
  */
  public Property peek(String propertyName){
      Stack stack = getStack(propertyName);
      return (Property)stack.peek();
  }

  /**
  * Returns all elements of a given stack
  * 
  * @param propertyName name of a <code>Property</code>
  * @return an Enumeration containing all Elements of a stack
  */
  public Enumeration peekAll(String propertyName){
      Stack stack = getStack(propertyName);
      return stack.elements();
  }
}
