package tauzaman.property;


import java.util.*;
import java.net.*;
import org.w3c.dom.*;

import tauzaman.property.PropertyStackService;

/**
* <code>PropertyManager</code> class manages user specific set of <code>Properties</code>.
* Each <code>TauZaman</code> user (remote or local) will have his/her own <code>PropertyManager</code>.
*
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 11/15/02
* @see     tauzaman.property.Property
* @see     tauzaman.property.PropertyStackService
* @status design complete, implementation complete
*/


public class PropertyManager{

  /** 
  * Service that provides operations on active properties
  */
  private PropertyStackService propertyStackService = null; 

  /**
  * Url of default property list.
  */
  private URL defaultPropertyListUrl = null;

  /**
  * Unique <code>PropertyRepository</code> of this TauZaman system
  */
  private PropertyRepository pr = null;
 
  /** 
  * Constructs a <code>PropertyManager</code> object and 
  * initializes the default property list.
  *
  * @param url url of a default <code>Property</code> table, which this manager
  * starts with
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */ 
  public PropertyManager(PropertyRepository pr, URL url) throws PropertyFormationException {

      /* a service for managing property structures */
      propertyStackService = new PropertyStackService();

      /* get default property table url */
      defaultPropertyListUrl = url;

      /* get the Property Repository */
      this.pr = pr;

      /* initialize the default property table */
      propertyDefaultListInit();
  }

  /** 
  * Initializes this <code>PropertyManager</code> with a default property list. 
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */
  private void propertyDefaultListInit() throws PropertyFormationException{
        /* declare a Property list */
        Property propertyList[] = null;
        /* get the Properties by using Property Repository */ 
        propertyList = pr.loadProperty(defaultPropertyListUrl, null);
        /* size of plist should be 12, otherwise defaultPropertyList is not 
           pointing to a default Property Specification file */
        if(propertyList.length != 12){
            throw new PropertyFormationException("Exception when forming default properties from: " + 
                   defaultPropertyListUrl.toExternalForm() + " not enough Properties: " + propertyList.length);
        }
        /* if everything is good, then initialize Property Stack Service */
        for(int i = 0; i < propertyList.length; i++)
            propertyStackService.push(propertyList[i]);
  }

  /**
  * Loads <code>Properties</code> without activating it. This method is
  * unique to <code>Input</code> and <code>Output</code>.
  *
  * @param url URL of Property Specification file
  * @param name string name of <code>Property</code> to be loaded.
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */
  public Property getProperty(URL url, String name) throws PropertyFormationException, PropertyServiceException{

      /* declare a Property list */
      Property propertyList[] = null;

      if(pr.isValidPropertyName(name)){
          /* use loadProperty of PropertyRepository */
          String names [] = {name};
          /* get the Properties by using Property Repository */
          propertyList = pr.loadProperty(url, names);
      }
      else{
          throw new PropertyServiceException("Exception when servicing Properties, no such Property name: " + name);
      }
  
      return propertyList[0];
  }


  /** 
  * Inserts a new <code>Property</code>(possibly Properties, given by url) to property 
  * structure, which is handled by <code>PropertyStackService</code> and activates it. 
  * First property with the given names in the file pointed by given url will be activated.
  * If all Properties in the file pointed by URL will be activated, then pass null as names.
  * 
  * @param url URL of Property Specification file
  * @param names array of <code>Property</code> names to be activated
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */
  public void propertyActivate (URL url, String names []) throws PropertyFormationException{
        /* declare a Property list */
        Property propertyList[] = null;
        /* get the Properties by using Property Repository */ 
        propertyList = pr.loadProperty(url, names);
        /* if everything is good, then update Property Stack Service */
        for(int i = 0; i < propertyList.length; i++)
            propertyStackService.push(propertyList[i]);
  }

  /** 
  * Deactivates any active properties and removes them from <code>PropertyStackService</code>.
  * With an exception: It will not deactivate or remove any default property,
  * so any attempt will only succeed to deactivate and remove only non-default
  * active properties.
  */
  public void propertyDeactivateAll(){
      /* deactivates all active properties, but not defaults */
      propertyStackService.pop();
  }

  /** 
  * Deactivates a <code>Property</code> (given by its name) and removes it from property list,
  * which is managed by <code>PropertyStackService</code>. If the active <code>Property</code> 
  * is the default <code>Property</code>, then keep silent.
  *
  * @param name name of a <code>Property</code>
  * @throws PropertyServiceException if any abnormal condition occurs when deactivating
  * active <code>Property</code> with given name
  */
  public void propertyDeactivate(String name) throws PropertyServiceException{
      /* deactivates a single active property (from the stack, which corresponds 
         to name), but not default */
      if(pr.isValidPropertyName(name)){
          propertyStackService.pop(name);
      }
      else{
          throw new PropertyServiceException("Exception when servicing Properties, no such Property name: " + name);
      }
  }
  
  /** 
  * Sets all active <code>Properties</code> to their default values. 
  */
  public void propertySetDefaultAll(){
      /* sets all Property stacks to their default values */
      propertyStackService.popToLast();

  }

  /** 
  * Sets <code>Property</code> (given by its name) to its default value. If the
  * <code>Property</code> already has its default value active, then keep silent.
  *
  * @param name name of the <code>Property</code>  
  *
  * @throws PropertyServiceException if any abnormal condition occurs when setting
  * active <code>Property</code>, with given name, to its default value
  */
  public void propertySetDefault(String name) throws PropertyServiceException{
      /* sets a Property stack (which corresponds to given name) to its default values */
      if(pr.isValidPropertyName(name)){
          propertyStackService.popToLast(name);
      }
      else{
          throw new PropertyServiceException("Exception when servicing Properties, no such Property name: " + name);
      }
  }  
  
  /** 
  * Returns the active <code>Property</code> given its name. 
  *
  * @param name String name of a <code>Property</code>
  *
  * @return <code>Property</code> with the given name. 
  *
  * @throws PropertyServiceException if any abnormal condition occurs when setting
  * active <code>Property</code>, with given name, to its default value
  */
  public Property getProperty(String name) throws PropertyServiceException{
      Property p = null;
      /* peeks a Property from a stack, which corresponds to given name */
      if(pr.isValidPropertyName(name)){
          p = propertyStackService.peek(name);
      }
      else{
          throw new PropertyServiceException("Exception when servicing Properties, no such Property name: " + name);
      }
      return p;
  }

  /**
  * Gets all the current <code>Property</code> information of a specific
  * Property name in a String. Format of the return string is; 
  *
  * {urlOfProperty}%{urOfProperty}%....%{urlOfProperty}
  *
  * order is same as of Property stack from bottom to top.
  * 
  * @param name String name of a <code>Property</code>
  * @return a String containing current Properties activated for a 
  * specific Property name 
  * @throws PropertyServiceException if any abnormal condition occurs when
  * getting the image of <code>Property</code> stack given its name
  */
  public String getPropertyImage(String name) throws PropertyServiceException{
      String image = new String();
      if(pr.isValidPropertyName(name)){
          for(Enumeration e = propertyStackService.peekAll(name); e.hasMoreElements(); ){
              Property p = (Property)e.nextElement();
              image = image.concat((p.getUrl()).toExternalForm() + "%");
          }
      }
      else{
          throw new PropertyServiceException("Exception when servicing Properties, no such Property name: " + name);
      }
      return image;
  }


  /**
  * Returns this <code>PropertyManager</code>'s default property list URL.
  * 
  * @return String representations of default URL
  */
  public String toString(){
      String rep = "PropertyManager: default Property is: " + defaultPropertyListUrl.toExternalForm();
      return rep;
  }
}
