package tauzaman.property;

import java.util.*;
import java.net.*;
import org.w3c.dom.*;

import tauzaman.XMLParser;



/**
* <code>Property</code> class is the representation of a single property
*   Properties are two types: 
* <ul>
*   <li> Simple Type, which includes just a string, the value of Property.</li>
* </ul>
* For example; 
* <pre> 
* &lt;property name = "..." value = "..." /&gt;
* </pre> 
* <ul>
*   <li> Complex Type, which includes more detail information. (such as FV support, field names, format, etc.) </li>
* </ul>
* For example; 
* <pre> 
* &lt;property name = "..." &gt; 
* &lt;value&gt; ... &lt/value&gt;
* &lt;/property&gt;
* </pre> 
*
* 
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 11/15/02
* @see     tauzaman.property.Property
* @see     tauzaman.property.FieldInfo
* @see     tauzaman.property.ImportFormat
* @see     tauzaman.property.Format
* @status design complete, implementation complete
*/

public class Property{
  
  /** name of this <code>Property</code> **/
  private String name = null;  

  /** url of this <code>Property</code> */
  private URL url = null;

  /** Whether type of this <code>Property</code> is a simple one or not **/
  private boolean isSimple;
  
  /** String content of this <code>Property</code> in case its type is simple **/
  private String content = null;

  /** List of formats in case this <code>Property</code> has a complex type **/
  private Format [] formats = null;

  /** List of field infos in case this <code>Property</code> has a complex type **/
  private FieldInfo [] fieldInfos = null;

  /** List of imported formats in case this <code>Property</code> has a complex type **/
  private ImportFormat [] importFormats = null;

  /** 
  * Cached version of this <code>Property</code>'s format which includes
  * its parsed Fields, newFormat and whitespace attribute. But of course,
  * caching has also its side effects as every caching behavior. For example,
  * if a recursive <code>Format</code> is updated this previous version of it
  * will still be used by this cache. 
  */
  private PropertyCache cache = null;

  /**
  * Constructs a <code>Property</code>, from a given DOM Node, 
  * parsed from Property Specification file.
  *
  * @param root a DOM node of parsed Property Specification file
  *
  * @param fvMapper A Hashtable, which maps field value support labels to their urls.
  * This information is passed by <code>PropertyRepository</code> since this information
  * resides in Property specification file (as helper) rather than in individual Properties.
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  *
  */
  public Property(String name, URL url, Element root, Hashtable fvMapper) throws PropertyFormationException{
     this.name = name;
     this.url = url;
     /* initializes the state of this Property from given DOM Node */
     formProperty(root, fvMapper);
  }

  /**
  * Initializes this <code>Property</code> from DOM Node, which points to
  * a property element in the Property Specification file.
  *
  * @param root root Element of a single parsed <code>Property</code> xml 
  * specification
  * 
  * @param fvMapper A Hashtable, which maps field value support labels to their urls.
  * This information is passed by <code>PropertyRepository</code> since this information
  * resides in Property specification file (as helper) rather than in individual Properties.
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  *
  */
  private void formProperty(Element root, Hashtable fvMapper) throws PropertyFormationException{
      
      /* whether this Property is simple or complex type */
      if(root.hasAttribute("value")){
          isSimple = true;
          content = root.getAttribute("value");
      }
      else{

          XMLParser parser = new XMLParser();
          Vector valueNodes = parser.locator(root, "value");
          if(valueNodes.size() != 1)
              throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                 url.toExternalForm() + " there should be only one values!");

          Element valueNode = (Element)valueNodes.elementAt(0);
          
          Vector formatNodes = parser.locator(valueNode, "format");
          if(formatNodes.size() == 0)
              throw new PropertyFormationException("Exception when parsing xml file in: " + 
                             url.toExternalForm() + " there should be more than 0 formats!");

          formats = new Format [formatNodes.size()];
          for(int i = 0; i < formatNodes.size(); i++){
              Element formatNode = (Element)formatNodes.elementAt(i);
              formats[i] = new Format(formatNode);
          }

          Vector importFormatNodes = parser.locator(valueNode, "importFormat");
          /* no need to check the size for importFormats */
          importFormats = new ImportFormat [importFormatNodes.size()];
          for(int i = 0; i < importFormatNodes.size(); i++){
              Element importFormatNode = (Element)importFormatNodes.elementAt(i);
              importFormats[i] = new ImportFormat(importFormatNode, url);
          }

          Vector fieldInfoNodes = parser.locator(valueNode, "fieldInfo");

          fieldInfos = new FieldInfo [fieldInfoNodes.size()];
          for(int i = 0; i < fieldInfoNodes.size(); i++){
              Element fieldInfoNode = (Element)fieldInfoNodes.elementAt(i);
              fieldInfos[i] = new FieldInfo(fieldInfoNode, fvMapper, url);
          }
      }
  }

  /* Accessor methods */

  /**
  * Returns true if this <code>Property</code> has already been loaded for
  * Input/Output reasons. 
  *
  * @return true if there is a <code>PropertyCache</code> object of this
  * <code>Property</code>. False otherwise.
  */
  public boolean hasCache(){
      if(cache == null)
          return false;
      return true;
  }

  /**
  * Returns <code>PropertyCache</code> object of this
  * <code>Property</code>.
  *
  * @return <code>PropertyCache</code> object of this
  * <code>Property</code>
  */
  public PropertyCache getCache(){
     return cache;
  }  

  /**
  * Sets <code>PropertyCache</code> object of this
  * <code>Property</code>.
  *
  * @param  <code>PropertyCache</code> object of this
  * <code>Property</code>
  */
  public void setCache(PropertyCache cache){
      this.cache = cache;
  }  

  /** 
  * Returns the type of this Value 
  * @return kind of this Value
  */
  public boolean isSimpleType(){
      return isSimple; 
  }

  /**
  * Returns the string name of this <code>Property</code>.
  *
  * @return String name of this <code>Property</code>
  */
  public String getName(){
      return name;
  }

  /**
  * Returns URL of this <code>Property</code>.
  *
  * @return URL url of this <code>Property</code>
  */
  public URL getUrl(){
      return url;
  }

  /** 
  *   If simple type then return content,
  *   o.w. return null. We may also handle an error. 
  * @return null or content of this Value
  */
  public String getContent(){
      if(isSimple) return content;
      return null;
  }

  /** 
  * Return the field value List 
  * @return list of Field Infos
  * @see tauzaman.property.FieldInfo
  */
  public FieldInfo [] getFieldInfos(){
      return fieldInfos;
  }

  /** 
  * Return the format list 
  * @return list of Formats
  * @see tauzaman.property.Format
  */
  public Format [] getFormats(){
      return formats;
  }

  /** 
  * Return the import format list 
  * @return list of ImportFormats
  * @see tauzaman.property.ImportFormat
  */
  public ImportFormat [] getImportFormats(){
      return importFormats;
  }

  /** 
  * toString method
  *
  * @return string representation of this object
  */
  public String toString(){
      String result = "  Property: " + name + " with url " + url.toExternalForm();       
      result = result.concat("    Formats: ");
      if(isSimple)
          result = result.concat("    " + content);
      else{
          for(int i = 0; i < formats.length; i++)
              result = result.concat("    " + formats[i].toString());

          result = result.concat("    ImportFormats: ");
          for(int i = 0; i < importFormats.length; i++)
              result = result.concat("    " + importFormats[i].toString());

          result = result.concat("    FieldInfos: ");
          for(int i = 0; i < fieldInfos.length; i++)
              result = result.concat("    " + fieldInfos[i].toString());
      }
      return result;

  }  
}
