package tauzaman.io;

import java.net.*;
import tauzaman.property.*;
import tauzaman.field.*;



/**
* <code>FieldsBuilder</code> class builds <code>Fields</code> object
* of a <code>Property</code>. Given a <code>Property</code>, it parses
* its <code>Format</code>, and creates <code>Field</code> for each 
* variable in <code>Format</code>. If there are <code>ImportFormat</code>s
* it recursively request this class's service.
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/
class FieldsBuilder{

  /*
  * <code>PropertyManager</code> that this <code>FieldsBuilder</code>
  * uses to load <code>Properties</code> 
  */
  private PropertyManager pm = null;

  /*
  * <code>Property</code> that <code>Fields</code> will be built from
  */
  private Property property = null;

  /*
  * <code>Fields</code> that will be derived from a <code>Property</code>
  */
  private Fields fields = null;

  /*
  * new format string (might be formed recursively) that represents a 
  * <code>TemporalDataType</code>'s corresponding temporal constant
  */
  private StringBuffer newFormat = null;

  /**
  * Constructs a <code>FieldsBuilder</code> object ready to form <code>Fields</code>.
  * 
  * @param propertyName name of the <code>Property</code> whose <code>Fields</code> will be formed
  * @param url URL of the <code>Property</code> whose <code>Fields</code> will be formed
  * @param pm <code>PropertyManager</code> that will be used for <code>Property</code> services
  *
  * @throws IOException if any abnormal condition occurs when building <code>Fields</code>. These
  * are related to <code>Property</code> exceptions
  */
  FieldsBuilder(String propertyName, URL url, PropertyManager pm) throws IOException{

      /* Load the property without activating it */
      if(url == null){
          try{
              property = pm.getProperty(propertyName);
          }
          catch(PropertyServiceException pse){
              throw new IOException("Exception occurred when building fields", pse);
          }
      }
      else{
          try{
              property = pm.getProperty(url, propertyName);
          }
          catch(PropertyFormationException pfe){
              throw new IOException("Exception occurred when building fields", pfe);
          }
          catch(PropertyServiceException pse){
              throw new IOException("Exception occurred when building fields", pse);
          }
      }

      /* allocate memory for the Fields */
      fields = new Fields(propertyName);
      this.pm = pm;
  }

  /**
  * Returns built <code>Fields</code>.
  *
  *
  * @return <code>Fields</code> 
  *
  * @throws IOException if any abnormal condition occurs when building <code>Fields</code>. These
  * are related to parsing and building problems of <code>Fields</code>
  */ 
  Fields formFields() throws IOException{

      newFormat = new StringBuffer();

      /* get Property's Format information */
      Format formats[] = property.getFormats();

      String format = formats[0].getFormatString();

      /* form a FormatParser to parse format and fetch variable names(if any) */
      FormatParser fp = new FormatParser(format);

      /* we need all tokens to form a new format */
      String tokens [] = fp.parseFormat(false);

      /* gets Property's ImportFormats and FieldInfos */
      FieldInfo fieldInfos [] = property.getFieldInfos();
      ImportFormat importFormats [] = property.getImportFormats();      

      for(int i = 0; i < tokens.length; i++){
 
          /* if the token is variable*/
          if(tokens[i].matches("\\$[a-zA-Z]+")){
          
              boolean found = false;
              for(int j = 0; j < fieldInfos.length; j++){
                  if(tokens[i].equals("$" + fieldInfos[j].getVariableName())){
                      found = true;
                      fields.addElement(new Field(fieldInfos[j]));
                      newFormat.append(tokens[i]);
                      break;
                  }
              }
          
              if(!found){
                  for(int j = 0; j < importFormats.length; j++){
                      if(tokens[i].equals("$" + importFormats[j].getVariableName())){
                          found = true;
                          FieldsBuilder fb = new FieldsBuilder(importFormats[j].getFormatName(),
                                                       importFormats[j].getFormatPropertyUrl(), pm);
                          fields.addElement(fb.formFields());
                          newFormat.append(fb.getFormat());
                          break;
                      }
                  }
              }

              if(!found)
                  throw new IOException("Exception when building fields: can not find " + tokens[i] + " in Property definition");

          }
          /* token is not a variable */
          else{
              newFormat.append(tokens[i]);
          }
      }

      return fields;
  }

  /**
  * Returns <code>Property</code>'s newly built <code>Format</code>'s value.
  *
  * @return String <code>Format</code>'s value
  */
  String getFormat(){
      return newFormat.toString();
  }
  
  /**
  * Returns <code>Property</code>'s newly built <code>Fields</code>.
  *
  * @return <code>Fields</code>
  */
  Fields getFields(){
      return fields;
  }  

  /**
  * Returns whitespace characteristic of <code>Property</code>'s <code>Format</code>.
  *
  * @return boolean whitespace characteristic of <code>Format</code>
  */
  boolean getWhitespace(){
      Format formats[] = property.getFormats();
      return formats[0].isFormatWhitespaceFriendly();
  }    
  
}
