package tauzaman.field.fvsupport;

import java.net.URL;

import tauzaman.field.FVServiceException;

/** 
* <p>
* <code>FVSupport</code> abstrat class represents a generic field value 
* support information, that provides two basic functionality when converting 
* temporal constant strings to timestamps or vice versa. </p><p>
* One of them is called indexToString, which converts indices, such as 1, to
* corresponding string, such as January in english month names.
* The other functionality is called stringToIndex, and it does opposite
* of what indexToString does. Given January, for example, it converts it to
* corresponding index, which is 1 in again english month names.
* </p><p>
* This class is used as a base class for <code>FVTable</code> and 
* <code>FVFunction</code> classes. </p>
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1. 01/06/02
* @status design complete, implementation complete
*/

public abstract class FVSupport{

  /** url of this structure, which contains actual conversion informatio **/
  protected URL url = null;
 
  /** regex for this structure, that will be used to parse input string **/
  protected String regex = null;

  /** 
  * Converts a given string to corresponding index.
  * <pre>
  * <i>Given</i> <b>January</b> for Gregorian calendar and using 
  * a table called "EnglishMonthNames", this method returns
  * <b>1</b> as the index of <b>January</b>.
  * </pre>
  *
  * @param string string representation of input 
  *
  * @return long index corresponding of string  
  *
  * @throws FVServiceException if any problem occurs during process
  * of this field value support method
  *
  * @see tauzaman.field.FVTable#indexToString
  * @see tauzaman.field.FVFunction#indexToString
  */  
  public abstract long stringToIndex(String string) throws FVServiceException;

  /** 
  * Converts a given long to corresponding string.
  * <pre>
  * <i>Given</i> <b>1</b> for Gregorian calendar and using 
  * a table called "EnglishMonthNames", this method returns
  * <b>January</b> as the string corresponding of <b>1</b>.
  * </pre>
  *
  * @param index long representation of input 
  *
  * @return String string corresponding of long index
  * 
  * @throws FVServiceException if any problem occurs during process
  * of this field value support method
  *
  * @see tauzaman.field.FVTable#stringToIndex
  * @see tauzaman.field.FVFunction#stringToIndex
  */  
  public abstract String indexToString(long index) throws FVServiceException;

  /** 
  * Returns regex for this field value if it exists.
  * If it does not exists default regex, which is defined 
  * in <a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepcalendricspec.htm">
  * Calendric System Specification</a> file, will be used. 
  * 
  * @return String regular expression defined for this FVsupport
  */
  public String getRegex(){
      return regex;
  }

  /** 
  * Returns url for this field value if it exists.
  * 
  * @return URL url of this FVsupport
  */
  public URL getUrl(){
      return url;
  }
 
  /**
  * Returns a String representation of this <code>FVSupport</code>.
  * String is formed with the label and url of this <code>FVSupport</code>
  * 
  * @return a String representing this <code>FVSupport</code>
  */
  public abstract String toString();

}
