package tauzaman.field;

import org.w3c.dom.*;
import java.net.URL;
import java.util.*;


import tauzaman.field.fvsupport.FVSupport;
import tauzaman.XMLParser;
import tauzaman.XMLParseException;

/** 
* <code>FVTable</code> class represents a generic table 
* class for Field Value tables. 
*
*
* @author  Bedirhan Urgun and Curtis Dyreson
* @version 0.1, 10/10/02
* @status Design complete, implementation not started                     
*/

public class FVTable extends FVSupport{

  /**
  * A Hashtable, whose keys are indexes as String and values are string
  */
  private Hashtable fvtableIndexToString = null;

  /**
  * A Hashtable, whose keys are strings and values are indexes as String
  */
  private Hashtable fvtableStringToIndex = null;

  /** 
  * Constructs a <code>FVTable</code> object form a given label, url and 
  * a regex for this Table from Field Value Table XML file.
  *
  * @param url URL of actual information
  * @param regex regular expression that will be used to parse a temporal
  * constant input string
  *
  * @throws FVFormationException if any problem happens during formation of
  * field value table loading pointed by url
  *
  * @see tauzaman.calendricsystem.CalendricSystem 
  */
  public FVTable(URL url) throws FVFormationException{
      this.url = url;

      /* allocate memory for Hashtables */
      fvtableIndexToString = new Hashtable();
      fvtableStringToIndex = new Hashtable();

      /* get, parse and form */
      formFVTable();
  }
 
  /**
  * Forms a FVTable from url, which contains an xml file that includes
  * actual content of this <code>FVTable</code>.
  *
  * @throws FVFormationException if any problem happens during formation of
  * field value table loading pointed by url
  */
  private void formFVTable() throws FVFormationException{

      Element root = null;
      XMLParser parser = new XMLParser();
      
      /* first parse the specification file into DOM */
      try{
          root = parser.parseURL(url);
      }
      catch(XMLParseException xpe){
          throw new FVFormationException("Exception when parsing url: " + url.toExternalForm(), xpe);
      }
       
      /* get the regular expression */
      regex = root.getAttribute("regex");

      if(regex.equals(""))
          regex = null;

      /* get the rows */
      Vector rowNodes = parser.locator(root, "row");
      if(rowNodes.size() == 0)
          throw new FVFormationException("Exception when parsing url: " + 
           url.toExternalForm() + " no information in field value table!");

      for(int i = 0; i < rowNodes.size(); i++){
          Element rowNode = (Element)rowNodes.elementAt(i);

          /* assumption here is, both indexes and strings must be unique 
             through-out a field value table */

          String indexStr = rowNode.getAttribute("index");

          /* if no index is used then use default index (monotonically increasing, 
             which is loop counter) */
          if(indexStr.equals(""))
              indexStr = String.valueOf(i+1);

          if(fvtableIndexToString.containsKey(indexStr))
              throw new FVFormationException("Exception when parsing url: " + 
               url.toExternalForm() + " duplicate \"index\": "+ indexStr + " attr");

          String string = rowNode.getAttribute("string");
          if(string == null) 
              throw new FVFormationException("Exception when parsing url: " + 
                          url.toExternalForm() + " no \"string\" attr in row");
          
          if(fvtableStringToIndex.containsKey(string))
              throw new FVFormationException("Exception when parsing url: " + 
               url.toExternalForm() + " duplicate \"string\": "+ string + " attr");

          /* when we are done with all checks, add this tuple to the fvtable hashtables */
          fvtableIndexToString.put(indexStr, string);
          fvtableStringToIndex.put(string, indexStr);
      }
  }

  /** 
  * Converts a given string to corresponding index.
  * <pre>
  * <i>Given</i> <b>January</b> for Gregorian calendar and using 
  * a table called "EnglishMonthNames", this method returns
  * <b>1</b> as the index of <b>January</b>.
  * </pre>
  *
  * @param string string representation of input 
  *
  * @return long which is corresponding index of a string
  *
  * @throws FVServiceException if any problem occurs during process
  * of field value table method
  *
  * @see tauzaman.field.FVTable#indexToString
  */  
  public long stringToIndex(String string) throws FVServiceException{

      /* check if this fv table has corresponding index to this string */
      if(fvtableStringToIndex.containsKey(string)){
          String index = (String)(fvtableStringToIndex.get(string));
          return Long.parseLong((String)index);
      }
      throw new FVServiceException("Exception in FVservice in table(stringToIndex " + string + " ): " + 
                                                            url.toExternalForm() + " string not found!");
  }

  /** 
  * Converts a given long to corresponding string.
  * <pre>
  * <i>Given</i> <b>1</b> for Gregorian calendar and using 
  * a table called "EnglishMonthNames", this method returns
  * <b>January</b> as the string corresponding of <b>1</b>.
  * </pre>
  *
  * @param index long representation of input 
  *
  * @return String string corresponding of long index
  * @throws FVServiceException if any problem occurs during process
  * of field value table method
  *
  * @see tauzaman.field.FVTable#stringToIndex
  */  
  public String indexToString(long index) throws FVServiceException{
      
      String key = String.valueOf(index);

      /* check if fv table contains index */
      if(fvtableIndexToString.containsKey(key)){
          /* if it has get the corresponding value */
          return (String)fvtableIndexToString.get(key);
      }
      /* if fv table does not have corresponding value */
      throw new FVServiceException("Exception in FVservice in table (indexToString " + index + " ) : " + 
                                                              url.toExternalForm() + " index not found!");
  }  

  /**
  * toString method, which produces string representation of
  * this <code>FVTable</code>.
  * 
  * @return a String, which is a representation of this <code>FVTable</code> 
  */
  public String toString(){
      String result = "FVTable: " + url.toExternalForm();
      return result;
  }

}
