package tauzaman.calendricsystem.granularitylattice.cache;

/**
 * <p>Title: Cache</p>
 * <p>Description: This class implements a simple cache backed by a HashMap.</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * @author Jessica Miller
 * @version 1.0
 * @status implementation complete
 */

import java.util.HashMap;
import java.util.Iterator;

public class Cache
{
  /**
    * The cache data structure that stores cached objects indexed by keys.
    */
  private HashMap cacheMap;

  /**
    * The maximum size of the cache.
    */
  private int maxCacheSize;

  /**
    * This counter is used as a timestamp for cached objects.
    */
  private long timeCounter = 0;

  /**
    * Constructs a <code>Cache</code> object of a given size.
    *
    * @param cacheSize  the (maximal) size of the cache
    */
  public Cache(int cacheSize)
  {
    cacheMap     = new HashMap(cacheSize);
    maxCacheSize = cacheSize;
  }


  /**
    * Inserts an entry into the cache.
    *
    * @param key    the value by with the element is indexed in the cache
    * @param value  the element to be cached
    * @return       the old value that was in the cache if the key was already
    *                 present in the cache
    */
  public Object insert(Object key, Object value)
  {
    // if the cache has reached its maximal size, we must kick out the oldest
    //   cache entry
    if (cacheMap.size() >= maxCacheSize)
    {
      Iterator itr = cacheMap.keySet().iterator();

      Object oldestEntryKey  = itr.next();
      CacheEntry oldestEntry = (CacheEntry)cacheMap.get(oldestEntryKey);

      while (itr.hasNext())
      {
        Object nextKey       = itr.next();
        CacheEntry nextEntry = (CacheEntry)cacheMap.get(nextKey);

        if (nextEntry.timestamp < oldestEntry.timestamp)
        {
          oldestEntryKey = nextKey;
          oldestEntry    = nextEntry;
        }
      }

      cacheMap.remove(oldestEntryKey);
    }

    return cacheMap.put(key, new CacheEntry(value));
  }


  /**
    * Gets an element from the cache.
    *
    * @param key  the value which indexes the element to be gotten
    * @return     the element
    */
  public Object get(Object key)
  {
    CacheEntry ce = (CacheEntry)cacheMap.get(key);

    // if there was an entry in the cache for this key value, update its
    //   timestamp to the current time and return its value
    if (ce != null)
    {
      ce.timestamp = incrementTimeCounter();
      return ce.entry;
    }

    return ce;
  }


  /**
    * Increments the time counter of the cache.  If the time counter reaches
    *   the maximal value of a <code>long</code>, the cache is emptied and the
    *   time counter starts again at 0.
    *
    * @return  the incremented time counter
    */
  private long incrementTimeCounter()
  {
    timeCounter++;

    if (timeCounter == Long.MAX_VALUE)
    {
      // empty cache and start counter over
      cacheMap     = new HashMap(maxCacheSize);
      timeCounter = 0;
    }
    return timeCounter;
  }


  /**
    * Private inner class that is simply a wrapper around a cache entry so that
    *   the entry includes a timestamp.
    */
  private class CacheEntry
  {
    /**
      * The cache entry.
      */
    Object entry;

    /**
      * The timestamp that correlates to the last time that this cache entry was
      *   touched.
      */
    long timestamp;

    /**
      * Contruct a cache entry by wrapping this entry with a timestamp.
      *
      * @param entry  the entry to be cached.
      */
    public CacheEntry(Object entry)
    {
      this.entry = entry;
      timestamp = incrementTimeCounter();
    }
  }
}