package tauzaman.calendar.mapping;

import tauzaman.calendricsystem.Granularity;

/**
* <code>Mapping</code> abstract class represents the granularity
* mapping information of a <code>Calendar</code>.
*
* @author  Bedirhan Urgun
* @version 0.1, 11/15/02
*
* @see     tauzaman.calendar.Calendar
* @see     tauzaman.calendar.IrregularMapping
* @see     tauzaman.calendar.RegularMapping
* @see     tauzaman.calendricsystem.Granularity
*
* @status design complete, implementation complete
*/
public abstract class Mapping{

   /*
   * Constants corresponding to relationship between <code>Granularities</code>
   * of this <code>Mapping</code>. There are three possible values for a relation;
   * <ul>
   *  <li>
   *    This <code>Mapping</code> might be declared from a finer <code>Granularity</code> to
   *    coarser <code>Granularity</code>,
   *  </li>
   *  <li>
   *    This <code>Mapping</code> might be declared from a coarser <code>Granularity</code> to
   *    finer <code>Granularity</code>,
   *  </li>
   *  <li>
   *    This <code>Mapping</code> might be declared between <code>Granularities</code>, which
   *    does not have a rigid relation between them.
   *  </li>
   * </ul>
   */
   public static final int FINER_TO_COARSER = 0;
   public static final int COARSER_TO_FINER = 1;
   public static final int CONGRUENT = 2;

   /**
   * Relationship between the <code>Granularities</code> of this <code>Mapping</code>
   */
   protected int relationship;

   /**
   * to <code>Granularity</code>
   */
   protected Granularity to = null;

   /**
   * from <code>Granularity</code>
   */
   protected Granularity from = null;

   /**
   * Constructs a <code>Mapping</code> from "from" <code>Granularity</code>
   * to "to" <code>Granularity</code>.
   *
   * @param to <code>Granularity</code> that this <code>Mapping</code> starts
   * @param from <code>Granularity</code> that this <code>Mapping</code> ends
   * @param relationship (finer/coarser/congruent) relationship between
   * <code>Mapping</code> <code>Granularities</code>
   *
   */
   public Mapping(Granularity to, Granularity from, int relationship){
       this.to = to;
       this.from = from;
       this.relationship = relationship;
   }

   /**
   * Returns relationship of this <code>Mapping</code>
   *
   * @return relationship type of this <code>Mapping</code>
   */
   public int getRelationship(){
       return relationship;
   }

   /**
   * Given a string corresponding of a relationship, used as in xml spec. files,
   * returns int code of that relationship.
   *
   * @param relationship String corresponding of a relationship
   * @return int corresponding of input relationship, if input does not
   * map into any valid relationships, then -1 is returned
   */
   public static int fetchRelationship(String relationship){
       if(relationship.equals("finerToCoarser") || relationship.equals(""))
           return 0;
       else if(relationship.equals("coarserToFiner"))
           return 1;
       else if(relationship.equals("congruent"))
           return 2;
       return -1;
   }

   /**
   * Given a string corresponding of a relationship, used as in xml spec. files,
   * returns int code of that relationship.
   *
   * @param relationship integer corresponding of a relationship
   * @return String corresponding of input relationship, if input does not
   * map into any valid relationships, then null is returned
   */
   // this may be changed to public later on
   private static String fetchRelationship(int relationship){
       switch(relationship){
           case 0: return "finerToCoarser";
           case 1: return "coarserToFiner";
           case 2: return "congruent";
       }
       return null;
   }

   /**
   * Returns to <code>Granularity</code>
   *
   * @return to <code>Granularity</code> of this <code>Mapping</code>
   */
   public Granularity getTo(){
       return to;
   }

   /**
   * Returns from <code>Granularity</code>
   *
   * @return from granularity of this mapping
   */
   public Granularity getFrom(){
       return from;
   }

   /**
   * Returns true if this <code>Mapping</code> is equal to the
   * given <code>Mapping</code>. False otherwise.
   *
   * @param other <code>Mapping</code> to be compared with this
   * <code>Mapping</code>
   *
   * @return boolean value true if <code>Granularities</code> of
   * two <code>Mapping</code> matches. False otherwise.
   */
   public boolean equals(Object other){
       Mapping otherMapping = (Mapping)other;
       if( to.equals(otherMapping.getTo()) && from.equals(otherMapping.getFrom()) )
           return true;
       return false;
   }

   /**
   * Casts a given anchored point.
   *
   * @param point point in time to be casted
   * @return long casted point
   */
   public abstract long performAnchoredCast(long point);

   /**
   * Scales a given anchored point.
   *
   * @param point point in time to be scaled
   * @return long value(s) of scaled point
   */
   public abstract long [] performAnchoredScale(long point);

   /**
   * Casts a given unanchored point.
   *
   * @param point point in time to be casted
   * @return long casted point
   */
   public abstract long performUnanchoredCast(long point);

   /**
   * Scales a given unanchored point.
   *
   * @param point point in time to be scaled
   * @return long value(s) of scaled point
   */
   public abstract long [] performUnanchoredScale(long point);

   /**
   * toString method as abstract
   *
   * @return string representation of this object
   */
   public String toString() {
     return "Mapping: " + fetchRelationship(relationship) + " " +
         from.getLocalName() + "->" + to.getLocalName();
   }

}
