package tauzaman.calendar;

import java.net.*;
import java.util.*;
import org.w3c.dom.*;
import java.lang.*;
import java.lang.reflect.*;


import tauzaman.calendar.mapping.Mapping;
import tauzaman.field.Field;
import tauzaman.field.Fields;
import tauzaman.timestamp.*;
import tauzaman.calendricsystem.*;

import tauzaman.*;

/**
* <p>
* <code>Calendar</code> class represents a Calendar, which defines the time values
* of interest to user, usually over a specific segment of the physical time-line.
* <code>Calendar</code> keeps Granularity/additional FieldName(s) information and
* provides Calendar dependent functions for time-stamp/temporal constant conversions.
* </p> <p>
* <code>Calendar</code>s can be formed given Calendar specification files' <code>URL
* </code>s, which are fetched as String and formed by <code>CalendricSystem</code>
* from Calendric System specification files. For example;
* <pre>
*  <code>URL</code> <b>url</b> = new <code>URL</code>("http://www.eecs.wsu.edu/burgun/calendars/Gregorian/GregV1.cal");
*  Calendar calendar = new Calendar(<b>url</b>);
* </pre>
* </p> <p>
* Also <code>Calendar</code>s are kept in a database called <code>CalendarRepository</code>.
* </p>
*
*
* @author  Bedirhan Urgun and Curtis Dyreson
* @version 0.1, 10/10/02
* @status Design complete, implementation not started
* @see     tauzaman.calendar.CalendarRepository
* @see     tauzaman.calendricsystem.CalendricSystem
* @see     tauzaman.ClassLoaderMethodCaller
*/

public class Calendar {

  /**
  * Url of this <code>Calendar</code>, which is the Primary Key in <code>CalendarRepository</code>.
  */
  private URL url = null;

  /**
  * Url of this <code>Calendar</code> implementation.
  */
  private URL implUrl = null;

  /**
  * A <code>ClassLoaderMethodCaller</code> instance that provides access to
  * <b>regular</b> functions, which are loaded dynamically.
  * <b>regular</b> functions refer to two functions, which all <code>Calendar</code>s
  * should have: "<i>granuleToFields</i> and <i>fieldsToGranule</i>"
  */
  private ClassLoaderMethodCaller clmc = null;

  /**
  * Array of <code>Granularities</code> that this <code>Calendar</code> contains.
  */
  private Granularity granularities []= null;

  /**
  * Array of <code>Mappings</code> that this <code>Calendar</code> contains.
  */
  private Mapping mappings []= null;

  /**
  * Underlying <code>Granularity</code> of this <code>Calendar</code>.
  */
  private Granularity underlyingGranularity = null;

  /**
  * Constructs a Calendar object from a given
  * <a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepcalendarspec.htm">Calendar Specification</a> url in a
  * <a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepcalendricspec.htm">Calendric System Specification</a> file.
  *
  * @param url URL of Calendar Specification file.
  *
  * @throws CalendarFormationException If any problem
  * happens while parsing the file; Any problem related to xml parsing,
  * any bad information fetched from parsed file, such as, a bad URL or
  * a URL that causes problems with dynamic loading.
  */
  public Calendar(URL url) throws CalendarFormationException{
    this.url = url;

    /* call formCalendar() to initialize the Calendar's state */
    formCalendar();
  }

  /**
  * Initializes the state of this <code>Calendar</code> using
  * the Calendar Specification file pointed by a URL.
  *
  * @throws CalendarFormationException If any problem
  * happens while parsing the file; Any problem related to xml parsing,
  * any bad information fetched from parsed file, such as, a bad URL or
  * a URL that causes problems with dynamic loading.
  *
  */
  private void formCalendar() throws CalendarFormationException {

     /* Use XMLParser.parse(url) to get the root node, then parse and form it */

      XMLParser parser = new XMLParser();

      Element root = null;

      try{
          root = parser.parseURL(url);
      }
      catch(XMLParseException xpe){
          throw new CalendarFormationException("Exception when parsing xml file in: " + url.toExternalForm(), xpe);
      }


      /* calendar implementation url as string */
      String stringImplUrl = (root.getAttribute("implUrl")).trim();

      /* form URL */
      try{
          implUrl = new URL(stringImplUrl);
      }
      catch(MalformedURLException e){
          throw new CalendarFormationException("Exception when parsing xml file in: " + url.toExternalForm() + " bad URL!", e);
      }

      /***************CLMC FORMATION*****************/

      /* form CLMC instance here by using implUrl */
      URL urls[] = new URL[1];

      String classUrl = null, className = null;

      //get the useful part of URL
      if(stringImplUrl.endsWith(".class")){
          classUrl = stringImplUrl.substring(0, stringImplUrl.lastIndexOf('/') + 1);
          className = stringImplUrl.substring(stringImplUrl.lastIndexOf('/') + 1, stringImplUrl.length());
          className = className.substring(0, className.lastIndexOf(".class"));
      }
      else
          throw new CalendarFormationException("Exception when forming calendar: "+ url.toExternalForm() +
                                                                   " should end with .class: " + stringImplUrl);
      /* form the url array */
      try{ urls[0] = new URL(classUrl); }
      catch(MalformedURLException e){
          throw new CalendarFormationException("Exception when forming URL from: " + classUrl, e);
      }

      /* form instance of CLMC */
      clmc = new ClassLoaderMethodCaller(urls);

      /* load the class in CLMC */
      try{
          /* fix method names */
          clmc.loadClass(className, "fieldsToGranule");
          clmc.loadClass(className, "granuleToFields");
      }
      catch(DynamicLoadException dle){
          throw new CalendarFormationException("Exception when loading class:" + className +
                                                                    " in: " + url.toExternalForm(), dle);
      }

      /***************GRANULARITY AND MAPPING FORMATION*****************/

      /* get Granularity declarations */
      Vector granularityNodes = parser.locator(root, "granularity");

      /* allocate memory for granularity and mapping hashtables */
      Hashtable granularityTable = new Hashtable(), mappingTable =  new Hashtable();

      for(int i = 0; i < granularityNodes.size(); i++){
          Element granularityNode = (Element)(granularityNodes.elementAt(i));

          /* from granularity name and calendar url form a Granularity object */
          Granularity granularity = new Granularity(granularityNode.getAttribute("name"), url);

          /* check if this granularity is already declared */
          if(granularityTable.containsKey(granularity))
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                                       url.toExternalForm() + " granularity " +
                                                granularity.getLocalName() + " already defined!");

          /* add this granularity to granularity table */
          granularityTable.put(granularity, "dummy");

          /* declare regular mappings of this granularity, if exists */
          Vector regularMappingNodes = parser.locator(granularityNode, "regularMapping");
          for(int j = 0; j < regularMappingNodes.size(); j++){
              Element regularMappingNode = (Element)(regularMappingNodes.elementAt(j));

              /* from from attribute value and this calendar url form a granularity */
              Granularity fromGranularity = new Granularity(regularMappingNode.getAttribute("from"), url);

              /* get relationship between granularity and fromGranularity */
              int relationship = Mapping.fetchRelationship(regularMappingNode.getAttribute("relationship"));

              if(relationship == -1)
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " relationship undefined given as: " +
                                     regularMappingNode.getAttribute("relationship"));

              /* form a regular mapping */
              RegularMapping rm = new RegularMapping(granularity, fromGranularity, relationship, regularMappingNode);
                 /* *** jkm */
              RegularMapping rm2 = new RegularMapping(fromGranularity, granularity, getConverseRel(relationship), regularMappingNode);

              /* check for duplicate regular mappings */
              if(mappingTable.containsKey(rm) ||  mappingTable.containsKey(rm2))
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " regular mapping between " +  granularity.getLocalName() +
                                     " and " + fromGranularity.getLocalName() + " already defined!");

              /* add this regular mapping to mapping table */
              mappingTable.put(rm, "dummy");
              mappingTable.put(rm2, "dummy");
          }

          /* declare irregular mappings, if exists */
          Vector irregularMappingNodes = parser.locator(granularityNode, "irregularMapping");
          for(int j = 0; j < irregularMappingNodes.size(); j++){
              Element irregularMappingNode = (Element)(irregularMappingNodes.elementAt(j));

              /* from from attribute value and this calendar url form a granularity */
              Granularity fromGranularity = new Granularity(irregularMappingNode.getAttribute("from"), url);

              /* get irregular mapping url */
              String irregularMappingUrl = (irregularMappingNode.getAttribute("url")).trim();

              /* default url for irregular mappings is the implementation url of its calendar */
              if(irregularMappingUrl.equals(""))
                  irregularMappingUrl = implUrl.toExternalForm();

              /* get relationship between granularity and fromGranularity */
              int relationship = Mapping.fetchRelationship(irregularMappingNode.getAttribute("relationship"));

              if(relationship == -1)
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " relationship undefined given as: " +
                                     irregularMappingNode.getAttribute("relationship"));

              /* form the Irregular Mapping */
              IrregularMapping irm = null;
              try{
                  irm = new IrregularMapping(granularity, fromGranularity, relationship,
                                     new URL(irregularMappingUrl), irregularMappingNode);
              }
              catch(MalformedURLException e){
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                                          url.toExternalForm() + " bad URL!", e);
              }
              catch(IrregularMappingFormationException ie){
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                    url.toExternalForm() + " cannot form irregular mapping", ie);
              }

              /* check for duplicate Irregular Mappings */
              if(mappingTable.containsKey(irm))
                  throw new CalendarFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " irregular mapping between " +  granularity.getLocalName() +
                                     " and " + fromGranularity.getLocalName() + " already defined!");

               /* put the Irregular Mapping into the mapping Table */
               mappingTable.put(irm, "dummy");

          }
      }

      /* convert mappingTable Hashtable to mappings array and
         granularityTable to granularities array */

      mappings = new Mapping [mappingTable.size()];
      int i = 0;
      for(Enumeration e = mappingTable.keys(); e.hasMoreElements();){
          mappings[i] = (Mapping)e.nextElement();
          i++;
      }

      granularities = new Granularity [granularityTable.size()];
      i = 0;
      for(Enumeration e = granularityTable.keys(); e.hasMoreElements();){
          granularities[i] = (Granularity)e.nextElement();
          i++;
      }


      /********UNDERLYING (default) GRANULARITY************/

      /* default granularity is the underlying granularity specified in spec. */

      underlyingGranularity = getGranularityByLocalName(root.getAttribute("underlyingGranularity"));

      if(underlyingGranularity == null)
          throw new CalendarFormationException("Exception when parsing xml file in: " +
                             url.toExternalForm() + " underlyingGranularity: " +
                             root.getAttribute("underlyingGranularity") + " is undefined");

  }

  /**
  * Returns <code>Granularity</code> of this <code>Calendar</code> given its local name.
  *
  * @param localName String local name of a <code>Granularity</code>
  * @return <code>Granularity</code> of this <code>Calendar</code>
  * given its local name. Returns null if no such <code>Granularity</code> exists.
  */
  public Granularity getGranularityByLocalName(String localName){
      for(int i = 0; i < granularities.length; i++)
          if(localName.equals(granularities[i].getLocalName()))
              return granularities[i];
      return null;
  }

  /**
  * Returns the URL, which is also Primary Key in
  * <code>CalendarRepository</code>, of this <code>Calendar</code>.
  *
  * @return URL of this <code>Calendar</code>
  */
  public URL getUrl() {
    return url;
  }

  /**
  * Returns true if URL of this <code>Calendar</code> is equal to
  * given <code>Calendar</code>.
  *
  * @return boolean true if this <code>Calendar</code> equals to
  * given <code>Calendar</code>
  */
  public boolean equals(Object obj){
      Calendar otherCalendar = (Calendar)obj;
      if( url.equals(otherCalendar.getUrl()) )
          return true;
      return false;
  }

  /**
  * <p>
  * Forms and returns a <code>Granule</code>, which contains a <code>Granularity</code>,
  * given <code>Fields</code> object, which is formed by <code>Input</code> by parsing
  * a temporal constant input string. This is a <b>regular</b> <code>Calendar</code>
  * function, which means every <code>Calendar</code> is required to implement it.
  * </p>
  *
  * <p>
  * This method calls actual implementation, which is loaded dynamically from given
  * URL in <a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepcalendarspec.htm">Calendar
  * Specification</a> file.
  * </p>
  *
  *
  * @param fields temporal constant input string, which is parsed into its <code>Field</code>s
  * @param helper name of the <code>Property</code> format, which is used successfully to parse
  * temporal constant. This might be used by <code>Calendar</code> to implement this method
  * efficiently.
  * @return a <code>Granule</code> formed given <code>Fields</code> object
  *
  * @throws CalendarServiceException if any problem occurs when performing
  * this dynamically loaded method, this also includes problems with dynamic loading
  * exceptions.
  *
  * @see tauzaman.field.Fields
  * @see tauzaman.field.Field
  * @see tauzaman.timestamp.Granule
  * @see tauzaman.property.Property
  */
  /* should return one granule, takes nothing but fields */
  public Granule fieldsToGranule(Fields fields) throws CalendarServiceException {

      /* prepare the parameters, which is "Fields" passed, and
         call dynamically loaded method */

      Granule granule = null;

      Class parameterTypes [] = new Class [1];

      try{
          parameterTypes[0] = Class.forName("tauzaman.field.Fields");
      }
      catch(ClassNotFoundException cnfe){
          throw new CalendarServiceException("Exception in Calendar's fieldsToGranule method: package corrupted", cnfe);
      }

      try{

          Object args[] = new Object[1];
          args[0] = fields;

          granule = (Granule) (clmc.callMethod("fieldsToGranule", parameterTypes, args));

      }
      catch(DynamicLoadException dle){
          throw new CalendarServiceException("Exception in Calendar's fieldsToGranule method: " +
                                                                        url.toExternalForm(), dle);
      }

      return granule;
  }

  /**
  * <p>
  * Forms and returns a <code>Fields</code>, which is unpacked version of a
  * timestamp parsed into each <code>Field</code>. This is a <b>regular</b>
  * <code>Calendar</code> function, which means every <code>Calendar</code>
  * is required to implement it.
  * </p>
  *
  * <p>
  * This method calls actual implementation, which is loaded dynamically from given
  * URL in <a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepcalendarspec.htm">Calendar
  * Specification</a> file.
  * </p>
  *
  *
  * @param granules an array of <code>Granule</code> objects, which is a timestamp(s) that
  * corresponds to a temporal constant string
  * @param helper name of the <code>Property</code> format, which will be used to parse
  * timestamp. This might be used by <code>Calendar</code> to implement this method
  * efficiently.
  *
  * @throws CalendarServiceException if any problem occurs when performing
  * this dynamically loaded method, this also includes problems with dynamic loading
  * exceptions.
  *
  * @see tauzaman.field.Fields
  * @see tauzaman.field.Field
  * @see tauzaman.timestamp.Granule
  * @see tauzaman.property.Property
  */
  /* should not return anything, instead of helper takes fields */
  public boolean granuleToFields(Granule granule, Fields fields) throws CalendarServiceException{

      /* prepare the parameters, which is "Fields" and "Granule" passed, and
         call dynamically loaded method */

      Class parameterTypes [] = new Class [2];

      try{
          parameterTypes[0] = Class.forName("tauzaman.timestamp.Granule");
          parameterTypes[1] = Class.forName("tauzaman.field.Fields");
      }
      catch(ClassNotFoundException cnfe){
          throw new CalendarServiceException("Exception in Calendar's granuleToFields method: package corrupted", cnfe);
      }

      try{

          Object args[] = new Object[2];
          args[0] = granule;
          args[1] = fields;

          Boolean returnValue = (Boolean)clmc.callMethod("granuleToFields", parameterTypes, args);

          if(returnValue.booleanValue())
              return true;

      }
      catch(DynamicLoadException dle){
          throw new CalendarServiceException("Exception in Calendar's granuleToFields method: " +
                                                                       url.toExternalForm(), dle);
      }

      return false;
  }


  /**
  * Returns <code>Granularities</code> of this
  * <code>Calendar</code>.
  * @return an array of <code>Granularities</code>
  */
  public Granularity [] getGranularities(){
      return granularities;
  }

  /**
  * Returns <code>Mapping</code>s of this
  * <code>Calendar</code>.
  * @return an array of <code>Mapping</code>s
  */
  public Mapping [] getMappings(){
      return mappings;
  }

  /**
  * Returns underlying <code>Granularity</code> of this
  * <code>Calendar</code>.
  * @return underlying <code>Granularity</code>
  */
  public Granularity getUnderlyingGranularity(){
      return underlyingGranularity;
  }

  /**
  * Returns string representation of this <code>Calendar</code>.
  *
  * @return String representation of a URL
  */
  public String toString() {
    StringBuffer rep = new StringBuffer();
    rep.append("\nCalendar: " + url.toExternalForm() + "\n with impl url " + implUrl.toExternalForm());
    rep.append("\nUnderlying Granularity: " + underlyingGranularity.getLocalName() + "\n");
    for(int i = 0; i < granularities.length; i++)
        rep.append(granularities[i].toString());
    for(int i = 0; i < mappings.length; i++)
        rep.append(mappings[i].toString());
    return rep.toString();
  }

  /* *** jkm */
  private int getConverseRel(int rel)
  {
    if (rel == Mapping.FINER_TO_COARSER)
      return Mapping.COARSER_TO_FINER;
    else if (rel == Mapping.COARSER_TO_FINER)
      return Mapping.FINER_TO_COARSER;
    else
      return Mapping.CONGRUENT;
  }
}

