package tauzaman;

import java.rmi.*;
import java.net.*;
import java.io.*;

import tauzaman.field.*;
import tauzaman.property.*;
import tauzaman.calendricsystem.*;
import tauzaman.calendar.*;

/**
* <code>TauZamanSystem</code> is the entrance point for TauZaman services.
* It contains methods, which provides services to interact further Calendar
* related services. It contains all repositories, which will be used as 
* a database for all services.
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/

public class TauZamanSystem{

  /** <code>TauZamanRemoteServiceHandler</code> object, which handles remote services of 
      this <code>TauZamanSystem</code> when it is set as server  **/
  private TauZamanRemoteServiceHandler tzRemoteServiceHandler = null;

  /** A handle to unique <code>CalendricSystemRepository</code> of this <code>TauZamanSystem</code> **/
  private CalendricSystemRepository csr = null;
 
  /** A handle to unique <code>PropertyRepository</code> of this <code>TauZamanSystem</code> **/
  private PropertyRepository pr = null;
 
  /** A handle to unique <code>FVSupportRepository</code> of this <code>TauZamanSystem</code> **/
  private FVSupportRepository fvsr = null;

  /** A handle to unique <code>CalendarRepository</code> of this <code>TauZamanSystem</code> **/
  private CalendarRepository cr = null;

  /** Active <code>TauZamanService</code> of this <code>TauZamanSystem</code> **/
  static private TauZamanService activeService = null;

  /** Name that is used for RMI registry name binding **/
  private String serverRegistryName = null;


  /* Constructor will not form a default local or remote service. Since there is no way
     to return the formed service cleanly back to user. Remember, our assumption is,
     user should have a clear view of what is remote service or what is local service. */

  /**
  * Constructs a <code>TauZamanSystem</code> object, which provides 
  * local or remote <code>TauZamanService</code>s, and also has the
  * capability of being a TauZaman server.
  *
  * @see tauzaman.calendricsystem.CalendricSystemRepository
  * @see tauzaman.calendar.CalendarRepository
  * @see tauzaman.field.FVSupportRepository
  * @see tauzaman.property.PropertyRepository
  * @see tauzaman.TauZamanService 
  */ 
  public TauZamanSystem(){

      cr = new CalendarRepository();

      /* CalendricSystemRepository will use CalendarRepository */
      csr = new CalendricSystemRepository(cr); 

      pr = new PropertyRepository();

      fvsr = new FVSupportRepository();

  }

  /**
  * Returns a local <code>TauZamanService</code> object, with a default <code>CalendricSystem</code> and
  * a default <code>Properties</code>, ready to serve.
  * Additionally, this <code>TauZamanService</code> will be set as active <code>TauZamanService</code>
  * in this system. And also loaded default <code>CalendricSystem</code> will be set as active
  * <code>CalendricSystem</code> in this <code>TauZamanService</code> service.
  * 
  * @param calendricSystemName String name, given by user, which loaded default <code>CalendricSystem</code>
  * will be referred as
  * @param calendricSystemUrl default <code>CalendricSystem</code> for this <code>TauZamanService</code>
  * @param defaultPropertiesUrl default <code>Properties</code> related to default <code>CalendricSystem</code>
  * 
  * @return a <code>TauZamanLocalService</code> ready to serve
  * @throws TauZamanException if any abnormal condition occurs when getting a local service
  */
  public TauZamanLocalService getLocalService(String calendricSystemName, 
                              URL calendricSystemUrl, URL defaultPropertyListUrl) throws TauZamanException{

      TauZamanLocalService tzls = new TauZamanLocalService(this, calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);

      /* side effect of this method */
      setActiveService(tzls);
      return tzls;
  }

  /**
  * Returns a remote <code>TauZamanService</code> object, with a default <code>CalendricSystem</code> and
  * a default <code>Property</code> table, ready to serve.
  * Additionally, this <code>TauZamanService</code> will be set as active <code>TauZamanService</code>
  * in this system. And also loaded default <code>CalendricSystem</code> will be set as active
  * <code>CalendricSystem</code> in this <code>TauZamanService</code> service.
  * 
  * @param host url of remote TauZaman system
  * @param port port of remote TauZaman system
  * @param serviceName String name under which remote TauZaman system serves
  * @param calendricSystemName String name, given by user, which loaded default <code>CalendricSystem</code>
  * will be referred as
  * @param calendricSystemUrl default <code>CalendricSystem</code> for this <code>TauZamanService</code>
  * @param defaultPropertiesUrl default <code>Properties</code> related to default <code>CalendricSystem</code>
  *
  * @return a <code>TauZamanRemoteService</code> ready to serve
  * @throws TauZamanException if any abnormal condition occurs when getting requested remote service
  */
  public TauZamanRemoteService getRemoteService(String host, String port, String serviceName, String calendricSystemName, 
                                 URL calendricSystemUrl, URL defaultPropertyListUrl) throws TauZamanException{

      TauZamanRemoteServiceHandler tzRemoteServiceHandler = null;
      TauZamanRemoteService tzrs = null;
      
      /* we do we specify default host and port names? Java RMI has already defaults. Answer: We may change defaults. */
      if(host == null)
          host = "localhost";
      if(port == null)
          port = "1099";
      if(serviceName == null)
          serviceName = "TauZamanService";

      try{
          tzRemoteServiceHandler = (TauZamanRemoteServiceHandler) Naming.lookup("rmi://" + host + ":" + port + "/" + serviceName);
          tzrs = tzRemoteServiceHandler.getService(calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);
      }
      catch(RemoteException re){
          throw new TauZamanException("Exception when getting a remote service!", re);
      }
      catch(NotBoundException nbe){
          throw new TauZamanException("Exception when getting a remote service!", nbe);
      }
      catch(MalformedURLException murle){   
          throw new TauZamanException("Exception when getting a remote service!", murle);
      }
  
      /* side effect of this method */
      setActiveService(tzrs);

      return tzrs;
  }

  /**
  * Returns a remote <code>TauZamanService</code> object, with a default <code>CalendricSystem</code> and
  * a default <code>Property</code> table, ready to serve.
  * Additionally, this <code>TauZamanService</code> will be set as active <code>TauZamanService</code>
  * in this system. And also loaded default <code>CalendricSystem</code> will be set as active
  * <code>CalendricSystem</code> in this <code>TauZamanService</code> service.
  * 
  * @param calendricSystemName String name, given by user, which loaded default <code>CalendricSystem</code>
  * will be referred as
  * @param calendricSystemUrl default <code>CalendricSystem</code> for this <code>TauZamanService</code>
  * @param defaultPropertiesUrl default <code>Properties</code> related to default <code>CalendricSystem</code>
  *
  * @return a <code>TauZamanRemoteService</code> ready to serve
  * @throws TauZamanException if any abnormal condition occurs when getting requested remote service
  */
  public TauZamanRemoteService getRemoteService(String calendricSystemName, URL calendricSystemUrl, 
                                               URL defaultPropertyListUrl) throws TauZamanException{

      TauZamanRemoteServiceHandler tzRemoteServiceHandler = null;
      TauZamanRemoteService tzrs = null;
      
      try{
          tzRemoteServiceHandler = (TauZamanRemoteServiceHandler) Naming.lookup("rmi://localhost:1099/TauZamanService");
          tzrs = tzRemoteServiceHandler.getService(calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);
      }
      catch(RemoteException re){
          throw new TauZamanException("Exception when getting a remote service!", re);
      }
      catch(NotBoundException nbe){
          throw new TauZamanException("Exception when getting a remote service!", nbe);
      }
      catch(MalformedURLException murle){   
          throw new TauZamanException("Exception when getting a remote service!", murle);
      }
  
      /* side effect of this method */
      setActiveService(tzrs);
      return tzrs;
  }


  /* we can't use registry of a non-local host for remote service handler object */

  /**
  * Sets this <code>TauZamanService</code> as a server, which provides 
  * Calendar related services, by using Java RMI, to various TauZaman clients. 
  * 
  * @param port port of host registry that this server will make use of for publishing
  * remote service
  * @param serviceName String name of remote service that clients can use for
  * their connection
  * @param log a boolean value, which determines if this server will keep logs
  * of remote calls. If it is true, server keeps log, o.w. server will not keep log  
  * 
  * @return true if remote service is set successfully, and returns false if and only
  * if this <code>TauZamanSystem</code> is already set as a server
  *
  * @throws TauZamanException if any abnormal condition occurs when setting this
  * <code>TauZamanSystem</code> as a server. A frequent exception might be using a service
  * name, which is already in use. A possible solution is to use package-like names
  * for services, such as; myTauZaman.myRemoteService
  */
  public boolean setRemoteService(String port, String serviceName, boolean log) throws TauZamanException{

      if(tzRemoteServiceHandler == null){

          /* set the defaults if necessary */
 
          /* we do we specify default host and port names? Java RMI has already defaults. 
             Answer: We may change defaults. */

          if(port == null)
              port = "1099";
          if(serviceName == null)
              serviceName = "TauZamanService";

          /* keep registry name for unbind purposes */
          serverRegistryName = "rmi://localhost:" + port + "/" + serviceName;

          try{
              tzRemoteServiceHandler = new TauZamanRemoteServiceHandlerImpl(this, log);
              Naming.bind("rmi://localhost:" + port + "/" + serviceName, tzRemoteServiceHandler);
          }
          catch(MalformedURLException mue){
              throw new TauZamanException("Exception when setting a remote service!", mue);
          }
          catch(AlreadyBoundException abe){
              throw new TauZamanException("Exception when setting a remote service!", abe);
          }
          catch(AccessException ae){
              throw new TauZamanException("Exception when setting a remote service!", ae);
          }
          catch(RemoteException re) {
              throw new TauZamanException("Exception when setting a remote service!", re);
          }
          catch(FileNotFoundException fnfe){
              throw new TauZamanException("Exception when setting a remote service!", fnfe);
          }

          return true;
      }      

      /* if it is not null, that means we already set it as remote */
      return false;
  }

  /**
  * Sets this <code>TauZamanService</code> as a server, which provides 
  * Calendar related services, by using Java RMI, to various TauZaman clients. 
  * 
  * @param log a boolean value, which determines if this server will keep logs
  * of remote calls. If it is true, server keeps log, o.w. server will not keep log  
  *
  * @return true if remote service is set successfully, and returns false if and only
  * if this <code>TauZamanSystem</code> is already set as a server
  *
  * @throws TauZamanException if any abnormal condition occurs when setting this
  * <code>TauZamanSystem</code> as a server.A frequent exception might be using a service
  * name, which is already in use
  */
  public boolean setRemoteService(boolean log) throws TauZamanException{

      if(tzRemoteServiceHandler == null){

          /* keep registry name for unbind purposes */
          serverRegistryName = "rmi://localhost:1099/TauZamanService";

          try{
              tzRemoteServiceHandler = new TauZamanRemoteServiceHandlerImpl(this, log);
              Naming.bind("rmi://localhost:1099/TauZamanService", tzRemoteServiceHandler);
          }
          catch(AccessException ae){
              throw new TauZamanException("Exception when setting a remote service!", ae);
          }
          catch(MalformedURLException mue){
              throw new TauZamanException("Exception when setting a remote service!", mue);
          }
          catch(AlreadyBoundException abe){
              throw new TauZamanException("Exception when setting a remote service!", abe);
          }
          catch(RemoteException re) {
              throw new TauZamanException("Exception when setting a remote service!", re);
          }
          catch(FileNotFoundException fnfe){
              throw new TauZamanException("Exception when setting a remote service!", fnfe);
          }

          return true;
      }      

      /* if it is not null, that means we already set it as remote */
      return false;
  }

  /**
  * Unsets this <code>TauZamanService</code> as a server.
  *
  * @return true when this <code>TauZamanService</code> successfully unset. Returns false
  * if and only if this <code>TauZamanService</code> was never set as a server.
  *
  * @throws TauZamanException if any abnormal condition occurs when unsetting this
  * <code>TauZamanSystem</code> as a server.
  */
  public boolean unsetRemoteService() throws TauZamanException{
      
      if(tzRemoteServiceHandler != null){

          try{
              Naming.unbind(serverRegistryName);
          }
          catch(NotBoundException nbe){
              throw new TauZamanException("Exception when setting a remote service!", nbe);
          }
          catch(RemoteException re) {
              throw new TauZamanException("Exception when setting a remote service!", re);
          }
          catch(MalformedURLException mue){
              throw new TauZamanException("Exception when setting a remote service!", mue);
          }

          return true;
      }

      /* if it is null, that means it was never set as remote */
      return false;
  }


  /**
  * Sets a <code>TauZamanLocalService</code> object as the active 
  * <code>TauZamanService</code> of this <code>TauZamanSystem</code>.
  *
  * @param tzls <code>TauZamanLocalService</code> object to be set active
  * @see tauzaman.TauZamanLocalService
  * @throws TauZamanException if this service was not set to a valid <code>TauZamanLocalService</code>
  */
  public void setActiveService(TauZamanLocalService tzls) throws TauZamanException{
      activeService = new TauZamanService(tzls);
  }

  /**
  * Sets a <code>TauZamanService</code> object as the active 
  * <code>TauZamanService</code> of this <code>TauZamanSystem</code>.
  *
  * @param tzs <code>TauZamanService</code> object to be set active
  * @see tauzaman.TauZamanService
  */
  public void setActiveService(TauZamanService tzs){
      activeService = tzs;
  }

  /**
  * Sets a <code>TauZamanRemoteService</code> object as the active 
  * <code>TauZamanService</code> of this <code>TauZamanSystem</code>.
  *
  * @param tzrs <code>TauZamanRemoteService</code> object to be set active
  * @see tauzaman.TauZamanRemoteService
  * @throws TauZamanException if this service was not set to a valid <code>TauZamanRemoteService</code>
  */
  public void setActiveService(TauZamanRemoteService tzrs) throws TauZamanException{
      activeService = new TauZamanService(tzrs);
  }

  /**
  * Returns a handle to active <code>TauZamanService</code> object
  * of this <code>TauZamanSystem</code>.
  *
  *
  * @return a handle to active <code>TauZamanService</code>
  * @see tauzaman.TauZamanService
  */
  public static TauZamanService getActiveService(){
      return activeService;
  }

  /* as a system owner, do I need to access them? For now, no */
  /* it would be better to change them to package access modifier,
     remove protected. */
  
  /**
  * Returns a handle to <code>CalendricSystemRepository</code> object, which provides all
  * <code>Calendar</code> related services of this system. 
  *
  * @return a handle to <code>CalendricSystemRepository</code> object
  * @see tauzaman.calendricsystem.CalendricSystemRepository
  */
  protected CalendricSystemRepository getCalendricSystemRepository(){
      return csr;
  }
 
  /* seems we don't need getCalendarRepository method */

  /**
  * Returns a handle to <code>CalendarRepository</code> object, which provides all
  * <code>Calendar</code>s to <code>CalendricSystemRepository</code> this system. 
  *
  * @return a handle to <code>CalendarRepository</code> object
  * @see tauzaman.calendricsystem.CalendricSystemRepository
  */
  //protected CalendarRepository getCalendarRepository(){
  //    return cr;
  //}

  /**
  * Returns a handle to <code>PropertyRepository</code> object, which provides all
  * <code>Property</code> related services of this system. 
  *
  * @return a handle to <code>PropertyRepository</code> object
  * @see tauzaman.property.PropertyRepository
  */
  protected PropertyRepository getPropertyRepository(){
      return pr;
  }

  /**
  * Returns a handle to <code>FVSupportRepository</code> object, which provides all
  * <code>FVSupport</code> related services of this system.
  *
  * @return a handle to <code>FVSupportRepository</code> object
  * @see tauzaman.field.fvsupport.FVSupport
  */ 
  protected FVSupportRepository getFVSupportRepository(){
      return fvsr;
  }

}
