package tauzaman;

import java.rmi.*;
import tauzaman.timestamp.*;
import tauzaman.io.*;
import tauzaman.calendricsystem.*;

/**
* <p> <code>TauZamanService</code> class is a wrapper class for <code>TauZamanLocalService</code>
* and <code>TauZamanRemoteService</code>. It has two rationales; </p><p>
* First, it encapsulates active <code>TauZamanService</code>, which can be either
* <code>TauZamanLocalService</code> or <code>TauZamanRemoteService</code>. Second,
* it caches <code>TauZamanService</code>, which is used during creation of a temporal
* data type, such as Instant, Interval, Period or more fundamentally during 
* creation of a Granule. </p><p>
* There are two kinds of <code>TauZamanService</code>s; <code>TauZamanLocalService</code>
* and <code>TauZamanRemoteService</code>. The only relation between them can be explained 
* as; <code>TauZamanRemoteService</code> <b>"has a"</b> <code>TauZamanLocalService</code>. 
* Hence, this class keeps two kinds of services seperately in a mutually exclusive manner. 
* That's to say, if one service (remote/local) is null, the other should be valid.
*
* @see tauzaman.TauZamanLocalService
* @see tauzaman.TauZamanRemoteService
* @see tauzaman.timestamp.Granule
* @see tauzaman.temporaldatatypes.Instant
* @see tauzaman.temporaldatatypes.Interval
* @see tauzaman.temporaldatatypes.Period
*/

public class TauZamanService{

  /** A handle to <code>TauZamanLocalService</code> **/
  private TauZamanLocalService tzls = null;

  /** A handle to <code>TauZamanRemoteService</code> **/
  private TauZamanRemoteService tzrs  = null;

  /** Cached active <code>CalendricSystem</code> name of valid <code>TauZamanService</code> **/
  private String calendricSystemName = null;

  /**
  * Constructs a <code>TauZamanService</code> from given 
  * <code>TauZamanRemoteService</code>.
  * 
  * @param tzrs <code>TauZamanRemoteService</code> that this <code>TauZamanService</code>
  * will be initialized
  * @throws TauZamanException if this service was not set to a valid <code>TauZamanRemoteService</code>
  */
  public TauZamanService(TauZamanRemoteService tzrs) throws TauZamanException {
      if(tzrs == null)
          throw new TauZamanException("Exception when parsing input: TauZamanService was not set to a valid TauZamanRemote service");
      this.tzrs = tzrs;
      tzls = null;
  }

  /**
  * Constructs a <code>TauZamanService</code> from given 
  * <code>TauZamanLocalService</code>.
  * 
  * @param tzls <code>TauZamanLocalService</code> that this <code>TauZamanService</code>
  * will be initialized
  * @throws TauZamanException if this service was not set to a valid <code>TauZamanLocalService</code>
  */
  public TauZamanService(TauZamanLocalService tzls) throws TauZamanException{
      if(tzls == null)
          throw new TauZamanException("Exception when parsing input: TauZamanService was not set to a valid TauZamanLocal service");
      this.tzls = tzls;
      tzrs = null;
  }

  /**
  * Caches <code>CalendricSystem</code> name of valid TauZaman service
  * that this <code>TauZamanService</code> keeps.
  * This method will be called when a temporal data type
  * needs to cache <code>CalendricSystem</code> name along
  * with active <code>TauZamanService</code>.
  *
  * @throws RemoteException if any abnormal condition occurs when returning 
  * active <code>CalendricSystem</code> name. The only source of exceptions here is
  * Remote connections, if underlying service is Remote.
  */
  public void cacheCalendricSystemName() throws RemoteException{
      calendricSystemName = getActiveCalendricSystemName();
  }

  /**  
  * Returns name of active <code>CalendricSystem</code> of this service.
  *
  * @return String name of active <code>CalendricSystem</code> of this service
  *
  * @throws RemoteException if any abnormal condition occurs when returning 
  * active <code>CalendricSystem</code> name. The only source of exceptions here is
  * Remote connections, if underlying service is Remote.
  */
  public String getActiveCalendricSystemName() throws RemoteException{
 
      String activeCalendricSystemName = null;

      // determine the service type
      if(tzls != null)
          activeCalendricSystemName = tzls.getActiveCalendricSystemName();
      else{ // (tzrs != null)
          activeCalendricSystemName = tzrs.getActiveCalendricSystemName();
      }

      return activeCalendricSystemName;

  }

  /**
  * Returns default <code>Granularity</code> of active <code>CalendricSystem</code> of this 
  * <code>TauZamanService</code>.
  *
  * @return default <code>Granularity</code> of active <code>CalendricSystem</code> of this  
  * <code>TauZamanService</code>
  *
  * @throws RemoteException if any abnormal condition occurs when returning 
  * active <code>CalendricSystem</code>'s default <code>Granularity</code>. 
  * The only source of exceptions here is Remote connections, if underlying service is Remote.
  */
  public Granularity getActiveCalendricSystemDefaultGranularity() throws RemoteException{

      Granularity defaultGranularity = null;

      if(tzls != null)
          defaultGranularity = tzls.getActiveCalendricSystemDefaultGranularity();
      else{ // if(tzrs != null)
          defaultGranularity = tzrs.getActiveCalendricSystemDefaultGranularity();
      }
 
      return defaultGranularity;
  } 



  /**
  * Returns an array of <code>Granule</code> objects formed by using input temporal constant and
  * given <code>Property</code> name.
  * 
  * @param input String that represents input temporal constant
  * @param propertyName name of the <code>Property</code> to be used when parsing input
  * 
  * @return array of <code>Granule</code> objects
  * 
  * @throws RemoteException if any abnormal condition occurs in the connection if
  * this service is remote. 
  * @throws IOException if any abnormal condition occurs while parsing input
  * temporal constant
  */
  public Granule [] parseInput(String input, String propertyName) throws RemoteException, IOException{

      Granule granules [] = null;

      // determine service type
      if(tzls != null){
          // local service
          granules = tzls.parseInput(input, propertyName);
      }
      else{ // if(tzrs != null)
          // remote service
          granules = tzrs.parseInput(input, propertyName);
      }

      return granules;
  }

  /**
  * Converts a timestamp (in terms of <code>Granule</code>(s)) into temporal
  * constant as string. Format of the output string, given <code>Property</code>'s
  * format template is used.
  * 
  * @param granules <code>Calendar</code> parsed timestamp
  * @param propertyName name of the <code>Property</code>, which is used to
  * format of the output string, temporal constant
  * 
  * @return formatted String, which is a temporal constant
  * 
  * @throws RemoteException if any abnormal condition occurs in the connection if
  * this service is remote. 
  * @throws IOException if any abnormal condition occurs while parsing input
  * temporal constant
  */
  public String parseOutput(Granule [] granules, String propertyName) throws RemoteException, IOException{

      String output = null;

      /* if calendricSystemName is null, that means a user formed a TauZamanService without caching
         its formation time active calendric system name. In this case, for outputting, we'll 
         use currently active calendric system name, instead of throwing an exception. */

      if(calendricSystemName == null)
          cacheCalendricSystemName();
      
      // determine service type
      if(tzls != null){
          // local service
          output = tzls.parseOutput(granules, propertyName, calendricSystemName);
      }
      else{ //if(tzrs != null)
          // remote service
          output = tzrs.parseOutput(granules, propertyName, calendricSystemName);
      }

      return output;
  }

  /* there will be methods for scale and cast here... */

}
