package tauzaman.timestamp;

import java.lang.*;
import java.math.*;
import tauzaman.timestamp.*;

/**
* <code>TimeValue</code> class implements the arithmetic and 
* comparison operations for an underlying domain of times 
* times that are bounded by "MIN" and "MAX" values.
* MIN (MAX) are before (after) the smallest (largest) times.  
* Operations on TimeValues usually construct a new TimeValue, they
* never <em>mutate</em> a TimeValue.
*
* For example, assuming a discrete domain of times implemented
* using BigIntegers.
* <pre>
*    TimeValue min = TimeValue.BEGINNING_OF_TIME;
*    TimeValue max = TimeValue.END_OF_TIME;
*    TimeValue one = new TimeValue(new BigInteger("1"));
*    TimeValue three = new TimeValue(new BigInteger("3"));
*    TimeValue three = new TimeValue(TimeValue.NORMAL, threeGranule);
*    TimeValue result = max.add(min);
*    System.out.println(max.image());  // Prints [TimeValue MAX]
*    System.out.println(min.image()); // Prints [TimeValue MIN]
*    System.out.println(result.image()); // prints [TimeValue NORMAL 0]
*    result = one.add(three); 
*    System.out.println(one.image()); // Prints [TimeValue NORMAL 1]
*    System.out.println(three.image()); // Prints [TimeValue NORMAL 3]
*    System.out.println(result.image()); //Prints [TimeValue NORMAL 4]
* </pre>
*
* @author  Curtis Dyreson
* @version 0.9, Jan/31/2003
* @see     tauzaman.timestamp.Granule
* @status  NOT IMPLEMENTED
**/
public class TimeValue implements Comparable {

  // MAX indicates a maximum time value, larger than any other time value (save another MAX)
  public static final int MAX = 0;

  // MIN indicates a minimum time value, smaller than any other time value (save another MIN)
  public static final int MIN = 1;

  // NORMAL indicates a non-special time value, could be determinate or indeterminate
  public static final int NORMAL = 2;

  // Largest possible time value 
  public static final TimeValue END_OF_TIME = new TimeValue(MAX);
  public static final TimeValue ALL_OF_TIME = new TimeValue(MAX);

  // Smallest possible time value 
  public static final TimeValue BEGINNING_OF_TIME = new TimeValue(MIN);
  public static final TimeValue NEGATIVE_ALL_OF_TIME = new TimeValue(MAX);

  public static final TimeValue ZERO_TIMEVALUE = 
    new TimeValue(NORMAL, new BigInteger("0"));
  public static final TimeValue ONE_TIMEVALUE = 
    new TimeValue(NORMAL, new BigInteger("1"));
  public static final TimeValue MINUS_ONE_TIMEVALUE = 
    new TimeValue(NORMAL, new BigInteger("-1"));

  // Kind of the TimeValue ranges from MIN to NORMAL
  private int kind;

  // Actual value of the TimeValue
  private BigInteger value;

  /**
  * Construct a NORMAL time value.
  * The big integer represents the displacement from the origin.
  * @param kind - must be NORMAL
  * @param p - granule count
  **/ 
  public TimeValue(int kind, BigInteger count) {
    if (kind == NORMAL) {
      this.kind = kind;
      this.value = count;
      }
    else {
      // Bad kind passed
      Internal.Error("TimeValue Constructor, Can't use MAX or MIN");
      }
    }

  /**
  * Construct a NORMAL time value.
  * The integer represents the displacement from the origin.
  * @param kind - must be NORMAL
  * @param p - granule count
  **/ 
  public TimeValue(int kind, int count) {
    if (kind == NORMAL) {
      this.kind = kind;
      this.value = new BigInteger(Integer.toString(count));
      }
    else {
      // Bad kind passed
      Internal.Error("TimeValue Constructor, Can't use MAX or MIN");
      }
    }

  /**
  * A special TimeValue only needs a kind.  
  * If the kind is NORMAL, then an exception will
  * be thrown.
  * @param kind - MAX, MIN 
  * @throws BadTimeValue
  **/ 
  public TimeValue(int kind) {
    if (kind == NORMAL) Internal.Error("TimeValue Constructor, Can't use NORMAL");
    this.kind = kind;
    this.value = new BigInteger("0");
    }

  /**
  * Create a nice string image of a time value
  * @return String image of time value
  **/
  public String image() {
    switch (kind) {
      case MAX: return "[TimeValue MAX]";
      case MIN: return "[TimeValue MIN]";
      case NORMAL: return "[TimeValue NORMAL " + value.toString() + "]";
      default: Internal.Error("unknown TimeValue in TimeValue image");
      }
    return ""; //dummy return
    }

  /**
  * Accessor function to get the kind represented by this TimeValue.
  * @return the kind
  **/ 
  public int kind() {
    return kind;
    }

  /**
  * Accessor function to get the value represented by this TimeValue.
  * @return the value
  **/ 
  public BigInteger value() {
    return value;
    }

  /**
  * Get the hashCode for this TimeValue.
  * @return the hashCode
  **/ 
  public int hashCode(TimeValue other) {
    switch (kind) {
      case MAX: return MAX;
      case MIN: return MIN;
      case NORMAL: return value.hashCode();
      }
    Internal.Error("unknown TimeValue in TimeValue equals");
    return 0; //dummy return since compiler is brain-dead
    }

  /**
  * Compare this TimeValue with another.   Implements the Comparable interface.
  *   if this < other then returns negative number
  *   if this == other then returns 0
  *   if this > other then returns positive number
  * @param other - The TimeValue to compare
  * @return integer representing result of comparison
  * @see    java.lang.Comparable
  * @status NOT TESTED
  **/ 
  public int compareTo(Object otherTimeValue) {
    TimeValue other = (TimeValue)otherTimeValue;
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return 0;
        case MIN: return 1;
        case NORMAL: return 1;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return -1;
        case MIN: return 0;
        case NORMAL: return -1;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return -1;
        case MIN: return 1;
        case NORMAL: return value.compareTo(other.value());
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue equalTo");
    return 0; //dummy return since compiler is brain-dead
    }

  /**
  * Construct the result of "this minus other".
  * @param other - The TimeValue to subtract
  * @return this - other
  **/ 
  public TimeValue subtract(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return ZERO_TIMEVALUE;
        case MIN: return ALL_OF_TIME;
        case NORMAL: return ALL_OF_TIME;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return NEGATIVE_ALL_OF_TIME;
        case MIN: return ZERO_TIMEVALUE;
        case NORMAL: return NEGATIVE_ALL_OF_TIME;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return NEGATIVE_ALL_OF_TIME;
        case MIN: return ALL_OF_TIME;
        case NORMAL: 
          return new TimeValue(NORMAL,value.subtract(other.value()));
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue subtract");
    return null;
    }

  /**
  * Construct the result of "this plus other".
  * @param other - The TimeValue to add
  * @return this + other
  **/ 
  public TimeValue add(TimeValue other) {
    switch (kind) {
      case MAX: switch (other.kind()) {
        case MAX: return ALL_OF_TIME;
        case MIN: return ZERO_TIMEVALUE;
        case NORMAL: return ALL_OF_TIME;
        } break;
      case MIN: switch (other.kind()) {
        case MAX: return ZERO_TIMEVALUE;
        case MIN: return NEGATIVE_ALL_OF_TIME;
        case NORMAL: return NEGATIVE_ALL_OF_TIME;
        } break;
      case NORMAL: switch (other.kind()) {
        case MAX: return ALL_OF_TIME;
        case MIN: return NEGATIVE_ALL_OF_TIME;
        case NORMAL: 
          return new TimeValue(NORMAL,value.add(other.value()));
        } 
      }
    Internal.Error("unknown TimeValue in TimeValue add");
    return null;
    }

  /**
  * Construct the result of "minus this".
  * @return this * -1
  **/ 
  public TimeValue negate() {
    switch (kind) {
      case MAX: return NEGATIVE_ALL_OF_TIME;
      case MIN: return ALL_OF_TIME;
      case NORMAL: 
        return new TimeValue(NORMAL, value.multiply(MINUS_ONE_TIMEVALUE.value()));
      }
    Internal.Error("unknown TimeValue in TimeValue negate");
    return null;
    }

  /**
  * Construct the result of "this times N".
  * @param other - The integer to multiply
  * @return this * N
  **/ 
  public TimeValue multiply(int n) {
    switch (kind) {
      case MAX: return ALL_OF_TIME;
      case MIN: return NEGATIVE_ALL_OF_TIME;
      case NORMAL:
        return new TimeValue(NORMAL,value.multiply(new BigInteger(Integer.toString(n))));
      }
    Internal.Error("unknown TimeValue in TimeValue multiply");
    return null;
    }

  /**
  * Construct the result of "this div N".
  * @param other - The integer to divide by
  * @return this/N
  **/ 
  public TimeValue divide(int n) {
    switch (kind) {
      case MAX: return ZERO_TIMEVALUE;
      case MIN: return ZERO_TIMEVALUE;
      case NORMAL: 
        return new TimeValue(NORMAL,value.divide(new BigInteger(Integer.toString(n))));
      }
    Internal.Error("unknown TimeValue in TimeValue divide");
    return null;
    }

//End of class
}
