package tauzaman.timestamp;

import tauzaman.timestamp.domain.*;
import tauzaman.temporaldatatypes.*;

/**
* <code>LeftOperandSemantics</code> implements a left operand semantics whereby
* the right operand is cast to the granularity of the left operand
* prior to performing the operation.
* Below is an example of the semantics.
*  <pre>
*     LeftOperandSemantics ops = new LeftOperandSemantics();
*     Instant i = new Instant("July 3, 2002");
*     Interval j = new Interval("1 month");
*     Instant k = ops.add(i,j); // k is in the granularity of "days"
*     Instant m = ops.add(j,i); // m is in the granularity of "months"
*  </pre>
**/
public class LeftOperandSemantics implements Semantics {

  /** 
  * Does one Anchorable precede another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return true or false 
  **/
  public ExtendedBoolean precedes(Anchorable alpha, Anchorable beta) {
    TimeValue endAlpha = alpha.latest();
    TimeValue startBeta = beta.earliest();
    TimeValue gamma = startBeta.cast(endAlpha.granularity());
    return endAlpha.lessThanOrEqualTo(gamma);
    }

  /** 
  * Does one Anchorable overlap another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return true or false 
  **/
  public ExtendedBoolean overlaps(Anchorable alpha, Anchorable beta) {
    // grab the latest of the earliest instants
    Instant startAlpha = alpha.earliest();
    Instant startBeta = beta.earliest();
    Instant gamma = startBeta.cast(startAlpha.granularity());
    Instant last = startAlpha.last(gamma);

    // grab the earliest of the pair of latest instants
    Instant endBeta = beta.latest();
    Instant endAlpha = alpha.latest();
    gamma = endBeta.cast(endAlpha.granularity());
    Instant first = endAlpha.first(gamma);

    // is the last after the first?
    // Do not assume both are at the same granularity.
    gamma = last.cast(first.granularity());
    return first.lessThanOrEqualTo(gamma);
    }

  /** 
  * Does one Anchorable meet another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean meets(Anchorable alpha, Anchorable beta) {
    // does the end of alpha meet the start of beta?
    Instant endAlpha = alpha.latest();
    Instant startBeta = beta.earliest();
    Instant gamma = startBeta.cast(endAlpha.granularity());
    ExtendedBoolean cond1 = endAlpha.equalTo(startBeta);
    if (cond1.satisfied()) return cond1; // early exit test

    // does the end of beta meet the start of alpha?
    Instant endBeta = beta.earliest();
    Instant startAlpha = alpha.earliest();
    gamma = endBeta.cast(startAlpha.granularity());
    return startAlpha.equalTo(gamma);
    }

  /** 
  * Does one Anchorable contain another?
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean contains(Anchorable alpha, Anchorable beta) {
    // does the end of alpha come after the end of beta?
    Instant endAlpha = alpha.latest();
    Instant endBeta = beta.latest();
    Instant gamma = endBeta.cast(endAlpha.granularity());
    ExtendedBoolean cond1 = endAlpha.greaterThan(endBeta);
    if (cond1.unsatisfied()) return cond1;
 
    // does the start of beta come after the start of alpha?
    Instant startBeta = beta.earliest();
    Instant startAlpha = alpha.earliest();
    gamma = startBeta.cast(startAlpha.granularity());
    return startAlpha.lessThanOrEqualTo(gamma);
    }

  /** 
  * Is one Anchorable equalTo another?  Currently this does not 
  * distinguish the "kinds" of Anchorables, although we could
  * change this as needed to just do each kind.
  * @param alpha - Anchorable to compare
  * @param beta - Anchorable to compare
  * @return ExtendedBoolean
  **/
  public ExtendedBoolean equalTo(Anchorable alpha, Anchorable beta) {
    // Must match each instant in both structures.
    Instant[] arrayBeta = beta.asInstantArray();
    Instant[] arrayAlpha = alpha.asInstantArray();
    if (arrayBeta.length != arrayAlpha.length) 
      return new ExtendedBoolean(false);
    for (int i = 0; i < arrayAlpha.length; i++) {
      boolean found = false;
      for (int j = 0; j < arrayBeta.length && !found; j++) {
        Instant gamma = arrayBeta[j].cast(arrayAlpha[i].granularity());
        // try the next instant, if we haven't already found it
        if (arrayAlpha[j] != null) {
          ExtendedBoolean cond = gamma.equalTo(arrayAlpha[i]);
          found = cond.satisfied();
          if (found) arrayAlpha[j] = null;
          }
        if (!found) return new ExtendedBoolean(false);
        }
      }
      return new ExtendedBoolean(true);
    }

  /** 
  * Displace (add) an Anchorable with an Unanchorable.  
  * @param alpha - displacee
  * @param beta - displacor
  * @return - (alpha + cast(beta,alpha.granularity))
  **/
  public Anchorable add(Anchorable alpha, Unanchorable beta) {
    // Only Instants and Intervals know how to add themselves
    // so must break this down in to the constituent parts and
    // do the addition.
    Instant[] a = alpha.asInstantArray();
    Interval[] b = beta.asIntervalArray();
    Instant[] result = new Instant[a.length * b.length];

    int index = 0;
    for (int i = 0; i < a.length; i++) {
      for (int j = 0; j < b.length; j++) {
        result[index++] = a[i].plus(b[j].cast(a[i].granularity()));
        }
      }
    return alpha.newInstance(result);
    }

  /** 
  * Displace (add) an Unanchorable with an Unanchorable.  
  * @param alpha - displacee
  * @param beta - displacor
  * @return - (alpha + cast(beta,alpha.granularity))
  **/
  public Unanchorable add(Unanchorable alpha, Unanchorable beta) {
    // Only Intervals know how to add themselves
    // so must break this down in to the constituent parts and
    // do the addition.
    Interval[] a = alpha.intervalArray();
    Interval[] b = beta.intervalArray();
    Interval[] result = new Interval[a.length * b.length];

    int index = 0;
    for (int i = 0; i < a.length; i++) {
      for (int j = 0; j < b.length; j++) {
        result[index++] = a[i].plus(b[j].cast(a[i].granularity()));
        }
      }
    return alpha.newInstance(result);
    }

  /** 
  * Displace (add) an Unanchorable with an Anchorable.  
  * @param alpha - displacor
  * @param beta - displacee
  * @return - (alpha + cast(beta,alpha.granularity))
  **/
  public Anchorable add(Unanchorable alpha, Anchorable beta) {
    // Only Instants and Intervals know how to add themselves
    // so must break this down in to the constituent parts and
    // do the addition.
    Instant[] a = beta.asInstantArray();
    Interval[] b = alpha.asIntervalArray();
    Instant[] result = new Instant[a.length * b.length];
 
    int index = 0;
    for (int i = 0; i < a.length; i++) {
      for (int j = 0; j < b.length; j++) {
        Instant g = a[i].cast(b[j].granularity());
        result[index++] = g.add(b[j]);
        }
      }
    return beta.newInstance(result);
    }

  /** 
  * Displace (subtract) an Anchorable with an Unanchorable.  
  * @param alpha - displacee
  * @param beta - displacor
  * @return - (alpha - cast(beta,alpha.granularity))
  **/
  public Anchorable subtract(Anchorable alpha, Unanchorable beta) {
    // Only Instants and Intervals know how to add themselves
    // so must break this down in to the constituent parts and
    // do the addition.
    Instant[] a = alpha.asInstantArray();
    Interval[] b = beta.asIntervalArray();
    Instant[] result = new Instant[a.length * b.length];
 
    int index = 0;
    for (int i = 0; i < a.length; i++) {
      for (int j = 0; j < b.length; j++) {
        result[index++] = a[i].subtract(b[j].cast(a[i].granularity()));
        }
      }
    return alpha.newInstance(result);
    }

  /** 
  * Displace (subtract) an Unanchorable with an Anchorable.  
  * @param alpha - displacor
  * @param beta - displacee
  * @return - (-alpha + cast(beta,alpha.granularity))
  **/
  public Anchorable subtract(Unanchorable alpha, Anchorable beta) {
    // Only Instants and Intervals know how to add themselves
    // so must break this down in to the constituent parts and
    // do the addition.
    Instant[] a = beta.asInstantArray();
    Interval[] b = alpha.asIntervalArray();
    Instant[] result = new Instant[a.length * b.length];
 
    int index = 0;
    for (int i = 0; i < a.length; i++) {
      for (int j = 0; j < b.length; j++) {
        Instant g = a[i].cast(b[j].granularity());
        result[index++] = g.subtract(b[j]);
        }
      }
    return beta.newInstance(result);
    }

  /** 
  * Negate an Unanchorable. Note this will mutate 
  * the result parameter.
  * @param alpha - Unanchorable to negate
  * @return - -alpha
  **/
  public Unanchorable negate(Unanchorable alpha) {
    // Only Intervals know how to negate themselves
    // so must break this down in to the constituent Intervals and
    // do the negation.
    Interval[] a = alpha.asIntervalArray();
    Interval[] result = new Interval[a.length];
 
    for (int i = 0; i < a.length; i++) {
      result[i] = a[i].negate();
      }
    return alpha.newInstance(result);
    }

// Removed by Curt, Jan-28-03, removed multiplication
//  /** 
//  * Multiply an Anchorable by a TimeValue.  
//  * @param alpha - Anchorable to multiply
//  * @param n - how many 
//  * @return result - (alpha * n)
//  **/
//  public Unanchorable multiply(Unanchorable alpha, TimeValue n) {
//    // Only Intervals know how to multiply themselves
//    // so must break this down in to the constituent Intervals and
//    // do the multiplication.
//    Interval[] a = alpha.asIntervalArray();
//    Interval[] result = new Interval[a.length];
// 
//    for (int i = 0; i < a.length; i++) {
//      result[i] = a[i].multiply(n);
//      }
//    return alpha.newInstance(result);
//    }
//
//  /** 
//  * Divide an Unanchorable by an Unanchorable. 
//  * @param alpha - divisee
//  * @param beta - divisor
//  * @param result - (TimeValue) alpha/beta
//  **/
//  public TimeValue[] divide(Unanchorable alpha, Unanchorable beta) {
//    // Only Intervals know how to divide themselves
//    // so must break this down in to the constituent parts and
//    // do the division.
//    Interval[] a = beta.asIntervalArray();
//    Interval[] b = alpha.asIntervalArray();
//    TimeValue[] result = new TimeValue[a.length * b.length];
// 
//    int index = 0;
//    for (int i = 0; i < a.length; i++) {
//      for (int j = 0; j < b.length; j++) {
//        result[index++] = a[i].divide(b[j].cast(a[i].granularity()));
//        }
//      }
//    return result;
//    }
//
//  /** 
//  * Divide an Unanchorable by a TimeValue. 
//  * @param alpha - divisee
//  * @param n - divisor
//  * @param result - alpha/n
//  **/
//  public Unanchorable divide(Unanchorable alpha, TimeValue n) {
//    // Only Intervals know how to divide themselves
//    // so must break this down in to the constituent Intervals and
//    // do the division.
//    Interval[] a = alpha.asIntervalArray();
//    Interval[] result = new Interval[a.length];
// 
//    for (int i = 0; i < a.length; i++) {
//      result[i] = a[i].divide(n);
//      }
//    return alpha.newInstance(result);
//    }

  ///** 
  //* Grab the first Anchorable in an Anchorable, and make a new Anchorable 
  //* with it.
  //* Note, this is not a mutator, it will create a new Anchorable.
  //* @param alpha - Anchorable to retrieve from
  //* @return first Anchorable in alpha
  //**/
  //public Anchorable first(Anchorable alpha) {
  //  return alpha.earliest(beta.cast(alpha.granularity()));
  //  }

  ///** 
  //* Grab the last Anchor in an Anchorable, and make a new Anchorable with it.
  //* Note, this is not a mutator, it will create a new Anchorable.
  //* @param alpha - Anchorable to retrieve from
  //* @return last Anchorable in alpha
  //**/
  //public Anchorable last(Anchorable alpha) {
  //  return alpha.latest(beta.cast(alpha.granularity()));
  //  }
}
