package tauzaman.temporaldatatypes;

import java.util.*;
import java.lang.*;
import java.math.*;
import java.rmi.*;

import tauzaman.timestamp.*;
import tauzaman.calendricsystem.*;
import tauzaman.*;
import tauzaman.io.*;

/**
* The <code>Interval</code> class provides a set of constructor and accessor
* operations for intervals.
* (Arithmetic and comparison operations are implemented in a @see Semantics.)
* An interval is a length of time, unanchonred to any specific
* point or region on the time-line.  For example, the interval "1 day" 
* represents a length.  It can be used, for example, to displace values
* anchored to the time-line, e.g., "July 3, 2002", by a distance of "1 day",
* moving the instant to "July 4, 2002".
* This class implements intervals that have a known, fixed length as well as
* intervals that are indeterminately known.
*
* To perform comparisons and arithmetic operations on Intervals, 
* it is first necessary to create a @see Semantics interface.
* A Semantics implements common comparison and arithmetic operations on 
* temporal entities.  For example a user would write code like the following
*  <pre>
*     // Left operand semantics casts operands in binary operations 
*     // to the granularity of the left operand, and then performs
*     // the desired operation. 
*     Ops = new LeftOperandSemantics();
*     // Create an Instant and an Interval.
*     // In this example, both are at the granularity of days.
*     Instant i = new Instant("July 3, 2002");
*     Interval j = new Interval("1 day");
*     // Displace i by length j, k will represent the day "July 4, 2002" 
*     Instant k = Ops.add(i,j);
*  </pre>
* We employ this strategy so that interval operations can be
* performed under varying Semantics.
*
* @author  Curtis Dyreson 
* @version 0.9, Jan/31/2003
* @see     tauzaman.temporaldatatypes.DeterminateSemantics
* @see     tauzaman.temporaldatatypes.IndeterminateSemantics
* @status  design complete, NOT IMPLEMENTED
**/
public class Interval implements Cloneable {
  // The ALL_OF_TIME Interval constant represents the distance of  
  // the entire time-line.  It is a granularity-less constant.
  //public static final Interval ALL_OF_TIME = 
  //    new Interval(new Granule(TimeValue.ALL_OF_TIME));

  // The NEGATIVE_ALL_OF_TIME Interval constant represents the negated
  // distance of the entire time-line.  It is a granularity-less constant.
  //public static final Interval NEGATIVE_ALL_OF_TIME = 
  //    new Interval(new Granule(TimeValue.NEGATIVE_ALL_OF_TIME));

  // Intervals have a single Granule
  private Granule granule;

  /** <code>TauZamanService</code> that this <code>Instant</code> is created under **/
  TauZamanService cachedTauZamanService = null;


  /**
  * Construct an Interval from a count of granules.  <em>Convert</em> 
  * the resulting
  * interval to the specified granularity using the default conversion
  * semantics.
  * @param g - granularity
  * @param granule - length in granules
  * @status NOT IMPLMENTED
  **/
  public Interval(Granularity g, Granule granule) {
    this.granule = granule;
    }

  /**
  * Construct an Interval from a count of granules. 
  * @param granule - length in granules
  **/
  public Interval(Granule granule) {
    this.granule = granule;
    }
 
  /**
  * Construct a <code>Interval</code> given a String literal.
  *
  * @param literal String representing temporal constant
  *
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs during parsing of    
  * string temporal constant to <code>Instant</code> timestamp
  */
  public Interval(String literal) throws TemporalDataTypeFormationException{

      Granule [] granules = null;
  
      // get active TauZamanService
      TauZamanService activeTauZamanService = TauZamanSystem.getActiveService();
     
      // throws TauZamanException
      try{
          granules = activeTauZamanService.parseInput(literal, "IntervalInputFormat");
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Interval data type from: " + literal, re);
      }
      catch(IOException ioe){
          throw new TemporalDataTypeFormationException("Exception when forming Interval data type from: " + literal, ioe);
      }
     
      // get the produced granule
      granule = granules[0];
  
      // cache the service
      cachedTauZamanService = activeTauZamanService;
  
      // cache the active CalendricSystem name
      try{ 
         cachedTauZamanService.cacheCalendricSystemName();
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Interval data type from: " + literal, re);
      }
      
  }

  /**  
  * Construct an Interval as the distance between a period's endpoints
  * @param period - the input period
  * @status NOT IMPLEMENTED
  **/
  public Interval (Period p) {
   this.granule = null;  // Can we do this one?
   }


  /**
  * Returns string representation of this <code>Interval</code>.
  *  
  * @return String temporal constant that corresponds to this <code>Interval</code>
  *   
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs when forming
  * string temporal constant from timestamp
  */
  public String output() throws TemporalDataTypeFormationException{
  
       Granule granules [] = new Granule[1];
       granules[0] = granule;
  
      String output = null; 
      
      try{
          output = cachedTauZamanService.parseOutput(granules, "IntervalOutputFormat");
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when outputting Interval data type", re);
      }
      catch(IOException ioe){
          throw new TemporalDataTypeFormationException("Exception when outputting Interval data type", ioe);
      }

      return output;
  }  

  /** 
  * Build a nice string image of an Interval, for debugging mostly.
  * Provides an alternative to toString().
  * @return String image of Interval
  **/
  public String image() {
    return "[Interval " + granule.image() + "]";
    }
 
  /**
  * Cast this Interval to the indicated granularity.  It will create a 
  * new Interval.
  * @param g - Granularity
  * @status NOT IMPLEMENTED
  **/
  public Interval cast(Granularity g) {
    return this;
    }
 
  /**
  * Cast this Interval to the default granularity.  It will create a 
  * new Interval.
  * @status NOT IMPLEMENTED
  **/
  public Interval cast() {
    return this;
    }
 
  /**
  * Scale this Interval to the indicated granularity.  It will
  * create a new Interval.
  * @param g - Granularity
  * @status NOT IMPLEMENTED
  **/
  public Interval scale(Granularity g) {
    return this;
    }

  /**
  * Scale this Interval to the default granularity.  It will create a new
  * Interval.
  * @status NOT IMPLEMENTED
  **/
  public Interval scale() {
    return this;
    }
 
  /** 
  * Get the smallest (only!) Granule from this Interval.
  * @return the Interval's granule
  * @status Is this used?
  **/
  public Granule smallest() {
    return this.granule;
    }

  /** 
  * Get the largest (only!) Granule in this Interval.
  * @return the Interval's granule
  * @status Is this used?
  **/
  public Granule largest() {
    return this.granule;
    }

  /** 
  * Accessor - retrieve the Granule from this Interval.
  * @return the Interval's granule
  **/
  public Granule granule() {
    return this.granule;
    }

  /**
  * Retrieve the Granularity from this Interval
  * @return Granularity
  **/
  public Granularity getGranularity() {
    return granule.getGranularity();
    }

}
