package tauzaman.temporaldatatypes;

import java.util.*;
import java.math.*;
import java.rmi.*;

import tauzaman.timestamp.*;
import tauzaman.calendricsystem.*;
import tauzaman.*;
import tauzaman.io.*;

/**
* The <code>Instant</code> class provides a set of constructor and accessor
* operations on Instants.
* (Arithmetic and comparison operations are implemented in a Semantics.)
* An instant is a
* point on the time-line, e.g., 'July 3, 2002'.
* An instant could also be indeterminately known, 
* e.g., "July 3, 2002 ~ July 4, 2002".
*
* To perform comparisons and arithmetic operations on Instants, 
* it is first necessary to create a Semantics interface.
* A Semantics implements common comparison and arithmetic operations on 
* temporal entities.  For example a user would write code like the following
*  <pre>
*     // Left operand semantics casts operands in binary operations 
*     // to the granularity of the left operand, and then performs
*     // the desired operation. 
*     Ops = new LeftOperandSemantics();
*     // Create two Instants, in this case, both are at the granularity of days
*     Instant i = new Instant("July 3, 2002");
*     Instant j = new Instant("July 4, 2002");
*     // Is i earlier on the time-line than j?
*     if (Ops.precedes(i,j)) ...
*  </pre>
* We employ this strategy so that all the instant operations can be
* performed under varying Semantics.
*
* @author  Curtis Dyreson 
* @version 0.9, Jan/31/2003
* @see     tauzaman.temporaldatatypes.DeterminateSemantics
* @see     tauzaman.temporaldatatypes.IndeterminateSemantics
* @status  design complete, NOT IMPLEMENTED
**/
public class Instant implements Cloneable {
  // The BEGINNING_OF_TIME Instant constant represents a value that
  // is the start of the time-line.  
  // It is a granularity-less constant.
  public static final Instant BEGINNING_OF_TIME = 
      new Instant(new Granule(TimeValue.BEGINNING_OF_TIME));

  // The END_OF_TIME Instant constant represents a value that
  // is the end of the time-line.  
  // It is a granularity-less constant.
  public static final Instant END_OF_TIME =
      new Instant(new Granule(TimeValue.END_OF_TIME));

  // CURT - REMOVED since a Zeroth Instant should have a Granularity
  // The ZERO_TIME Instant constant represents a value that
  // is the zeroth granule in any granularity on the time-line.  
  // It is a granularity-less constant.
  public static final Instant ZERO_TIME =
      new Instant(new Granule(0));

  // Instants have a single Granule
  private Granule granule;

  /** <code>TauZamanService</code> that this <code>Instant</code> is created under **/
  TauZamanService cachedTauZamanService = null;

  /**
  * Construct an Instant from a granule representing 
  * the distance from the granularity anchor point
  * @param granule - distance in granules from the anchor
  * @status NOT IMPLEMENTED
  **/
  public Instant(Granule granule) {
    this.granule = granule;
    }

  /**
  * Construct an Instant from a granule representing 
  * the distance from the granularity anchor point
  * Convert the 
  * Instant to the desired granularity.
  * @param g - granularity
  * @param granule - distance in granules from the anchor
  * @status NOT IMPLEMENTED
  **/
  public Instant(Granularity g, Granule granule) {
    this.granule = granule;
    }

  /**
  * Construct an Instant by parsing a string, using the default granularity.
  * @param literal - a string representation of the instant
  * 
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs during parsing of
  * string temporal constant to <code>Instant</code> timestamp
  **/
  public Instant(String literal) throws TemporalDataTypeFormationException{

      Granule [] granules = null;

      // get active TauZamanService 
      TauZamanService activeTauZamanService = TauZamanSystem.getActiveService();

      /*
          "InstantInputFormat" will be tried, if there is no IOException
          "NowRelativeInstantInputFormat" will be tried, if there is no IOException
          "IndeterminateInstantInputFormat" will be tried, if there is IOException   
           it will be thrown.

           Note that any thrown RemoteException stops any further parsing.
      */

      try{ // RemoteException is a severe exception
        try{ // upon IOException other option will be tried
            granules = activeTauZamanService.parseInput(literal, "InstantInputFormat");
        }
        catch(IOException ioeDeterminate){
            try{
                granules = activeTauZamanService.parseInput(literal, "NowRelativeInstantInputFormat");
            }
            catch(IOException ioeNowRelative){
                try{
                    granules = activeTauZamanService.parseInput(literal, "IndeterminateInstantInputFormat");
                }
                catch(IOException ioeIndeterminate){
                    // last IOException is thrown
                    throw new TemporalDataTypeFormationException("Exception when forming Instant (Indeterminate) data type from: " + 
                                                                                         literal, ioeIndeterminate);
                }
            }
        }
        
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Instant data type from: " + literal, re);
      }

      // get the produced granule   
      granule = granules[0];

      // cache the service
      cachedTauZamanService = activeTauZamanService;

      // cache the active CalendricSystem name
      try{
          cachedTauZamanService.cacheCalendricSystemName();
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when forming Instant data type from: " + literal, re);
      }

    }


  /**
  * Construct an Instant by parsing a string.  The resulting Instant will
  * be <em>converted</em> to the desired Granularity using the default 
  * conversion semantics (cast or scale).
  * @param literal - a string representation of the instant
  * @param g - granularity
  * @status NOT IMPLMENTED
  **/
  public Instant(String literal, Granularity g) {
    }

  /**
  * Construct a zero Instant in the desired granularity.  The Granule in the
  * Instant defaults to the zeroth @see Granule.
  * @param g - Granularity
  * @status NOT IMPLEMENTED.  
  **/
  public Instant(Granularity g) {
    }

  /**
  * Construct a zero Instant in the default granularity.  The Granule in the
  * Instant defaults to the zeroth @see Granule.
  * @param g - Granularity
  * @status NOT IMPLEMENTED.  
  **/
  public Instant() {
    }

  /**
  * Returns string representation of this <code>Instant</code>. 
  * 
  * @return String temporal constant that corresponds to this <code>Instant</code>
  *
  * @throws TemporalDataTypeFormationException if any abnormal condition occurs when forming 
  * string temporal constant from timestamp
  */

  public String output() throws TemporalDataTypeFormationException{

      Granule granules [] = new Granule[1];
      granules[0] = granule;

      String format = null;
      if(granule.getKind() == Granule.DETERMINATE)
          format = "InstantOutputFormat";
      else if(granule.getKind() == Granule.NOWRELATIVE)
          format = "NowRelativeInstantOutputFormat";
      else  /* if(granule.getKind() == Granule.INDETERMINATE) */
          format = "IndeterminateInstantOutputFormat";

      String output = null;

      try{
          output = cachedTauZamanService.parseOutput(granules, format);
      }
      catch(RemoteException re){
          throw new TemporalDataTypeFormationException("Exception when outputting Instant data type", re);
      }
      catch(IOException ioe){
          throw new TemporalDataTypeFormationException("Exception when outputting Instant data type", ioe);
      }
      return output;
  }

  /** 
  * Build a nice string image of an Instant, for debugging mostly.  Provides
  * an alternative to toString().
  * @return String image of Instant
  **/
  public String image() {
    return "[Instant " + granule.image() + "]";
    }

  /** 
  * Generate the hash code value, needed for supporting Hashtables.
  * @return int 
  **/
//  public int hashCode() {
//    return granule.hashCode();
//    }

  /**
  * Returns the earliest (only!) Granule in the Instant.
  * @return Granule - the Granule for this Instant
  **/
  public Granule earliest() {
    return this.granule;
    }

  /**
  * Returns the latest (only!) Granule in the Instant.
  * @return Granule - the latest Granule
  **/
  public Granule latest() {
    return this.granule;
    }

  /**
  * Cast this Instant to the indicated granularity.
  * Note that it is not a mutator, it will create a new Instant.
  * @param g - desired Granularity
  * @status NOT IMPLEMENTED
  **/
  public Instant cast(Granularity g) {
    return this;
    }

  /**
  * Scale this Instant to the indicated granularity.
  * Note that it is not a mutator, it will create a new Instant.
  * @param g - desired Granularity
  * @status NOT IMPLEMENTED
  **/
  public Instant scale(Granularity g) {
    return this;
    }

  /**
  * Retrieve the Granularity from this Instant
  * @return Granularity object
  **/
  public Granularity getGranularity() {
    return granule.getGranularity();
    }

  /**
  * Retrieve the granule from this Instant
  * @return Granule
  **/
  public Granule getGranule() {
    return granule;
    }

}
