package tauzaman.temporaldatatypes;

import tauzaman.timestamp.*;
import tauzaman.temporaldatatypes.*;

/**
* The <code>IndeterminateSemantics</code> interface contains all the 
* arithmetic and comparison operations that are available on 
* indeterminate time values. 
* In an indeterminate semantics, granularity conversions can be casts 
* or scales, and ExtendedBoolean values are used for truth values. 
* This is an interface because there are lots of different semantics
* for even standard operations such as precedes, such as left-operand
* semantics where the operation is performed at the granularity of the left 
* operand.
* Below is an example of how one might use a semantics (assume that
* LeftOperandSemantics implements IndeterminateSemantics).
*  <pre>
*     LeftOperandSemantics ops = new LeftOperandSemantics();
*     Instant j = new Instant("July 3, 2003");
*     Interval i = new Interval("1 month");
*     Instant k = ops.add(i,j); // k is in the granularity of "days"
*     Instant m = ops.add(j,i); // k is in the granularity of "months"
*  </pre>
* Just a few operations are currently represented to give an idea of the
* eventual interface.  All the operations are on individual kinds of
* time values, e.g., Instant, Period, Interval.   We need to expand this
* with a complete set of temporal operators implementing Allen's algebra.
*
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.9, Jan/31/2003
* @see     tauzaman.temporaldatatypes.Instant
* @see     tauzaman.temporaldatatypes.Period
* @see     tauzaman.temporaldatatypes.Interval
**/
public interface IndeterminateSemantics {
  /** 
  * Does one Instant precede another, that is, is alpha < beta?
  * @param alpha - Instant to compare
  * @param beta - Instant to compare
  * @return alpha < beta
  **/
  public ExtendedBoolean precedes(Instant alpha, Instant beta);

  /** 
  * Does an Instant precede the start of a Period, 
  * that is, is alpha < beta.start?
  * @param alpha - Instant to compare
  * @param beta - Period to compare
  * @return alpha < beta.start
  **/
  public ExtendedBoolean precedes(Instant alpha, Period beta);

  /** 
  * Does the end of a Period precede the start of an Instant, 
  * that is, is alpha.end < beta?
  * @param alpha - Period to compare
  * @param beta - Instant to compare
  * @return alpha.end < beta
  **/
  public ExtendedBoolean precedes(Period alpha, Instant beta);

  /** 
  * Does the end of a Period precede the start of another,
  * that is, is alpha.end < beta.start?
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.end < beta.start
  **/
  public ExtendedBoolean precedes(Period alpha, Period beta);

  /** 
  * Does this Period begin at the same time as another, but end before it,
  * that is, is alpha.start == beta.start but alpha.end < beta.end?  
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.start == beta.start AND alpha.end < beta.end
  **/
  public ExtendedBoolean starts(Period alpha, Period beta);

  /** 
  * Does this Period finish at the same time as another, but start after it,
  * that is, is alpha.end == beta.end but alpha.start < beta.start?  
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.end == beta.end AND alpha.start < beta.start
  **/
  public ExtendedBoolean finishes(Period alpha, Period beta);

  /** 
  * Does the end of a Period meet the start of another Period,
  * that is, is alpha.end == beta.start?  This is the inverse of metBy().
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.end == beta.start
  **/
  public ExtendedBoolean meets(Period alpha, Period beta);

  /** 
  * Is the start of a Period metBy the start of another Period,
  * that is, is beta.end == alpha.start?  This is the inverse of meets().
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.start == beta.end
  **/
  public ExtendedBoolean metBy(Period alpha, Period beta);

  /** 
  * Does one Period overlap another?  Two Periods overlap if they intersect
  * in time.  This operation is symmetric.
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha overlaps beta
  **/
  public ExtendedBoolean overlaps(Period alpha, Period beta);

  /** 
  * Does this Period contain another?  A periods contains another if
  * it starts before it and ends after it.  The containment need not
  * be strict.
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.start <= beta.start AND alpha.end >= beta.end
  **/
  public ExtendedBoolean contains(Period alpha, Period beta);

  /** 
  * Is this Period during another?  A periods is during another if
  * it starts after it and ends before it.  The containment need not
  * be strict. 
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.start >= beta.start AND alpha.end <= beta.end
  **/
  public ExtendedBoolean during(Period alpha, Period beta);

  /** 
  * Does this Period represent the same times as another?  
  * A period is equalTo another if it both starts and finishes it.
  * This operation is symmetric.
  * @param alpha - Period to compare
  * @param beta - Period to compare
  * @return alpha.start == beta.start AND alpha.end == beta.end
  **/
  public ExtendedBoolean equalTo(Period alpha, Period beta);

  /** 
  * Does this Instant represent the same times as another?  
  * An instant is equalTo another if it is at the same time.
  * This operation is symmetric.
  * @param alpha - Instant to compare
  * @param beta - Instant to compare
  * @return alpha == beta
  **/
  public ExtendedBoolean equalTo(Instant alpha, Instant beta);

  /** 
  * Displace (add) a Period on the time-line by an Interval 
  * @param alpha - displacee
  * @param beta - displacor
  * @return - [alpha.start + beta, alpha.end + beta]
  **/
  public Period add(Period alpha, Interval beta);

  /** 
  * Displace (add) an Instant on the time-line by an Interval 
  * @param alpha - displacee
  * @param beta - displacor
  * @return - alpha + beta
  **/
  public Instant add(Instant alpha, Interval beta);

  /** 
  * Add one Interval to another
  * @param alpha - displacee
  * @param beta - displacor
  * @return - alpha + beta
  **/
  public Interval add(Interval alpha, Interval beta);

  /** 
  * Displace (subtract) a Period on the time-line by an Interval 
  * @param alpha - displacee
  * @param beta - displacor
  * @return - [alpha.start - beta, alpha.end - beta]
  **/
  public Period subtract(Period alpha, Interval beta);

  /** 
  * Displace (subtract) an Instant on the time-line by an Interval 
  * @param alpha - displacee
  * @param beta - displacor
  * @return - alpha - beta
  **/
  public Instant subtract(Instant alpha, Interval beta);

  /** 
  * Subtract one Interval from another
  * @param alpha - displacee
  * @param beta - displacor
  * @return - alpha - beta
  **/
  public Interval subtract(Interval alpha, Interval beta);

  /** 
  * Negate an Interval. 
  * @param alpha - Interval to negate
  * @return - -alpha
  **/
  public Interval negate(Interval alpha);

  /** 
  * Multiply an Interval by an integer constant.  
  * @param alpha - Interval to multiply
  * @param n - by how many 
  * @return result - (alpha * n)
  **/
  public Interval multiply(Interval alpha, int n);

  /** 
  * Divide an Interval by an integer constant.  
  * @param alpha - Interval to divide
  * @param n - by how many 
  * @return result - (alpha div n)
  **/
  public Interval divide(Interval alpha, int n);

  /** 
  * Divide an Interval by an Interval (integer division). 
  * @param alpha - divisee
  * @param beta - divisor
  * @param result - alpha div beta
  **/
  public int divide(Interval alpha, Interval beta);

  /** 
  * Temporal constructor - Choose the first (earliest) from among a pair of instants
  * @param alpha - Instant 
  * @param beta - Instant 
  * @return first Instant on the time-line from among alpha and beta
  **/
  public Instant first(Instant alpha, Instant beta);

  /** 
  * Temporal constructor - Choose the last (latest) from among a pair of instants
  * @param alpha - Instant 
  * @param beta - Instant 
  * @return last Instant on the time-line from among alpha and beta
  **/
  public Instant last(Instant alpha, Instant beta);

}
