package tauzaman.property;

import java.util.*;
import java.net.*;
import org.w3c.dom.*;

import tauzaman.XMLParser;
import tauzaman.XMLParseException;

/**
* <code>PropertyRepository</code> class manages a pool of <code>Properties</code>.
* For example:
* <pre>
*    PropertyRepository cr = new PropertyRepository();
*    // null means load every single Property in the url
*    cr.loadProperty(new URL("http://www.eecs.wsu.edu/~cdyreson/pub/tauZaman/defaultProperties.xml"), null);
* </pre>
*
* For now it does not have unloadProperty() behavior.
*
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 11/15/02
* @see     tauzaman.property.Property
* @status design complete, implementation complete
*/
public class PropertyRepository{

  /**
  * Hashtable, which has url and name of a <code>Property</code> as 
  * keys and their corresponding <code>Property</code> object as values.
  */
  private Hashtable propertyRepository = null;

  /**
  * All currently avaliable <code>Property</code> names
  */
  private static String [] propertyNames = {"Locale", "InstantInputFormat", "InstantOutputFormat", "IntervalInputFormat",
 "IntervalOutputFormat", "NowRelativeInstantInputFormat", "NowRelativeInstantOutputFormat", "PeriodInputFormat",
 "PeriodOutputFormat", "IndeterminateInstantInputFormat", "IndeterminateInstantOutputFormat", "OverrideInputOrder"};


  /**
  * Constructs a <code>PropertyRepository</code> object.
  */ 
  public PropertyRepository(){
      propertyRepository = new Hashtable();
  }


  /**
  * Returns true if given property name is one of the valid names. 
  * False otherwise.
  *
  * @param propertyName String name of the <code>Property</code>
  * being checked
  *
  * @return boolean, which is true if given property name is valid
  * false otherwise
  */
  public boolean isValidPropertyName(String propertyName){
      for(int i = 0; i < propertyNames.length; i++)
          if(propertyNames[i].equals(propertyName))
              return true;
      return false;
  }
  
  /**
  * Loads <code>Property</code>(ies) into this <code>PropertyRepository</code>     
  * and also returns them.    
  *    
  * @param url URL pointing to Property Specification file
  * @param names String names of the <code>Property</code>(ies) that are wanted to be loaded
  *
  * @return an array of <code>Property</code>(ies) loaded
  *
  * @throws PropertyFormationException if any abnormal condition occurs when forming
  * <code>Property</code>(ies)
  */
  public Property [] loadProperty(URL url, String [] names) throws PropertyFormationException{
 
      Vector properties = null;
      /* load all the Properties in url */
      if(names == null)
          properties = loadPropertyInternal(url, propertyNames, true);
      else
          properties = loadPropertyInternal(url, names, false);
      
      /* convert it to array */
      Property propertyList [] = new Property [properties.size()];
      for(int i = 0; i < properties.size(); i++)
          propertyList[i] = (Property)properties.elementAt(i);

      return propertyList;
  }

  /**
  * Loads <code>Property</code>(ies) into this <code>PropertyRepository</code>
  * and also returns them.
  *
  * @param url URL pointing to Property Specification file
  * @param names String names of the <code>Property</code>(ies) that are wanted to be loaded
  * @param dflt boolean representing the default behavior
  *
  * @return a dynamic array of <code>Property</code>(ies) loaded
  *
  * @throws PropertyFormationException if any abnormal condition occurs when forming
  * <code>Property</code>(ies)
  * <i>(internal use only)</i>
  */
  private Vector loadPropertyInternal(URL url, String names [], boolean dflt) throws PropertyFormationException{
      
      Vector leftNames = new Vector(), properties = new Vector();

      for(int i = 0; i < names.length; i++){
          String key = url.toExternalForm() + "%" + names[i];
          if(propertyRepository.containsKey(key))
              properties.addElement(propertyRepository.get(key));
          else 
              leftNames.addElement(names[i]);
      }
      
      if(leftNames.size()>0){
          XMLParser parser = new XMLParser();
          Element root = null;
          try{
              root = parser.parseURL(url);
          }
          catch(XMLParseException xpe){
              throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                url.toExternalForm() + " exception when parsing to DOM!", xpe);
          }

          /* first get field value support mapper, if exists */
          Hashtable fvMapper = new Hashtable();
          
          Vector fvMapperNodes = parser.locator(root, "fieldValueSupportMapper");
          if(fvMapperNodes.size() != 0){

              Element fvMapperNode = (Element)fvMapperNodes.elementAt(0);

              Vector fvNodes = parser.locator(fvMapperNode, "fieldvaluesupport");

              for(int i = 0; i < fvNodes.size(); i++){
                  Element fvNode = (Element)fvNodes.elementAt(i);
                  String label = fvNode.getAttribute("label"); 
                  URL fvUrl = null;
                  try{
                      fvUrl = new URL(fvNode.getAttribute("url"));
                  }
                  catch(MalformedURLException e){
                      throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                                              url.toExternalForm() + " bad URL!", e);
                  }

                  if(fvMapper.containsKey(label))
                      throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                       url.toExternalForm() + " mapping already defined!: " + label);

                  fvMapper.put(label, fvUrl);
              }
          }

          for(int i = 0; i < leftNames.size(); i++){
              String leftName = (String)leftNames.elementAt(i);
              boolean found = false;
              Vector propertyNodes = parser.locator(root, "property");
              for(int j = 0; j < propertyNodes.size(); j++){
                  Element propertyNode = (Element)propertyNodes.elementAt(j);
                  
                  String propertyName = propertyNode.getAttribute("name");
                  if(propertyName.equals(leftName)){
                      Property property = new Property(propertyName, url, propertyNode, fvMapper);
                      String key = url.toExternalForm() + "%" + leftName;
                      propertyRepository.put(key, property);
                      properties.addElement(property);
                      found = true;
                  }
              }
              if(!found && !dflt)
                  throw new PropertyFormationException("Exception when parsing xml file in: " + 
              url.toExternalForm() + " specified focus property is not found in specification!");
          }
      }
      return properties;
  }

}
