package tauzaman.property;

import org.w3c.dom.*;
import java.util.*;
import java.net.*;

/**
* <p> <code>FieldInfo</code> class keeps the content of an XML element, 
* called <i>fieldInfo</i>, in a <code>Property</code> specification 
* file.</p> <p><i>fieldInfo</i> element contains attributes that store a 
* Field Name, a Format Variable Name and a Field Value Table|Function 
* name.</p>
*
* <p><a href="http://www.eecs.wsu.edu/~burgun/research/xmlreptoc.htm"> XML Representation of Properties</a></p>
* @author  Curtis Dyreson and Bedirhan Urgun
* @version 0.1, 11/15/02   
* @see     tauzaman.property.Property
* @see     tauzaman.field.Field
* @see     tauzaman.field.Fields
* @status design complete, implementation complete
*/

public class FieldInfo{
  
  /** 
  * Variable name that is stored in format string. 
  */
  private String variable = null;

  /** 
  * Name of the <code>Field</code>.
  */
  private String name = null;

  /** 
  * URL of the Field Value service (name of the table or function) to be used. 
  */
  private URL using = null;
  
  /** 
  * Constructs a <code>FieldInfo</code> object, from given DOM node parsed
  * from Property Specification file. 
  *
  * @param root DOM Element
  *
  * @param fvMapper A Hashtable, which maps field value support labels to their urls.
  * This information is passed by <code>PropertyRepository</code> since this information
  * resides in Property specification file (as helper) rather than in individual Properties.
  *
  * @param url URL of Property Specification file this ImportFormat is gotten
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */
  public FieldInfo(Element root, Hashtable fvMapper, URL url) throws PropertyFormationException{
      formFieldInfo(root, fvMapper, url);
  }

  /**
  * Initializes this <code>FieldInfo</code> from DOM Node, which points to  
  * a fieldInfo element in the Property Specification file. 
  *
  * @param root DOM Element
  *
  * @param fvMapper A Hashtable, which maps field value support labels to their urls.
  * This information is passed by <code>PropertyRepository</code> since this information
  * resides in Property specification file (as helper) rather than in individual Properties.
  *
  * @param url URL of Property Specification file this ImportFormat is gotten
  *
  * @throws PropertyFormationException if any problem occurs
  * when parsing the property specification file
  */
  private void formFieldInfo(Element root, Hashtable fvMapper, URL url) throws PropertyFormationException{

      if(root.hasAttribute("variable"))
          variable = root.getAttribute("variable");
      else
          throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                      url.toExternalForm() + " no variable is defined!");
   
      if(root.hasAttribute("name"))
          name = root.getAttribute("name");
      else
          throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                          url.toExternalForm() + " no name is defined!");

      if(root.hasAttribute("using")){
          /* first check if it is a label and referes to some URL in fvMapper */
          String usingAttrValue = root.getAttribute("using");
          if(fvMapper.containsKey(usingAttrValue)){
              using = (URL)fvMapper.get(usingAttrValue);
          }
          else{
              try{
                  using = new URL(usingAttrValue);
              }
              catch(MalformedURLException e){
                  throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                                          url.toExternalForm() + " bad URL!", e);
              }
          }
      }
      else
          throw new PropertyFormationException("Exception when parsing xml file in: " + 
                                         url.toExternalForm() + " no using is defined!");
      
  }

  /* Accessor methods */

  /** 
  * Returns the name of the variable 
  * @return name of the field varible
  **/
  public String getVariableName(){
      return variable;
  }

  /** 
  * Returns the <code>Field</code> Name of this <code>FieldInfo</code>
  * @return name of the <code>Field</code>
  * @see tauzaman.field.Field
  */
  public String getFieldInfoName(){
      return name;
  }

  /** 
  * Returns the URL of the FVS service table|function.
  * @return URL url of the table/function that this FieldInfo will use
  **/
  public URL getFVSUrl(){
      return using;
  }

  
  /** 
  * toString method
  * @return string representation of this object
  **/
  public String toString(){
      /* return as string all three String variables */
      return "FieldInfo: " + getVariableName() + " " + getFieldInfoName() + " " + getFVSUrl().toExternalForm();
  }
  
}
