package tauzaman.io;

import java.net.*;

import tauzaman.*;
import tauzaman.field.*;
import tauzaman.field.fvsupport.*;
import tauzaman.property.*;
import tauzaman.calendricsystem.*;
import tauzaman.calendar.*;
import tauzaman.timestamp.*;

/**
* <code>Output</code> class handles conversion of timestamps
* to temporal constants.
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/
public class Output{
 
  /**
  * A handle to <code>CalendricSystem</code>, which is the cached
  * <code>CalendricSystem</code> of parent <code>TauZamanLocalService</code>
  * in a temporal data type.
  **/
  private CalendricSystem cachedCalendricSystem = null;

  /**
  * A handle to <code>PropertyManager</code>, which is the cached
  * <code>PropertyManager</code> of parent <code>TauZamanLocalService</code>
  * in a temporal data type.
  * Cached <code>PropertyManager</code> means the <code>PropertyManager</code>.
  * which corresponds to cached <code>CalendricSystem</code>.
  **/
  private PropertyManager cachedPropertyManager = null;


  /** A handle to <code>FVSupportRepository</code> of parent <code>TauZamanSystem</code> **/
  private FVSupportRepository fvsr = null;


  /**
  * Constructs an <code>Output</code> object.
  * 
  * @param cachedCalendricSystem A handle to <code>CalendricSystem</code>, which is the cached
  * <code>CalendricSystem</code> of parent <code>TauZamanLocalService</code> in a temporal
  * data type.
  * @param cachedPropertyManager A handle to <code>PropertyManager</code>, which is the cached
  * <code>PropertyManager</code> of parent <code>TauZamanLocalService</code> in a temporal
  * data type.
  * @param fvsr A handle to <code>FVSupportRepository</code> of parent <code>TauZamanSystem</code>
  */
  public Output(CalendricSystem cachedCalendricSystem, PropertyManager cachedPropertyManager, FVSupportRepository fvsr){
      this.cachedCalendricSystem = cachedCalendricSystem;
      this.cachedPropertyManager = cachedPropertyManager;
      this.fvsr = fvsr;
  }

  /**
  * Returns formed String temporal constant given <code>Granule</code>s
  * and name of the <code>Property</code> to be used.
  *
  * @param granules array of <code>Granule</code> objects to be parsed into temporal constant
  * @param propertyName String name of the <code>Property</code>
  * @param calendricSystemName String name of the <code>CalendricSystem</code>, which will be 
  * used to parse given timestamps
  *
  * @throws IOException if any abnormal condition occurs when parsing
  * <code>Granule</code>(s) and forming output string
  */
  public String parseOutput(Granule [] granules, String propertyName) throws IOException{

      /*
      * String that corresponds to given temporal data type (It might
      * consist of more than one Granule(s) ).
      */
      String output = null;

      /* first get the Property and check if it has a cache */
      Property property = null;
      try{
          property = cachedPropertyManager.getProperty(propertyName);
      }
      catch(PropertyServiceException pse){
          throw new IOException("Exception occurred when parsing input", pse);
      }

      /* Fields and format that correspond to Property */
      Fields fields = null;
      String format = null;

      if(property.hasCache()){     
          /* get the cached Property */
          PropertyCache propertyCache = property.getCache();
          
          /* get all information we need, format as string, fields */
          fields = propertyCache.getFields();
          format = propertyCache.getFormatString();
      }
      else{
          // Convert property to its Fields corresponding
          FieldsBuilder fieldsBuilder = new FieldsBuilder(propertyName, null, cachedPropertyManager);

          /* get all information we need, format as string, fields */
          fields = fieldsBuilder.formFields();
          format = fieldsBuilder.getFormat();

          /* now set the property's cache, true is passed as dummy actual param. */
          property.setCache(new PropertyCache(fields, null, format, true));
      }

      /* call IOMultiplexer's specific method */
      (new IOMultiplexer(cachedCalendricSystem, cachedPropertyManager)).fillFields(granules, fields);
      
      output = formOutputString(fields, format);
            
      return output;
  }

  /**
  * Returns String temporal constant given <code>Fields</code>, which are 
  * formed from <code>Granule</code>s, and String format, which comes form
  * <code>Property</code>'s <code>Format</code> information.
  *
  * @param fields <code>Fields</code> formed by using <code>Granule</code>s
  * @param format String format fetched from code>Property</code>'s <code>Format</code> information
  *
  * @return String temporal constant
  * 
  * @throws IOException if any abnormal condition occurs when forming the String
  * temporal constant
  */
  private String formOutputString(Fields fields, String format) throws IOException{

      StringBuffer output = new StringBuffer();

      /* tokenize format */
      String tokens [] = (new FormatParser(format)).parseFormat(false);

      /* query fields with variables and get their string values by using fv support 
         and also form the output string */
      for(int i = 0; i < tokens.length; i++){
          
          String token = tokens[i];

          if(token.matches("\\$[a-zA-Z]+")){
              /* token represents a variable */

              /* variables are kept "$abc" format in tokens, not just their names, in order
                 to differentiate them like above */

              /* get the Field corresponding to variable, we should be able to find it */
              Field field = fields.getFieldByVariableName(token.substring(1, token.length()));

              if(field == null){
                  System.out.println("This should not be happening!"); /* since fields are parsed by using same format before */
                  throw new IOException("Exception when parsing input: variable " + token + " does not match!");
              }

              // we should get fvsupport's indexToString here. 
 
              URL url = field.getUrl();

              FVSupport fvSupport = null;
              try{
                  fvSupport = fvsr.loadFVSupport(url);

                  // for now, forever, beginning, directions, periodDelimiter, distribution field does not
                  // has value, so, their index is always 1
                  long index;
                  if(field.getName().equals("now") || field.getName().equals("forever") 
                     || field.getName().equals("beginning") || field.getName().equals("directions") 
                     || field.getName().equals("periodDelimiter") || field.getName().equals("distribution"))
                      index = 1;
                  else
                      index = field.getValue();

                  output.append(fvSupport.indexToString(index));
              }
              catch(FVFormationException fvfe){
                  throw new IOException("Exception when parsing output: can not load fv support " +
                                                                            url.toExternalForm(), fvfe);
              }
              catch(FVServiceException fvse){
                   throw new IOException("Exception when parsing output: can not get string from fv support " +
                                                                            url.toExternalForm(), fvse);
              }

          }          
          else{
              /* handles escape characters */
              token = token.replaceAll("\\$\\$", "\\$");
 
              /* token represents a non-variable */
              output.append(token);
          }
      }

      return output.toString();
  }
}
