package tauzaman.io;

import java.util.regex.*;
import java.net.*;

import org.w3c.dom.*;

import tauzaman.*;
import tauzaman.field.*;
import tauzaman.field.fvsupport.*;
import tauzaman.property.*;
import tauzaman.timestamp.*;
import tauzaman.calendar.*;
import tauzaman.calendricsystem.*;


/* 
   the main source of latency is DOM parsing and some of it comes from 
   fillInFields, but not too much. 
*/

/**
* <code>Input</code> class handles conversion of temporal constants to
* timestamps. 
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/
public class Input{

    /* 
    * Determines parsing behaviour of Attr values and Text nodes of input 
    */
    private boolean whitespace = true;

    /*
    * <code>Fields</code> that is derived from the <code>Property</code>
    * and corresponds to unparsed timestamp
    */
    private Fields fields = null;

    /** 
    * A handle to <code>CalendricSystem</code>, which is the active 
    * <code>CalendricSystem</code> of parent <code>TauZamanLocalService</code>.
    **/
    private CalendricSystem activeCalendricSystem = null;

    /** 
    * A handle to <code>PropertyManager</code>, which is the active 
    * <code>PropertyManager</code> of parent <code>TauZamanLocalService</code>.
    * Active <code>PropertyManager</code> means the <code>PropertyManager</code>.
    * which corresponds to active <code>CalendricSystem</code>.
    **/
    private PropertyManager activePropertyManager = null;

      
    /** A handle to <code>FVSupportRepository</code> of parent <code>TauZamanSystem</code> **/
    private FVSupportRepository fvsr = null;

    /**
    * Constructs an <code>Input</code> object.
    * 
    * @param activeCalendricSystem A handle to <code>CalendricSystem</code>, which is the active
    * <code>CalendricSystem</code> of parent <code>TauZamanLocalService</code>
    * @param activePropertyManager A handle to <code>PropertyManager</code>, which is the active
    * <code>PropertyManager</code> of parent <code>TauZamanLocalService</code>
    * @param fvsr A handle to <code>FVSupportRepository</code> of parent <code>TauZamanSystem</code>
    * 
    */
    public Input(CalendricSystem activeCalendricSystem, PropertyManager activePropertyManager, FVSupportRepository fvsr){
        this.activeCalendricSystem = activeCalendricSystem;
        this.activePropertyManager = activePropertyManager;
        this.fvsr = fvsr;
    }

    /**
    * Returns formed <code>Granule</code>(s) given a temporal constant and 
    * name of the <code>Property</code> to be used.
    * 
    * @param input String temporal constant
    * @param propertyName String name of the <code>Property</code>
    *
    * @return an array of <code>Granule</code> objects
    *
    * @throws IOException if any abnormal condition occurs when parsing 
    * input and forming <code>Granule</code>(s)
    */
    public Granule [] parseInput(String input, String propertyName) throws IOException{

        long start;

        /* Granules to be formed */
        Granule granules [] = null;

        /* DOM Elements of parsed input and format */
        Element formatRootElement, inputRootElement = null;

        /* get a XMLParser to parse format and input into DOMs */
        XMLParser parser = new XMLParser();

        /* parse the input string into DOM first */
        try{
            inputRootElement = parser.parseFragment("<wrap>" + input + "</wrap>");
        }
        catch(XMLParseException xpe){
            throw new IOException("Exception when parsing input (XML Parsing exception): ", xpe);
        }

        /* get the Property and check if it has a cache */
        Property property = null;
        try{
            property = activePropertyManager.getProperty(propertyName);
        }
        catch(PropertyServiceException pse){
            throw new IOException("Exception occurred when parsing input", pse);
        }

        if(property.hasCache()){

            /* get the cached Property */
            PropertyCache propertyCache = property.getCache();

            /* get all information we need, formatRootElement, fields, whitespace behavior */
            fields = propertyCache.getFields();
            formatRootElement = propertyCache.getFormatDOM();
            whitespace = propertyCache.getWhitespace();

        }
        else{
            /* build a new FieldsBuilder, there is no cached Property */
            FieldsBuilder fieldsBuilder = new FieldsBuilder(propertyName, null, activePropertyManager);

            /* get all information we need, formatRootElement, fields, whitespace behavior */
            fields = fieldsBuilder.formFields();
            try{ 
                formatRootElement = parser.parseFragment("<wrap>" + fieldsBuilder.getFormat() + "</wrap>"); 
            }
            catch(XMLParseException xpe){
                throw new IOException("Exception when parsing input (XML Parsing exception): ", xpe);
            }
            whitespace = fieldsBuilder.getWhitespace();

            /* now set the property's cache, I could also set format as String but no need */
            property.setCache(new PropertyCache(fields, formatRootElement, null, whitespace));

        }

        /* fills the Fields with useful information from input */
        parseInputInternal(inputRootElement, formatRootElement);

        /* here call specific method of IOMultiplexer to convert fields into "granule" */
        granules = (new IOMultiplexer(activeCalendricSystem, activePropertyManager)).getGranule(fields);
        return granules;
    }

    /**
    * <p>
    * Traverses two DOM trees (parsed from temporal constant and 
    * <code>Property</code>'s <code>Format</code>) and checks whether they
    * match. During this operation it also fetches useful information from
    * temporal constant.</p><p>
    * In this sense, this operation has two phases concurrently; first matching 
    * phase, where structure of two DOM trees are checked, and second fetching phase,
    * where information in input DOM tree is fetched using information in format DOM
    * tree.
    * </p>
    *
    * @param input root Element of DOM parsed input
    * @param format root Element of DOM parsed format
    *
    * @throws IOException if any abnormal condition occurs during the matching or
    * fetching process of input
    */
    private void parseInputInternal(Element input, Element format) throws IOException {

        /*
           The whole idea is, both input and format should match with their XML nodes and their
           types no matter parsing behavior is whitespace friendly or not. They also have to
           match with their attribute names. Since we can only store useful information in
           attr values and text nodes, and since they will be parsed character based, there is
           yet another method "fillInFields", which fetches useful information and fills Fields.
        */

        Node inputNode = input.getFirstChild();
        Node formatNode = format.getFirstChild();
        
        while(inputNode != null && formatNode != null){

            if(inputNode.getNodeType() == formatNode.getNodeType()){
          
                switch(inputNode.getNodeType()){
            
                case Node.ELEMENT_NODE:

                    if(((Element)inputNode).getTagName() != ((Element)formatNode).getTagName())
                        throw new IOException("Exception when comparing input and format: elements don't match! " + 
                                                ((Element)inputNode).getTagName() + " and " + ((Element)formatNode).getTagName());

                    NamedNodeMap inputAttrs = inputNode.getAttributes();
                    NamedNodeMap formatAttrs = formatNode.getAttributes();

                    /* As can be seen, as long as all attributes of input matches with 
                       format, then we are ok. This relaxes things... */
                    for(int i = 0; i < inputAttrs.getLength(); i++){
                        Attr inputAttr = (Attr)inputAttrs.item(i);
 
                        boolean found = false;
                        for(int j = 0; j < formatAttrs.getLength(); j++){
                            Attr formatAttr = (Attr)formatAttrs.item(j);
   
                            if((inputAttr.getName()).equals(formatAttr.getName())){

                                /* fill the Fields, if any useful information exists in this attr value */
                                /* this part take nothing, do not cache!! */
                                fillInFields(inputAttr.getValue(), formatAttr.getValue());

                                found = true;
                                break;
                            }
                        }

                        if(!found)
                            throw new IOException("Exception when comparing input and format: attributes don't match!");
                    }

                    /* recursively call this method */
                    parseInputInternal((Element)inputNode, (Element)formatNode);
                    break;

                case Node.TEXT_NODE:
                    fillInFields(inputNode.getNodeValue(), formatNode.getNodeValue());
                    break;
                }

                inputNode = inputNode.getNextSibling();
                formatNode = formatNode.getNextSibling();

            }
            else{
                /* this part takes care of in one hand an empty text node and in the other another node type case */     
                if(inputNode.getNodeType() == Node.TEXT_NODE){
                    String value = inputNode.getNodeValue();
                    if(value.matches("\\s+"))
                        inputNode = inputNode.getNextSibling();
                    else
                        throw new IOException("Exception when parsing input: Node types don't match for input and format");
                }
                else if(formatNode.getNodeType() == Node.TEXT_NODE){
                    String value = formatNode.getNodeValue();
                    if(value.matches("\\s+"))
                        formatNode = formatNode.getNextSibling();
                    else
                        throw new IOException("Exception when parsing input: Node types don't match for input and format");
                }
                else{
                    throw new IOException("Exception when parsing input: Node types don't match for input and format");
                }
            }
        }
    }

    /**
    * Fetches useful information from input according to information
    * in format. This is the fetching phase. 
    *
    * @param input String part of (either Attr values or Text values) DOM parsed temporal constant
    * @param format String part of (either Attr values or Text values) DOM parsed format
    *
    * @throws IOException if any abnormal condition occurs during 
    * fetching process of input
    */
    /* input could be cloned... But I don't use it anyway. */
    private void fillInFields(String input, String format) throws IOException{
       
      /* get rid of leading and ending whitespaces */           
      if(whitespace){
          input = input.trim();
          format = format.trim();
      }

      /* build a format parser to parse the format into tokens */
      FormatParser fp = new FormatParser(format);
      
      /* tokenize format, we'll do a character based parsing */            
      String tokens [] = fp.parseFormat(false); /* very low processing time - 0~1 ms-, do not cache! */

      /* each token should match with some input */                      
      for(int i = 0; i < tokens.length; i++){
                  
          String token = tokens[i];

          /* get rid of leading and ending whitespaces */           
          if(whitespace){
              input = input.trim();
              token = token.trim();
          }


          /* think about this case: input " [ " format "$var", this gives error even there is a 
             whitespace behavior, since we apply lookingAt to the current input...*/
          
          if(token.matches("\\$[a-zA-Z]+")){


              /* token represents a variable */ 
              
              /* variables are kept "$abc" format in tokens, not just their names, in order
                 to differentiate them like above */

              /* get the Field corresponding to variable, we should be able to find it */
              Field field = fields.getFieldByVariableName(token.substring(1, token.length()));
              
              if(field == null){
                  System.out.println("This should not be happening!"); /* since fields are parsed by using same format before */
                  throw new IOException("Exception when parsing input: variable " + token + " does not match!");
              }       
              
              /* here get the regex from fields' fieldInfo or default regex from tzls */
              String regex = null;
                  
              URL url = field.getUrl();
              FVSupport fvSupport = null;
              try{
                  fvSupport = fvsr.loadFVSupport(url);
                  regex = fvSupport.getRegex();
              }
              catch(FVFormationException fvfe){
                  throw new IOException("Exception when parsing input: can not load fv support " + 
                                                                              url.toExternalForm() , fvfe);
              }
                  
              if(regex == null){
                  regex = activeCalendricSystem.getDefaultRegex();
              }
              
              /* use regex package to find a match in input */
              Pattern pattern = Pattern.compile(regex);
              Matcher matcher = pattern.matcher(input);

              /* lookingAt method is exactly what we want */
              if(matcher.lookingAt()){

                  /* set fields value using stringToIndex method of related fv support */
                  try{
                      // set fields value using stringToIndex method of related fv support
                      field.setValue(fvSupport.stringToIndex(matcher.group()));
                  }
                  catch(FVServiceException fvse){
                      throw new IOException("Exception when parsing input: can not get index from fv support " + 
                                                                                 url.toExternalForm() ,fvse);
                  }

                  /* field is filled, set as dirty */
                  field.setAsDirty();

                  if(matcher.end() < input.length())
                      input = input.substring(matcher.end(), input.length());
                  else
                      input = "";
              } 
              else{
                   throw new IOException("Can not get variable: " + token + " corresponding info in: " + input + "#");
              }
          }   
          else{
              /* token represents a non-variable */

              /* handles escape characters */
              token = token.replaceAll("\\$\\$", "\\$");

              /* a pure character comparison for non-variables, it could also be done by regular expressions,
                 but this way is more reliable. */

              while(0 < token.length() && 0 < input.length()){
                  if(token.charAt(0) == input.charAt(0)){
                      input = input.substring(1, input.length());
                      token = token.substring(1, token.length());
                  }
                  else
                      throw new IOException("Exception when parsing input: non-variables don't match: " + token + " in " + input);

                  if(whitespace){
                      input = input.trim();
                      token = token.trim();
                  }

              }
              if(token.length() > 0)
                  throw new IOException("Exception when parsing input: non-variables don't match: " + token + " in " + input);

          }
      }
      /* all the input should be consumed */
      if(input.length() > 0)
          throw new IOException("Exception when parsing input: non-variables don't match for input: " + input);

    } 

}
