package tauzaman.io;

import tauzaman.*;
import tauzaman.calendar.Calendar;
import tauzaman.calendar.*;
import tauzaman.calendricsystem.*;
import tauzaman.timestamp.*;
import tauzaman.property.*;
import tauzaman.field.*;

/**
* <code>IOMultiplexer</code>, abbreviation for Input/Output Multiplexer, class
* classifies behaviours of different temporal data types using their parsed 
* <code>Fields</code>. <code>Fields</code> sits between timestamp, <code>Granule</code>,
* and temporal constant, String.
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/
class IOMultiplexer{
  
  /** A handle to <code>CalendricSystem</code>, which will be used in Input/Output
     multiplexing **/
  private CalendricSystem calendricSystem = null;

  /** A handle to <code>PropertyManager</code>, which will be used in Input/Output
     multiplexing **/
  private PropertyManager propertyManager = null;


  /**
  * Constructs a <code>IOMultiplexer</code> object ready to serve.
  * 
  * @param calendricSystem <code>CalendricSystem</code>, which will be used in Input/Output
  * multiplexing
  * @param propertyManager <code>PropertyManager</code>, which will be used in Input/Output
  * multiplexing
  */
  IOMultiplexer(CalendricSystem calendricSystem, PropertyManager propertyManager){
      this.calendricSystem = calendricSystem;
      this.propertyManager = propertyManager;
  }

  /**
  * Returns an array of <code>Calendar</code>s, which are in order
  * to call for temporal constant or timestamp parsing.
  * 
  * @return an array of ordered <code>Calendar</code>s
  * 
  * @throws IOException if any abnormal condition occurs when getting
  * ordered <code>Calendar</code>s
  */
  private Calendar [] getOrderedCalendars() throws IOException{

      /* form ordered calendars by using OverrideInputOrder Property */
      Calendar calendars [] = null;
      try{
          calendars = calendricSystem.getOrderedCalendars(propertyManager.getProperty("OverrideInputOrder"));
      }
      catch(CalendricSystemServiceException csse){
          throw new IOException("Exception when parsing, cannot get ordered calendars", csse);
      }
      catch(PropertyServiceException pse){
          throw new IOException("Exception when parsing, cannot get ordered calendars", pse);
      }

      return calendars;
  }

  /**
  * Returns corresponding <code>Granule</code> of <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted into <code>Granule</code>
  * 
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to its corresponding <code>Granule</code>
  */
  private Granule fieldsToGranule(Fields fields) throws IOException{

      Granule granule = null;

      /* get ordered Calendars to apply to fields */
      Calendar calendars [] = getOrderedCalendars();
     
      /* try each Calendar to parse fields into Granule */
      for(int i = 0; i < calendars.length; i++){
          try{
              granule = calendars[i].fieldsToGranule(fields);
          }
          catch(CalendarServiceException cse){
              throw new IOException("Exception when parsing input", cse);
          }
          if(granule != null){
              // before returning it set its Calendar URL
              granule.getGranularity().setCalendarUrl(calendars[i].getUrl());
              return granule;
          }
      }
      
      /* none of them succeeds, throw an exception */
      throw new IOException("Exception when parsing input, no calendar was able to parse fields");
  }

  /**
  * Fills given <code>Fields</code>'s <code>Field</code> objects
  * according to information in <code>Granule</code>.
  *
  * @param granule <code>Granule</code> containing information to fill fields 
  * @param fields <code>Fields</code> to be filled
  * 
  * @throws IOException if any abnormal condition occurs when filling
  * <code>Fields</code> in according to information in <code>Granule</code>
  */
  private void granuleToFields(Granule granule, Fields fields) throws IOException{
      
      /* get ordered Calendars to apply to fields */
      Calendar calendars [] = getOrderedCalendars();
     
      /* try each Calendar to parse fields into Granule */
      for(int i = 0; i < calendars.length; i++){
          try{
              // granuleToFields returns true if parsing is successfull 
              if( calendars[i].granuleToFields(granule, fields) )
                  return;
          }
          catch(CalendarServiceException cse){
              throw new IOException("Exception when parsing output", cse);
          }
      }

      /* none of them succeeds, throw an exception */
      throw new IOException("Exception when parsing output, no calendar was able to parse granule");
  }

  /************* Output Behaviors *******************/

  /**
  * Fills the values of <code>Fields</code>, which was formed by an 
  * output <code>Property</code>, according to the information in 
  * <code>Granule</code>. This filled <code>Fields</code> will be used
  * to form a temporal constant, that corresponds to <code>Granule</code>.
  *
  * @param granules an array of <code>Granule</code> objects, which will be 
  * used to fill fields
  * @param fields <code>Fields</code> to be filled
  *
  * @throws IOException if any abnormal condition occurs when filling the
  * <code>Fields</code>
  */
  void fillFields(Granule [] granules, Fields fields) throws IOException{

      /* get the name of the Property that this fields are formed from */
      String fieldsName = fields.getName();

      if(fieldsName.equals("InstantOutputFormat")){
          fillInstantFields(granules[0], fields);
      }
      else if(fieldsName.equals("IntervalOutputFormat")){
          fillIntervalFields(granules[0], fields);
      }
      else if(fieldsName.equals("NowRelativeInstantOutputFormat")){
          fillNowRelativeInstantFields(granules[0], fields);
      }
      else if(fieldsName.equals("PeriodOutputFormat")){
          fillPeriodFields(granules, fields);
      }
      else if(fieldsName.equals("IndeterminateInstantOutputFormat")){
          fillIndeterminateInstantFields(granules[0], fields);
      }
      else
          throw new IOException("Exception when output parsing, No such format: " + fieldsName);
  }

  private void fillInstantFields(Granule granule, Fields fields) throws IOException {

      Fields fieldsArray [] = fields.getImmediateFields();
      Field fieldArray [] = fields.getImmediateField();

      if(fieldsArray.length != 0 || fieldArray.length == 0)
          throw new IOException("Exception when output parsing, more importFormat or less fieldInfo elements in InstantOutputFormat");
      
      if(fieldArray.length == 1){
          if("now".equals(fieldArray[0].getName())){

              // be sure that granule is NowRelativeGranule
              String whichGranule = granule.getClass().getName();
              if(!whichGranule.equals("tauzaman.timestamp.NowRelativeGranule"))
                  throw new IOException("Exception when output parsing, Granule type does not match InstantInputFormat with \"now\"");

          }
          else if("forever".equals(fieldArray[0].getName())){

              // be sure that granule contains TimeValue.END_OF_TIME
              if(granule.getGranule() != TimeValue.END_OF_TIME)
                  throw new IOException("Exception when output parsing, Granule type does not match InstantInputFormat with \"forever\"");

          }
          else if("beginning".equals(fieldArray[0].getName())){
              // be sure that granule contains TimeValue.BEGINNING_OF_TIME
              if(granule.getGranule() != TimeValue.BEGINNING_OF_TIME)
                  throw new IOException("Exception when output parsing, Granule type does not match InstantInputFormat with \"beginning\"");

          }
      }

      /* use calendars to parse granule and fill fields */
      granuleToFields(granule, fields);

  }

  private void fillIntervalFields(Granule granule, Fields fields) throws IOException {

      Fields fieldsArray [] = fields.getImmediateFields();
      Field fieldArray [] = fields.getImmediateField();

      if(fieldsArray.length != 0 || fieldArray.length == 0)
          throw new IOException("Exception when output parsing, more importFormat or less fieldInfo elements in IntervalOutputFormat");
      

      /* use calendars to parse granule and fill fields */
      granuleToFields(granule, fields);

  }

  private void fillNowRelativeInstantFields(Granule granule, Fields fields) throws IOException {

      Fields fieldsArray [] = fields.getImmediateFields();
      Field fieldArray [] = fields.getImmediateField();

      // be sure that granule is NowRelativeGranule
      String whichGranule = granule.getClass().getName();

      if(!whichGranule.equals("tauzaman.timestamp.NowRelativeGranule"))
          throw new IOException("Exception when output parsing, Granule type does not match NowRelativeInstantOutputFormat");

      if(fieldsArray.length > 1 || fieldArray.length == 0)
          throw new IOException("Exception when output parsing, more importFormat or less fieldInfo " + 
                                                              "elements in NowRelativeInstantOutputFormat");

      if(fieldsArray.length == 1){
          if(!fieldsArray[0].getName().equals("IntervalOutputFormat"))
              throw new IOException("Exception when output parsing, fields: " + fieldsArray[0].getName() + 
                                                     " does not match with NowRelativeInstantOutputFormat");

          // what happens when a granule does not have a granularity and asked to return one ???
          Granule intervalGranule = new Granule(granule.getGranularity(), granule.getGranule());
          fillIntervalFields(intervalGranule, fieldsArray[0]);
      }

     // if there is no interval, that is there is no fields, then we return fields as they are
  }

  private void fillPeriodFields(Granule [] granules, Fields fields) throws IOException {

      Fields fieldsArray [] = fields.getImmediateFields();
      Field fieldArray [] = fields.getImmediateField();

      /* for now we assume a Period consists of two fields. So, "now", "forever", "beginning" will be used inside 
         one of those fields as Instants */

      if(fieldsArray.length != 2 || fieldArray.length != 2 || granules.length != 2)
          throw new IOException("Exception when output parsing, wrong number of importFormat or " +
                                     "fieldInfo elements in PeriodOutputFormat or wrong number of Granules passed: " + granules.length);

      if(fieldsArray[0].getName().equals("IndeterminateInstantOutputFormat") && 
            fieldsArray[1].getName().equals("IndeterminateInstantOutputFormat")){

          fillIndeterminateInstantFields(granules[0], fieldsArray[0]);
          fillIndeterminateInstantFields(granules[1], fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("InstantOutputFormat") && 
              fieldsArray[1].getName().equals("InstantOutputFormat")){

          fillInstantFields(granules[0], fieldsArray[0]);
          fillInstantFields(granules[1], fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("IndeterminateInstantOutputFormat") && 
              fieldsArray[1].getName().equals("InstantOutputFormat")){

          fillIndeterminateInstantFields(granules[0], fieldsArray[0]);
          fillInstantFields(granules[1], fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("InstantOutputFormat") && 
              fieldsArray[1].getName().equals("IndeterminateInstantOutputFormat") ){

          fillInstantFields(granules[0], fieldsArray[0]);
          fillIndeterminateInstantFields(granules[1], fieldsArray[1]);

      }
      else{
          throw new IOException("Exception when output parsing, unmatched importFormat couple in PeriodOutputFormat: " +  
                                                               fieldsArray[0].getName() + " " + fieldsArray[1].getName());
      }

  }

  private void fillIndeterminateInstantFields(Granule granule, Fields fields) throws IOException {

      Fields fieldsArray [] = fields.getImmediateFields();
      Field fieldArray [] = fields.getImmediateField();

      /* for now we assume a Indeterminate Instant consists of two fields. So, "now", "forever", "beginning" will be used inside 
         one of those fields as Instants */

      if(fieldsArray.length != 2 || fieldArray.length > 1)
          throw new IOException("Exception when output parsing, wrong number of importFormat or fieldInfo elements in PeriodOutputFormat");

      // do I care about distribution, when outputting? 


      /* actually there are 4 combinations 

         InstantOutputFormat ~ InstantOutputFormat
         InstantOutputFormat ~ NowRelativeInstantOutputFormat
         NowRelativeInstantOutputFormat ~ InstantOutputFormat         
         NowRelativeInstantOutputFormat ~ NowRelativeInstantOutputFormat

      */

      if(fieldsArray[0].getName().equals("InstantOutputFormat") && 
          fieldsArray[1].getName().equals("InstantOutputFormat")){

          fillInstantFields(new Granule(granule.getGranularity(), granule.getLower()), fieldsArray[0]);
          fillInstantFields(new Granule(granule.getGranularity(), granule.getUpper()), fieldsArray[1]);

       }
       else if(fieldsArray[0].getName().equals("InstantOutputFormat") &&
          fieldsArray[1].getName().equals("NowRelativeInstantOutputFormat")){ 

          fillInstantFields(new Granule(granule.getGranularity(), granule.getLower()), fieldsArray[0]);
          fillNowRelativeInstantFields(new NowRelativeGranule(granule.getGranularity(), granule.getUpper()), fieldsArray[1]);

       }
       else if(fieldsArray[1].getName().equals("InstantOutputFormat") &&
          fieldsArray[0].getName().equals("NowRelativeInstantOutputFormat")){ 

          fillNowRelativeInstantFields(new NowRelativeGranule(granule.getGranularity(), granule.getLower()), fieldsArray[0]);
          fillInstantFields(new Granule(granule.getGranularity(), granule.getUpper()), fieldsArray[1]);

       }
       else if(fieldsArray[0].getName().equals("NowRelativeInstantOutputFormat") &&
          fieldsArray[1].getName().equals("NowRelativeInstantOutputFormat")){ 

          fillNowRelativeInstantFields(new NowRelativeGranule(granule.getGranularity(), granule.getLower()), fieldsArray[0]);
          fillNowRelativeInstantFields(new NowRelativeGranule(granule.getGranularity(), granule.getUpper()), fieldsArray[1]);

       }
       else 
           throw new IOException("Exception when output parsing, or unmatched importFormat couple in IndeterminateInstantOutputFormat:" + 
                                                                      fieldsArray[0].getName() + " " + fieldsArray[1].getName());

  }


  /**************** Input Behaviors ******************/
 
  /**
  * Returns an array of <code>Granule</code>, which is formed by using 
  * given <code>Fields</code>.
  *
  * @param fields <code>Fields</code>, which represents temporal constant
  *
  * @return an array of <code>Granule</code> objects. Other than 
  * PeriodInputFormat only one <code>Granule</code> will be returned
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>s
  */
  Granule [] getGranule(Fields fields) throws IOException{

      /* for Instant, Interval, NowRelativeInstant and IndeterminateInstant Temporal Data Types */
      Granule granules [] = new Granule[1];

      /* Name of the Property used to form this fields */
      String fieldsName = fields.getName();

      /* if fields was formed by using Property named InstantInputFormat, 
         then we have to have one Granule */
      if(fieldsName.equals("InstantInputFormat")){
          granules[0] = getInstantGranule(fields);
          return granules;
      }
      /* if fields was formed by using Property named IntervalInputFormat, 
         then we have to have one Granule */
      else if(fieldsName.equals("IntervalInputFormat")){
          granules[0] = getIntervalGranule(fields);
          return granules;
      }
      /* if fields was formed by using Property named NowRelativeInstantInputFormat, 
         then we have to have one Granule */
      else if(fieldsName.equals("NowRelativeInstantInputFormat")){
          granules[0] = getNowRelativeInstantGranule(fields);
          return granules;
      }
      /* if fields was formed by using Property named IndeterminateInstantInputFormat, 
         then we have to have one Granule */
      else if(fieldsName.equals("IndeterminateInstantInputFormat")){
          granules[0] = getIndeterminateInstantGranule(fields);
          return granules;
      }
      /* if fields was formed by using Property named PeriodInputFormat, then we have 
         to have two Granules */
      else if(fieldsName.equals("PeriodInputFormat")){
          return getPeriodGranule(fields);
      }
      
      /* if fields was formed by any other Property names, then throw an exception */
      throw new IOException("Exception when input parsing, No such format: " + fieldsName);

  } 
 
  /**
  * Returns a <code>Granule</code>, which corresponds to an Instant, by using 
  * <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted to <code>Granule</code>
  *
  * @return a <code>Granule</code>, which corresponds to an Instant
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>
  */
  private Granule getInstantGranule(Fields fields) throws IOException{

      /* Fields objects that are directly under fields are returned */
      Fields fieldsArray [] = fields.getImmediateFields();

      /* Field objects that are directly under fields are returned */
      Field fieldArray [] = fields.getImmediateField();

      /* for an Instant there should be no Fields under fields and
         there should be at least one Field under fields */
      if(fieldsArray.length != 0 || fieldArray.length == 0)
          throw new IOException("Exception when input parsing, importFormat or less fieldInfo elements in InstantInputFormat");

      /* if there is only one Field in fields, that may be now | forever | beginning
         which need special handling  */
      if(fieldArray.length == 1){
          if("now".equals(fieldArray[0].getName())){
              return new NowRelativeGranule(calendricSystem.getDefaultGranularity());
          }
          else if("forever".equals(fieldArray[0].getName())){
              return new Granule(calendricSystem.getDefaultGranularity(), TimeValue.END_OF_TIME);
          }
          else if("beginning".equals(fieldArray[0].getName())){
              return new Granule(calendricSystem.getDefaultGranularity(), TimeValue.BEGINNING_OF_TIME);
          }
      }

      /* else, call Calendar to parse this fields, which only contains Field objects */
      return fieldsToGranule(fields);      
  }

  /**
  * Returns a <code>Granule</code>, which corresponds to an Interval, by using 
  * <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted to <code>Granule</code>
  *
  * @return a <code>Granule</code>, which corresponds to an Interval
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>
  */
  private Granule getIntervalGranule(Fields fields) throws IOException{
  
      /* Fields objects that are directly under fields are returned */
      Fields fieldsArray [] = fields.getImmediateFields();

      /* Field objects that are directly under fields are returned */
      Field fieldArray [] = fields.getImmediateField();

      /* for an Interval there should be no Fields under fields and
         there should be at least one Field under fields */
      if(fieldsArray.length != 0 || fieldArray.length == 0)
          throw new IOException("Exception when input parsing, importFormat or less fieldInfo elements in IntervalInputFormat");

      /* call Calendar to parse this fields, which only contains Field objects */
      return fieldsToGranule(fields);      
  }

  /**
  * Returns a <code>Granule</code>, which corresponds to an NowRelativeInstant, by using 
  * <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted to <code>Granule</code>
  *
  * @return a <code>Granule</code>, which corresponds to a NowRelativeInstant
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>
  */
  private Granule getNowRelativeInstantGranule(Fields fields) throws IOException{

      /* Fields objects that are directly under fields are returned */
      Fields fieldsArray [] = fields.getImmediateFields();

      /* Field objects that are directly under fields are returned */
      Field fieldArray [] = fields.getImmediateField();

      /* for an NowRelativeInstant there should be at most one Fields under fields or
         there should be at least one Field under fields */
      if(fieldsArray.length > 1 || fieldArray.length > 2)
          throw new IOException("Exception when input parsing, more importFormat or less fieldInfo" + 
                                                            " elements in NowRelativeInstantInputFormat");

      if(fieldsArray.length == 1){
          if(!fieldsArray[0].getName().equals("IntervalInputFormat"))
              throw new IOException("Exception when input parsing, fields: " + fieldsArray[0].getName() + 
                                                    " does not match with NowRelativeInstantInputFormat");

          Granule intervalGranule = getIntervalGranule(fieldsArray[0]);
          for(int i = 0; i < fieldArray.length; i++){
              if(fieldArray[i].getName().equals("directions")){
                  if(fieldArray[i].getValue() % 2 == 1) // odd index value means direction is negative
                      intervalGranule = intervalGranule.negate();
              }
              // we don't care about now or anyother wrong field
              // if there is no "directions" Field then default behavior is
              // direction +
          }

          return new NowRelativeGranule(intervalGranule.getGranularity(), intervalGranule.getGranule());

      }

      // if there is no interval then we don't care about direction
      return new NowRelativeGranule(calendricSystem.getDefaultGranularity());

  }

  /**
  * Returns an array of <code>Granule</code>, which corresponds 
  * to an Period, by using <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted to <code>Granule</code>
  *
  * @return a <code>Granule</code>, which corresponds to a Period
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>(s)
  */
  private Granule [] getPeriodGranule(Fields fields) throws IOException{

      /* Fields objects that are directly under fields are returned */
      Fields fieldsArray [] = fields.getImmediateFields();

      /* Field objects that are directly under fields are returned */
      Field fieldArray [] = fields.getImmediateField();

      /* Assumption is that a Period consists of two "Fields". So, "now", "forever", "beginning" will be used inside 
         one of those fields as Instants */

      /* A Period should have two Fields objects and at most two Field objects(delimiters) */
      if(fieldsArray.length != 2 || fieldArray.length > 2)
          throw new IOException("Exception when input parsing, wrong number of importFormat or fieldInfo elements in PeriodInputFormat");

      /* left and right closed/open behaviors of this Period */
      boolean leftClosed = false, rightClosed = false;

      /* default behaviors for delimiters is "open". For now it works all or none basis */
      if(fieldArray.length == 2){

          if(fieldArray[0].getName().equals("periodDelimiter")){
              if(fieldArray[0].getValue() % 2 == 0) //even indexes mean closed
                  leftClosed = true;
          }
          else
              throw new IOException("Exception when input parsing, no left Period Delimiter in PeriodInputFormat");

          if(fieldArray[1].getName().equals("periodDelimiter")){
              if(fieldArray[1].getValue() % 2 == 0) //even indexes mean closed
                  rightClosed = true;
          }
          else
              throw new IOException("Exception when input parsing, no right Period Delimiter in PeriodInputFormat");

      }
 
      /* A Period always contain two Granules. */
      Granule granules [] = new Granule[2];

      /* Period is formed as both Instants are indeterminate */
      if(fieldsArray[0].getName().equals("IndeterminateInstantInputFormat") && 
                                            fieldsArray[1].getName().equals("IndeterminateInstantInputFormat")){

          // get the indeterminate instant granule
          granules[0] = getIndeterminateInstantGranule(fieldsArray[0]);
          granules[1] = getIndeterminateInstantGranule(fieldsArray[1]);
          
      }
      /* Period is formed as both Instants are determinate */
      else if(fieldsArray[0].getName().equals("InstantInputFormat") && fieldsArray[1].getName().equals("InstantInputFormat")){


          // get the determinate instant granules and assign them to output granule array in order
          granules[0] = getInstantGranule(fieldsArray[0]);
          granules[1] = getInstantGranule(fieldsArray[1]);

      }
      /* Period is formed as start Instant is indeterminate and ending instant is determinate */
      else if(fieldsArray[0].getName().equals("IndeterminateInstantInputFormat") && fieldsArray[1].getName().equals("InstantInputFormat")){

          // get the indeterminate instant granule
          granules[0] = getIndeterminateInstantGranule(fieldsArray[0]);
          // get the determinate instant granule
          granules[1] = getInstantGranule(fieldsArray[1]);

      }
      /* Period is formed as start Instant is determinate and ending instant is indeterminate */
      else if(fieldsArray[0].getName().equals("InstantInputFormat") && fieldsArray[1].getName().equals("IndeterminateInstantInputFormat")){

          // get the determinate instant granule
          granules[0] = getInstantGranule(fieldsArray[0]);
          // get the indeterminate instant granule
          granules[1] = getIndeterminateInstantGranule(fieldsArray[1]);

      }
      else{
          throw new IOException("Exception when input parsing, unmatched importFormat couple in PeriodInputFormat:" +  
                                                            fieldsArray[0].getName() + " " + fieldsArray[1].getName());
      }

      // according to period delimiters update Granules
      if(!leftClosed)
          granules[0] = granules[0].increment();

      if(!rightClosed){
          granules[1] = granules[1].decrement();
      }

      return granules;
  }

  /**
  * Returns a <code>Granule</code>, which corresponds to an IndeterminateInstant, by using 
  * <code>Fields</code>.
  *
  * @param fields <code>Fields</code> to be converted to <code>Granule</code>
  *
  * @return a <code>Granule</code>, which corresponds to a IndeterminateInstant
  *
  * @throws IOException if any abnormal condition occurs when converting 
  * <code>Fields</code> to <code>Granule</code>
  */
  private Granule getIndeterminateInstantGranule(Fields fields) throws IOException{

      /* Fields objects that are directly under fields are returned */
      Fields fieldsArray [] = fields.getImmediateFields();

      /* Field objects that are directly under fields are returned */
      Field fieldArray [] = fields.getImmediateField();

      /* Assumption is that a Indeterminate Instant consists of two fields. So, "now", "forever", 
         "beginning" will be used inside one of those fields as Instants */

      /* An IndeterminateInstant should contain two Fields objects and at most one Field object*/
      if(fieldsArray.length != 2 || fieldArray.length > 1)
          throw new IOException("Exception when input parsing, wrong number of importFormat or fieldInfo elements in PeriodInputFormat");

      // distribution ???
      if(fieldArray.length == 1){
          Field aField = fieldArray[0];
          long distribution = aField.getValue();
      }

      /* actually there are 4 combinations 

         InstantOutputFormat ~ InstantOutputFormat
         InstantOutputFormat ~ NowRelativeInstantOutputFormat
         NowRelativeInstantOutputFormat ~ InstantOutputFormat         
         NowRelativeInstantOutputFormat ~ NowRelativeInstantOutputFormat

      */

      Granule granules [] = new Granule[2];

      if(fieldsArray[0].getName().equals("InstantInputFormat") && 
         fieldsArray[1].getName().equals("InstantInputFormat")){

          // get the instant granules and assign them to output granule array in order
          granules[0] = getInstantGranule(fieldsArray[0]);
          granules[1] = getInstantGranule(fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("InstantInputFormat") &&
              fieldsArray[1].getName().equals("NowRelativeInstantInputFormat")){

          // get the instant granules and assign them to output granule array in order
          granules[0] = getInstantGranule(fieldsArray[0]);
          granules[1] = getNowRelativeInstantGranule(fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("NowRelativeInstantInputFormat") &&
              fieldsArray[1].getName().equals("InstantInputFormat")){

          // get the instant granules and assign them to output granule array in order
          granules[0] = getNowRelativeInstantGranule(fieldsArray[0]);
          granules[1] = getInstantGranule(fieldsArray[1]);

      }
      else if(fieldsArray[0].getName().equals("NowRelativeInstantInputFormat") &&
              fieldsArray[1].getName().equals("NowRelativeInstantInputFormat")){

          // get the instant granules and assign them to output granule array in order
          granules[0] = getNowRelativeInstantGranule(fieldsArray[0]);
          granules[1] = getNowRelativeInstantGranule(fieldsArray[1]);

      }
      else 
          throw new IOException("Exception when input parsing, unmatched importFormat couple in IndeterminateInstantInputFormat:" + 
                                                                      fieldsArray[0].getName() + " " + fieldsArray[1].getName());

      // check their granularities, they should be equal, get one of them, get TimeValues...
      // according to distribution get a PMF instance, there should be a default value for that
      return new Granule(granules[0].getGranularity(), granules[0].getGranule(), granules[1].getGranule());


  }

}

