package tauzaman.io;

import java.util.regex.*;
import java.util.*;

/**
* <code>FormatParser</code> class tokenizes given <code>Format</code>
* content to its variables and non-variables. Variables are String sequences
* , which matches with regular expression, "$[a-zA-Z]+" . Non-variables
* are the String tokens that are left when variables are fetched.
* 
* <pre>
*  "$month $year $day" => "$month", " ", "$year", " ", "$day"
* </pre>
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1 03/03/02
* @status design complete, implementation complete
*/

class FormatParser{

  /** String input that will be tokenized **/
  private String input = null;

  /* a variable starts with a $ sign. However, $ is also an 
     escape character for printing $. Also, after $ sign
     a variable should contain one or more letters. */
  private String varPattern = "(\\$\\$)*\\$[a-zA-Z]+";
      
  /** Pattern class of java.util.regex, that will be used when
      finding variable patterns in input string **/
  private Pattern pattern = null;

  /** Matcher class of java.util.regex, that will be used when
      finding variable patterns in input string **/
  private Matcher matcher = null;

  /**
  * Constructs a <code>FormatParser</code> object, which is ready to parse
  * given input.
  *
  * @param input String, which will be tokenized into its tokens
  */
  FormatParser(String input){

      /* param. input is modified during parsing */
      this.input = new String(input);

      pattern= Pattern.compile(varPattern);
  }


  /**
  * Returns tokens, which are resulted from parsing process.
  *
  * @param variablesOnly a boolean variable, which determines if
  * all tokens or just variable tokens will be returned
  *
  * @return array of String, which include resulted tokens
  */
  String [] parseFormat(boolean variablesOnly){

      Vector outputRaw  = new Vector();

      int index = 0;

      /* first find the first occurence of $ sign for a potential variable */
      while( (index = input.indexOf('$')) != -1){

          /* take all the rest of the input (after found $ sign) */
          String rest = input.substring(index, input.length());
          
          /* prepare to look for a variable pattern */
          matcher= pattern.matcher(rest);

          /* we have to find a variable starting from begginning of rest */
          if(matcher.lookingAt()){

              /* matching is "($$)*$abc", we have to trim it if necessary */
              int start = 0; // lookingAt always returns 0
              while(rest.charAt(start+1) == '$')
                  start++;

              /* get the variable with $ sign at the begginning */
              String variable = rest.substring(start, matcher.end());
    
              /* if we have to tokenize whole input */
              if(!variablesOnly){
                  /* also tokenize non-variable strings */
                  String nonVariable = input.substring(0, start + index);
                  /* it might be the case that "$abc$def" gives you "" as a non-variable */
                  if(nonVariable.length() != 0)
                      outputRaw.addElement(nonVariable);
              }
              /* add the variable */
              outputRaw.addElement(variable);
              /* update the input; index shows the remainder index in input, 
                 add matcher.end() to it, and get to the end of the input  */
              input = input.substring(index + matcher.end(), input.length());
          }
          /* although we found $ sign in input, it does not correspond to a variable */
          else{
              /* first get consequtive $ signs to prevent a false behaviour */
              while( (index < input.length()) && input.charAt(index) == '$')
                  index++;
              
              /* if we have to tokenize whole input */
              if(!variablesOnly){
                  /* also tokenize non-variable strings */
                  String nonVariable = input.substring(0, index);
                  /* it might be the case that "$" gives you "" as a non-variable */
                  if(nonVariable.length() != 0)
                      outputRaw.addElement(nonVariable);
              }
              /* "$" case */
              if(index < input.length())
                  input = input.substring(index, input.length());
              else
                  input = "";
          }
      }
 
      /* add what is left, if we have tokenize whole input */
      if( (input.length() != 0) && !variablesOnly ){
              outputRaw.addElement(input);
      }

      /* convert the vector into a string array */
      String output [] = new String[outputRaw.size()];

      for(int i = 0; i < outputRaw.size(); i++){
          output[i] = (String)outputRaw.elementAt(i);
      }
      return output;
  }


}

