package tauzaman.field;

import java.util.Vector;

/**
* <p><code>Fields</code> class represents a list of <code><b>Field</b></code>s.
* In other words <code>Fields</code> is simply the unparsed version
* of a time-stamp.</p>
* <p>If a <code><b>Field</b></code> exists in <code>Fields</code>, it means it is valid.
* Otherwise it means temporal constant that is being parsed does not have
* that component in it. </p>
* <p> An element of <code>Fields</code> can be either a <code><b>Field</b></code> or
* a <code>Fields</code> object. As an example; </p>
*  <pre>
*    A <code>Period</code> is defined with two <i>period delimiters</i> and
*    two <code>Instant</code>s. That means there are <code>FieldInfo</code>s
*    in corresponding <code>Property</code>
*    for <i>period delimiters</i> and also for each <code>Instant</code>s.
*    <code><b>Field</b></code>s that are derived from <code>FieldInfo</code>s for
*    <code>Instant</code>s should be kept seperately from the <code>Field</code>s
*    that are derived from <code>FieldInfo</code>s for <i>period delimiters</i>.
*    So, a <code>Fields</code> object will contain <code><b>Field</b></code>s for
*    <i>period delimiters</i> and also two <code>Fields</code> objects for
*    two <code>Instant</code>s.
*  </pre>
*  <i> This is a metamorphic structure </i>
*
* <p><a href="http://www.eecs.wsu.edu/~burgun/research/xmlrepproperties.htm#propertyPeriodInputFormat"> 
* Property Period Input Format</a></p>
*
* @author  Bedirhan Urgun and Curtis Dyreson
* @version 0.1, 10/10/02
* @see tauzaman.field.Field
* @see tauzaman.property.FieldInfo
* @see tauzaman.temporaldatatypes.Instant
* @status design complete, implementation complete
*/
public class Fields implements Cloneable{

  /**
  * A vector that is a list of <code>Field</code>s.
  */ 
  private Vector fields = null;

  /*
  * <code>Property</code> name that this <code>Fields</code> corresponds to
  */
  private String name = null;

  /**
  * Constructs a <code>Fields</code> object, which allocates memory for
  * a list that contains a set of <code>Field</code>s.
  * 
  * @param name String name of corresponding <code>Property</code> name that
  * this <code>Fields</code> is formed
  */
  public Fields(String name){
      /* allocate memory for the Field list */
      fields = new Vector();
    
      this.name = name;
  }

  /**
  * Returns a cloned version of this <code>Fields</code>. It produces
  * a deep copy of this <code>Fields</code>.
  *
  * @return clone of this <code>Fields</code>
  */ 
  public Object clone(){
      Fields clonedFields = new Fields(name);
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName(); 
          if(className.equals("tauzaman.field.Field")){
              Field aField = (Field)obj;
              clonedFields.addElement(aField.clone());
          }
          else{
              Fields aFields = (Fields)obj;
              /* call this method recursively because of recursive 
                 definition of this class */
              clonedFields.addElement(aFields.clone());
          }
      }
      return clonedFields;
  }

  /**
  * Adds a <code>Field</code> or a <code>Fields</code> to this
  * <code>Fields</code>.
  *
  * @param aFieldsElement which might either be <code>Fields</code> or
  * <code>Field</code>
  */
  public void addElement(Object aFieldsElement){
      fields.addElement(aFieldsElement);
  }

  /**
  * Returns name of this <code>Fields</code>.
  *
  * @return String name of this <code>Fields</code>
  */
  public String getName(){
      return name;
  }
  
  /**
  * Returns the first unmarked (clean) <code>Field</code> with a given variable name
  * that this <code>Fields</code> contains.
  *
  * @return <code>Field</code> that is not dirty and has given variable name. Returns null if
  * not exists
  */
  public Field getFieldByVariableName(String varName){
    
      /* recursively search the fields vector in order */
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Field")){
              Field aField = (Field)obj;
              if(varName.equals(aField.getVariableName()) && !aField.isDirty()){
                  return aField;
              }
          }
          else{
              Fields aFields = (Fields)obj;
              Field field = aFields.getFieldByVariableName(varName);
              if(field != null)
                  return field;
          }
      }
  
      return null;
  }

  /**
  * Returns the first <code>Field</code> with a given name
  * that this <code>Fields</code> contains.
  *
  * @return <code>Field</code> that has given name. Returns null if
  * not exists.
  */
  public Field getFieldByName(String name){
      /* search onyl immediate Field objects, used only by Calendar implementation, where
         only determinate Instant and Interval Temporal data types are considered */
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Field")){
              Field aField = (Field)obj;
              if(name.equals(aField.getName())){
                  return aField;
              }
          }
      }
  
      return null;
  }

  /**
  * Returns an array of <code>Fields</code>, which are immediate
  * <code>Fields</code> objects of this <code>Fields</code>.
  *
  * @return array of <code>Fields</code>
  */
  public Fields [] getImmediateFields(){
      int numberOfFields = 0;
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Fields")){
              numberOfFields++;
          }          
      }

      Fields fieldsArray [] = new Fields[numberOfFields];

      int index = 0;
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Fields")){
              fieldsArray[index] = (Fields)obj; 
              index++;
          }
      }
      
      return fieldsArray;
  }

  /**
  * Returns an array of <code>Field</code>, which are immediate
  * <code>Field</code> objects of this <code>Fields</code>.
  *
  * @return array of <code>Field</code>
  */
  public Field [] getImmediateField(){

      int numberOfField = 0;
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Field")){
              numberOfField++;
          }          
      }

      Field fieldArray [] = new Field[numberOfField];

      int index = 0;
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Field")){
              fieldArray[index] = (Field)obj; 
              index++;
          }
      }
      
      return fieldArray;
  }


  /**
  * Returns String representation of this <code>Fields</code>.
  * 
  *
  * @return String representation of this <code>Fields</code>
  */
  public String toString(){
      StringBuffer rep = new StringBuffer();
      rep.append("\nFields name: " + name);
      for(int i = 0; i < fields.size(); i++){
          Object obj = fields.elementAt(i);
          String className = obj.getClass().getName();
          if(className.equals("tauzaman.field.Field")){
              Field aField = (Field)obj;
              rep.append("\n  " + aField.toString());
          }
          else{
              Fields aFields = (Fields)obj;
              rep.append(aFields.toString());
          }
      }
      return rep.toString();
  }  

}
