package tauzaman.field;

import tauzaman.property.FieldInfo;
import java.net.*;


/**
* <code>Field</code> class represents a single Field. A Field consists of
* two components; <p> <code>FieldInfo</code> and Value. <code>FieldInfo</code>, 
* which is fetched from the content of a <code>Property</code> Specification file,
* contains all the information that is sufficient to represent a Field;
* String name of this Field, String variable name that is used for this Field in
* <code>Format</code> of related <code>Property</code> and URL of <code>FVSupport</code>,
* which will be used calculate value of this Field.
* </p><p> Value is a long and contains Field Value, which is
* fetched from the content of a string temporal constant that is originally passed
* from one of the temporal data types; <code>Instant</code>, <code>Period</code>,
* <code>Interval</code>. </p>
*
* @author  Bedirhan Urgun and Curtis Dyreson
* @version 0.1, 10/10/02
* @see     tauzaman.calendar.Calendar
* @see     tauzaman.property.Property
* @see     tauzaman.temporaldatatypes.Instant
* @see     tauzaman.temporaldatatypes.Period
* @see     tauzaman.temporaldatatypes.Interval
* @status design complete, implementation complete
*/
public class Field implements Cloneable{

  /** <code>FieldInfo</code> of this <code>Field</code> 
     get from related <code>Property</code> **/
  private FieldInfo fieldInfo = null;

  /** A boolean marker, which is set true when a useful information
      found in parsed temporal constant that matches with this <code>Field</code> **/
  private boolean dirty = false;

  /** long value(index) of this <code>Field</code> **/
  private long value;

  /** 
  * Constructs <code>Field</code> object.
  *
  * @param fieldInfo a <code>FieldInfo</code> that includes all Field information
  * for this <code>Field</code>
  */
  public Field(FieldInfo fieldInfo){
      this.fieldInfo = fieldInfo;
  }

  /**
  * Returns cloned version of this <code>Field</code> object.
  * 
  * @return clone of this <code>Field</code> object
  */
  public Object clone(){
      /* form a new Field */
      Field field = new Field(fieldInfo);
      /* other than fieldInfo, which is a reference, 
         other data members should have values as 
         they are initilialized */
      return field;
  }

  /**
  * Returns value of this <code>Field</code>.
  *
  * @return long value 
  */
  public long getValue(){
      return value;
  }

  
  /**
  * Sets value of this <code>Field</code> to a given value.
  *
  * @param long value to be set 
  */
  public void setValue(long value){
      this.value = value;
  }

  /**
  * Returns variable name of this <code>Field</code>. 
  * This value is fetched from <code>FieldInfo</code> that this
  * <code>Field</code> contains.
  *
  * @param String variable name 
  */
  public String getVariableName(){
     return fieldInfo.getVariableName();
  }

  /**
  * Returns name of this <code>Field</code>. 
  * This value is fetched from <code>FieldInfo</code> that this
  * <code>Field</code> contains.
  *
  * @param String name 
  */  
  public String getName(){
     return fieldInfo.getFieldInfoName();
  }

  /**
  * Returns URL of <code>FVSupport</code> of this <code>Field</code>. 
  * This value is fetched from <code>FieldInfo</code> that this
  * <code>Field</code> contains.
  *
  * @param URL of <code>FVSupport</code> 
  */  
  public URL getUrl(){
     return fieldInfo.getFVSUrl();
  }

  /**
  * Return true if this <code>Field</code> is marked. False o.w.
  *
  * @return boolean true if this <code>Field</code> is marked. False o.w.
  */ 
  public boolean isDirty(){
    return dirty;
  }

  /**
  * Sets this <code>Field</code> as being used.
  */
  public void setAsDirty(){
     dirty = true;
  }

  /**
  * Returns String representation of this object.
  *
  * @return A String representing this object
  */
  public String toString(){
      String rep = null;

      if(isDirty())
          rep = "Field: " + getVariableName() + " " + getName() + " " + getValue() + " dirty";
      else
          rep = "Field: " + getVariableName() + " " + getName() + " " + getValue() + " clean";

     return rep;
  }
  
}
