package tauzaman.calendricsystem.granularitylattice.minpriorityqueue;

/**
  * <code>MinPriorityQueue</code> is a generic data structure to implement a
  *   minimum priority queue.  The implementation is done with a binary heap
  *   backed by an array.
  *
  * <p>Title: Minimum Priority Queue </p>
  * <p>Description: Generic data structure to implement a minimum priority
  *                 queue </p>
  * <p>Copyright: Copyright (c) 2003</p>
  * @author Jessica Miller
  * @version 1.0
  * @status implementation complete
  **/

import java.util.*;

public class MinPriorityQueue {

  /**
    * This is the starting size of the minimum priority queue's array.
    */
  private static final int DEFAULT_SIZE = 10;

  /**
    * This is the index of the root.  It is 1 because the arithmetic for the
    *   queue instructions is much cleaner if the root is at index 1.  This
    *   does have the downside of one wasted element (that element at index 0 in
    *   the array).
    */
  private static final int ROOT_INDEX   = 1;

  /**
    * The array that contains the queue's elements.
    */
  private Object[] qItems;

  /**
    * This hashtable maps each element's value to the index in the qItems array
    *   in which it resides.  This is needed to maintain quick access to each
    *   element when you want to decrease it's priority.  Otherwise, users would
    *   have to keep track of different elements' priorities in order to make a
    *   change in the element's priority.
    */
  private Hashtable valueToIndexTable;

  /**
    * The current size of the queue.
    */
  private int qSize;

  /**
    * Constructs a minimum priority queue by initializing its array to the
    *   default size and intializing the value -> index hashtable.
    */
  public MinPriorityQueue()
  {
    qItems            = new Object[DEFAULT_SIZE];
    valueToIndexTable = new Hashtable();
    qSize  = 0;  // since we don't use first element
  }

  /**
    * Inserts a new element into the priority queue.  First it inserts the
    *   element at the end of the queue (after possibly adjusting the size of
    *   the queue such that the new element will fit).  Second, it bubbles the
    *   element up the heap until it is smaller than its parent.
    *
    * @param priority  The priority of the new element.
    * @param value     The value of the new element.
    */
  public void enqueue(Comparable priority, Object value)
  {
    ensureCapacity(qSize + 1);
    MPQElement newItem = new MPQElement(priority, value);

    // insert item
    int newItemIndex;

    for (newItemIndex = ++qSize;
         newItemIndex > ROOT_INDEX  &&  priority.compareTo(((MPQElement)qItems[newItemIndex/2]).getPriority()) < 0;
         newItemIndex /= 2)
    {
      qItems[newItemIndex] = qItems[newItemIndex / 2];
      valueToIndexTable.put(((MPQElement)qItems[newItemIndex]).getValue(),
                            new Integer(newItemIndex));
    }

    qItems[newItemIndex] = newItem;
    valueToIndexTable.put(value, new Integer(newItemIndex));
  }

  /**
    * Deletes and returns the element at the root (the element with the minimum
    *   priority) of the heap.  First, save the element at the root to return.
    *   Then delete the last element in the heap and give its value to be the
    *   new root.  Lastly, bubble this element down until its children are both
    *   less than this element or until this element is at a leaf.
    *
    * @return The element with the minimum priority (or null if the queue is
    *           empty.
    */
  public MPQElement dequeue()
  {
    if (qSize == 0)
      return null;

    // save copy of element at the root node
    MPQElement result = (MPQElement)qItems[ROOT_INDEX];

    // move last node, node n, in heap to root and delete node n
    qItems[ROOT_INDEX] = qItems[qSize];
    qItems[qSize] = null;

    // decrement size of queue and if the queue is now empty, return the result
    if (--qSize == 0)
      return result;

    // otherwise, do element shifting, if needed
    int index             = ROOT_INDEX;
    MPQElement leftChild  = (ROOT_INDEX * 2 <= qSize)
                            ? (MPQElement)qItems[ROOT_INDEX * 2] : null;
    MPQElement rightChild = (ROOT_INDEX * 2 + 1 <= qSize)
                            ? (MPQElement)qItems[ROOT_INDEX * 2 + 1] : null;
    Comparable priority   = ((MPQElement)qItems[ROOT_INDEX]).getPriority();
    MPQElement element    = (MPQElement)qItems[ROOT_INDEX];

    // if new root <= its children, heap exists
    // otherwise, percolate this key down by exchanging it with its smallest
    //   child; repeat until key no longer needs to be moved or is in a leaf
    while (((leftChild != null) && (priority.compareTo(leftChild.getPriority()) > 0))
           || ((rightChild != null) && (priority.compareTo(rightChild.getPriority()) > 0)))
    {
      int childIndex;

      if (leftChild == null)
        childIndex = index * 2 + 1;
      else if (rightChild == null)
        childIndex = index * 2;
      else
        childIndex = (leftChild.getPriority().compareTo(rightChild.getPriority()) <= 0)
                       ? index * 2
                       : index * 2 + 1;

      qItems[index]      = qItems[childIndex];
      valueToIndexTable.put(((MPQElement)qItems[index]).getValue(),
                            new Integer(index));

      index      = childIndex;
      leftChild  = (index * 2 < qSize)
                   ? (MPQElement) qItems[index * 2] : null;
      rightChild = (index * 2 + 1 < qSize)
                   ? (MPQElement)qItems[index * 2 + 1] : null;
    }

    qItems[index] = element;
    valueToIndexTable.put(element.getValue(), new Integer(index));

    return result;
  }

  /**
    * Given a particular element already in the queue, decrease its priority to
    *   the given priority.
    *
    * @param value        The element whose priority we want to decrease.
    * @param newPriority  The new priority for the element.
    * @return             <code>true</code> if the priority was changed,
    *                       <code>false</code> if the priority given was greater
    *                       than the original priority or if the element is not
    *                       in the queue
    */
  public boolean decreasePriority(Object value, Comparable newPriority)
  {
    Object indexOfVal = valueToIndexTable.get(value);
    if (indexOfVal == null)
      return false;

    int valIndex = ((Integer)indexOfVal).intValue();
    MPQElement valElement = (MPQElement)qItems[valIndex];

    if (newPriority.compareTo(valElement.getPriority()) >= 0)
      return false;

    for (;
         valIndex > ROOT_INDEX  &&  newPriority.compareTo(((MPQElement)qItems[valIndex/2]).getPriority()) < 0;
         valIndex /= 2)
    {
      qItems[valIndex] = qItems[valIndex / 2];
      valueToIndexTable.put(((MPQElement)qItems[valIndex/2]).getValue(),
                            new Integer(valIndex));
    }

    valElement.setPriority(newPriority);
    qItems[valIndex] = valElement;
    valueToIndexTable.put(value, new Integer(valIndex));

    return true;
  }

  /**
    * Helper method that ensures the capacity of the queue by checking to see
    *   that the given minimum capacity can be supported in the current size
    *   of the queue.  If the given minimum capcity is greater than the number
    *   of elements the queue can hold, the capacity of the queue is increased
    *   by 50%.
    *
    * @param minCapacity  The minimum number of elements that the queue should
    *                       be able to support.
    */
  private void ensureCapacity(int minCapacity)
  {
    int oldCapacity = qItems.length;

    if (++minCapacity > oldCapacity)
    {
      Object oldData[] = qItems;
      int newCapacity = (oldCapacity * 3) / 2 + 1;
      if (newCapacity < minCapacity)
        newCapacity = minCapacity;
      qItems = new Object[newCapacity];
      System.arraycopy(oldData, 0, qItems, 0, qSize + 1);
    }
  }

  /**
    * Returns the number of elements currently in the queue.
    *
    * @return  The number of elements currently in the queue.
    */
  public int getSize()
  {
    return qSize;
  }

  /**
    * Returns whether or not the queue is empty.
    *
    * @return  <code>true</code> if the queue is empty, <code>false</code> if
    *            the queue has elements
    */
  public boolean isEmpty()
  {
    return qSize == 0;
  }

  /**
    * Returns a string representation of the minimum priority queue.
    *
    * @return  a string representation of the minimum priority queue
    */
  public String toString()
  {
    int index      = 1;
    int level      = 1;
    int elsAtLevel = 1;
    String res = "";

    while (index <= qSize)
    {
      res += "LEVEL " + level + "\n   ";
      for (int ii = 0; ii < elsAtLevel  &&  index <= qSize; ii++)
      {
        res += qItems[index].toString() + "\n   ";
        index++;
      }
      res += "\n";
      elsAtLevel = level * 2;
      level++;
    }

    return res;
  }

  // Simple tests for the minimum priority queue
  public static void main(String[] args)
  {
    MinPriorityQueue mpq = new MinPriorityQueue();

    mpq.enqueue(new Integer(225), "Jessica");
    mpq.enqueue(new Integer(2147483647), "Martin");
    mpq.enqueue(new Integer(225), "Meghan");
    mpq.enqueue(new Integer(2147483647), "Laura");

    System.out.println("Before dequeue: " + mpq);

    System.out.println("After dequeued " + mpq.dequeue().getValue());

    System.out.println(mpq);


/*
    mpq.enqueue(new Integer(13), "Jessica");
    mpq.enqueue(new Integer(10), "Martin");
    mpq.enqueue(new Integer(8), "Meghan");
    mpq.enqueue(new Integer(14), "Laura");
    mpq.enqueue(new Integer(15), "John");
    mpq.enqueue(new Integer(9), "Ursula");
    mpq.enqueue(new Integer(4), "Mary");
    mpq.enqueue(new Integer(12), "Brandon");
    mpq.enqueue(new Integer(8), "Chris");
    mpq.enqueue(new Integer(1), "Shannon");

    System.out.println(mpq);

    System.out.print("MinPriorityQueue: [");

    for (int ii = 1;  ii < mpq.getSize(); ii++)
      System.out.print(((MPQElement)mpq.qItems[ii]).getPriority() + ", ");

    System.out.println(((MPQElement)mpq.qItems[mpq.getSize()]).getPriority() + "]");

    mpq.decreasePriority("John", new Integer(3));

    System.out.print("MinPriorityQueue (after decreasing 15's priority to 3): [");

    for (int ii = 1;  ii < mpq.getSize(); ii++)
      System.out.print(((MPQElement)mpq.qItems[ii]).getPriority() + ", ");

    System.out.println(((MPQElement)mpq.qItems[mpq.getSize()]).getPriority() + "]");

    while (mpq.getSize() != 0)
    {
      System.out.println("Element dequeued" + mpq.dequeue());

      System.out.print("MinPriorityQueue (after dequeueing first element): [");

      for (int ii = 1; ii < mpq.getSize(); ii++)
        System.out.print( ( (MPQElement) mpq.qItems[ii]).getPriority() + ", ");

      if (mpq.getSize() != 0)
        System.out.print( ( (MPQElement) mpq.qItems[mpq.getSize()]).getPriority());

      System.out.println("]");
    }
*/
  }
}