package tauzaman.calendricsystem.granularitylattice;

/**
  * <code>GranularityLattice</code> class contains the data structure that
  *   models the lattice of a calendric system within tauZaman.  Conceptually,
  *   the lattice can be thought of as a two separate graphs combined.  The
  *   lattice combines both coarser relationships (edges) between nodes and also
  *   finer relationships (edges) between nodes.  The lattice also has congruent
  *   relationships between nodes, but for shortest path calculation purposes,
  *   the edges are also put into the coarser and finer lists of a node.  The
  *   label of the edge is the mapping that can be used to convert between one
  *   granularity and another.
  * <p>
  * The shortest path algorithm used to find converstion paths between different
  *   granularities is Djikstra's shortest path.  The particular version of this
  *   algorithm that I used can be found in the Introduction to Algorthms book.
  * <p>
  * The lattice will be used exclusively by the CalendricSystem in order for
  *   the calendric system to perform its conversions between granularities.
  *
  * @author  Jessica Miller
  * @version 0.1, March 2003
  * @see tauzaman.calendar.Calendar
  * @see tauzaman.calendricsystem.CalendricSystem
  * @see tauzaman.calendricsystem.granularitylattice.cache.Cache
  * @see tauzaman.calendricsystem.granularitylattice.minpriorityqueue.MinPriorityQueue
  * @status implementation completed
  */

import java.net.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import java.util.Vector;
import org.w3c.dom.*;

import tauzaman.calendar.mapping.Mapping;
import tauzaman.calendar.Calendar;
import tauzaman.calendar.RegularMapping;
import tauzaman.calendricsystem.*;
import tauzaman.calendricsystem.granularitylattice.cache.*;
import tauzaman.calendricsystem.granularitylattice.minpriorityqueue.*;
import tauzaman.timestamp.Granule;

public class GranularityLattice
{
  private static final int COARSER    = 1;
  private static final int FINER      = 2;
  private static final int CONGRUENT  = 3;
  private static final int SPACES     = 2;
  private static final int CACHE_SIZE = 10;
  private static final boolean BUILDPATH_DEBUG = false;
  public  static boolean CACHE_DEBUG           = false;
  private static final boolean SP_DEBUG        = false;

  /**
    * This hashtable will contain a reference to all nodes in the granularity
    *   lattice by mapping Granularity -> LatticeNode.
    */
  private HashMap nodeTable;                  // [Granularity --> LatticeNode]

  /**
    * Cache that maps particular granularities to an already computed table of
    *   shortest path information to finer granularities.
    */
  private Cache finerSpTableCache   = null;   // [Granularity --> HashMap of shortest path info.]

  /**
    * Cache that maps particular granularities to an already computed table of
    *   shortest path information to coarser granularities.
    */
  private Cache coarserSpTableCache = null;   // [Granularity --> HashMap of shortest path info.]

  /**
    * Cache that maps a string representing the source granularity to the
    *   destination granularity ("sourceGranStr$destGranStr") to mapping paths.
    */
  private Cache mappingCache        = null;   // [sourceGranStr$destGranStr --> mapping paths]

  /**
    * List of all the nodes in the lattice.
    */
  private ArrayList nodeList;

  /**
    * Constructs a <code>GranularityLattice</code> object.  The construction is
    *   done by:
    *
    *   (1) For each calendar in calendars, get the calendar's mappings and set
    *       up <code>LatticeNodes</code> for each granularity encountered and
    *       connecting the nodes with the mappings themselves.
    *
    *   (2) Connecting the granularities in different calendars using the
    *       mappings in the inter-calendar mappings array.
    *
    * @param calendars              the array of all Calendar objects that are
    *                               in this calendric system
    * @param interCalendarMappings  the array of mappings that are mappings
    *                               between calendars in the calendric system
    *
    * @throws CalendricSystemFormationException if any error occurs while
    *         forming <code>GranularityLattice</code>
    *
    */
  public GranularityLattice(Calendar[] calendars, Mapping[] interCalMappings)
      throws CalendricSystemFormationException
  {
    nodeTable             = new HashMap();
    nodeList              = new ArrayList();

    for (int ii = 0; ii < calendars.length; ii++)
      addCalendar(calendars[ii]);

    for (int ii = 0; ii < interCalMappings.length; ii++)
      addMapping(interCalMappings[ii]);

    induceMappings();

    finerSpTableCache        = new Cache(CACHE_SIZE);
    coarserSpTableCache      = new Cache(CACHE_SIZE);
    mappingCache             = new Cache(CACHE_SIZE);
  }


  /**
    * Adds an entire calendar to the lattice by going through each mapping in
    *   the calendar, adding any nodes that are not already in the lattice plus
    *   the edges connecting the nodes in each mapping.
    *
    * @param calendar  <code>Calendar</code> to be added to the lattice
    */
  public void addCalendar(tauzaman.calendar.Calendar calendar)
      throws CalendricSystemFormationException
  {
    Mapping[] mappings = calendar.getMappings();

    for (int ii = 0; ii < mappings.length; ii++)
      addMapping(mappings[ii]);
  }


  /**
    * Creates an edge between the "from" granularity and the "to" granularity
    *   in the <code>Mapping</code> object with the <code>Mapping</code> object
    *   itself as the edge's label.
    *
    * @param mapping  <code>Mapping</code> that is the edge's label
    *
    * @throws NoSuchGranularityException if either of the Granularities in the
    *         mapping are not recognized by the lattice (i.e., there is no node
    *         in the lattice that has this granularity)
    */
  public void addMapping(Mapping mapping)
      throws CalendricSystemFormationException
  {
    Granularity fromGran;
    Granularity toGran;
    LatticeEdge edge;
    LatticeNode fromNode;
    LatticeNode toNode;

    fromGran = mapping.getFrom();
    toGran = mapping.getTo();

    // if a mapping is added (esp. when lattice already constructed), destroy
    //   caches
    finerSpTableCache   = null;
    coarserSpTableCache = null;
    mappingCache        = null;

    // get fromNode (or create it if does not already exist)
    if (nodeTable.containsKey(fromGran))
      fromNode = (LatticeNode) nodeTable.get(fromGran);
    else
    {
      fromNode = new LatticeNode(fromGran);
      nodeList.add(0, fromNode);
      nodeTable.put(fromGran, fromNode);
    }

    // get toNode (or create it if does not already exist)
    if (nodeTable.containsKey(toGran))
      toNode = (LatticeNode) nodeTable.get(toGran);
    else
    {
      toNode = new LatticeNode(toGran);
      nodeList.add(0, toNode);
      nodeTable.put(toGran, toNode);
    }

    // create edge between the two nodes
    edge = new LatticeEdge(toNode, mapping);

    if (mapping.getRelationship() == Mapping.FINER_TO_COARSER)
      fromNode.coarserEdges.put(toGran, edge);
    else if (mapping.getRelationship() == Mapping.COARSER_TO_FINER)
      fromNode.finerEdges.put(toGran, edge);
    else if (mapping.getRelationship() == Mapping.CONGRUENT)
    {
      fromNode.congruentEdges.put(toGran, edge);
      fromNode.coarserEdges.put(toGran, edge);
      fromNode.finerEdges.put(toGran, edge);
    }

    else
      throw new CalendricSystemFormationException("Unknown relationship in mapping.");
  }


  /**
    * Uses the lattice to cast the given anchored granule from its granularity
    *   to another granularity.
    *
    * @param granule        the granule to cast from its granularity to
    *                       another granularity
    * @param toGranularity  the granularity to which to cast the given
    *                       granule
    *
    * @return               a Granule that represent the given granule
    *                       converted to the given granularity
    *
    * @throws CalendricSystemServiceException if either the Granularity of the
    *         given granule or the toGranularity are not recognized by the
    *         lattice (i.e., there is no node in the lattice that has this
    *         granularity)
    */
  public Granule castAnchored(Granule granule, Granularity toGranularity)
      throws CalendricSystemServiceException
  {
    Granularity fromGranularity = granule.getGranularity();
    long granuleValue           = granule.getGranule().getValue();

    // get the path of mappings to convert between the two granularities
    ArrayList path = findPath(fromGranularity, toGranularity);

//    assert(path != null);

    // iterate through the path, calculating the new granule's value
    Iterator i = path.iterator();

    while (i.hasNext())
    {
      Mapping thisMapping = (Mapping)i.next();
      granuleValue = thisMapping.performAnchoredCast(granuleValue);
    }

    return new Granule(toGranularity, granuleValue);
  }


  /**
    * Uses the lattice to cast the given unanchored granule from its granularity
    *   to another granularity.
    *
    * @param granule        the granule to cast from its granularity to
    *                       another granularity
    * @param toGranularity  the granularity to which to cast the given
    *                       granule
    *
    * @return               a Granule that represent the given granule
    *                       converted to the given granularity
    *
    * @throws CalendricSystemServiceException if either the Granularity of the
    *         given granule or the toGranularity are not recognized by the
    *         lattice (i.e., there is no node in the lattice that has this
    *         granularity)
    */
  public Granule castUnanchored(Granule granule, Granularity toGranularity)
      throws CalendricSystemServiceException
  {
    Granularity fromGranularity = granule.getGranularity();
    long granuleValue           = granule.getGranule().getValue();

    // get the path of mappings to convert between the two granularities
    ArrayList path = findPath(fromGranularity, toGranularity);

//    assert(path != null);

    // iterate through the path, calculating the new granule's value
    Iterator i = path.iterator();

    while (i.hasNext())
    {
      Mapping thisMapping = (Mapping)i.next();
      granuleValue = thisMapping.performUnanchoredCast(granuleValue);
    }

    return new Granule(toGranularity, granuleValue);
  }


  /**
    * Finds a path (made up of <code>Mapping</code> objects) that can be used to
    *   convert the given source granularity to the destination granularity.
    *   This method works by first finding all shortest paths between the source
    *   granularity and all other granularities.  If after this step, a path
    *   has been found from the source granularity to the destination
    *   granularity, the mapping path is built and returned.  However, if no
    *   shortest path has been found, all shortest paths between the destination
    *   granularity and all other granularities is found.  With these two lists
    *   of shortest paths, the greatest common granularity from which both the
    *   source and destination granularities can be reached is found and a
    *   V-path is built.
    *
    * @param sourceGranularity  the starting granularity
    * @param destGranularity    the destination granularity
    *
    * @return                   the list of mappings that can be used to convert
    *                           from the source granularity to the destination
    *                           granularity
    *
    * @throws CalendricSystemServiceException if either the Granularity of the
    *         given granule or the destGranularity are not recognized by the
    *         lattice (i.e., there is no node in the lattice that has this
    *         granularity)
    */
  private ArrayList findPath(Granularity sourceGranularity,
                             Granularity destGranularity)
      throws CalendricSystemServiceException
  {
    ArrayList mappingPath = null;
    LatticeNode sourceNode = (LatticeNode)nodeTable.get(sourceGranularity);
    LatticeNode destNode   = (LatticeNode)nodeTable.get(destGranularity);

    if (sourceNode == null)
      throw new CalendricSystemServiceException("Cannot do operation from non-existent granularity: "
                                                + sourceGranularity.getLocalName());
    if (destNode == null)
      throw new CalendricSystemServiceException("Cannot do operation to non-existent granularity: "
                                                + destGranularity.getLocalName());

    // first check mapping cache for the path
    String mapKey = sourceGranularity.toString() + "$"
                    + destGranularity.toString();
    if (mappingCache.get(mapKey) != null)
    {
      if (CACHE_DEBUG)
        System.out.println("Mapping cache hit: "
                           + sourceGranularity.getLocalName() + " --> "
                           + destGranularity.getLocalName());
      return (ArrayList) mappingCache.get(mapKey);
    }

    // get shortest path info. for finer relationships starting at source
    HashMap sourceSpTable = getSpTable(sourceNode, FINER);

    if (SP_DEBUG)
    {
      System.out.println("-- Shortest path info for " + sourceGranularity.getLocalName()
                         + ": --");
      printSpTable(sourceSpTable);
    }

    // if a straight path has been found between the source and dest.
    //   granularities, generate the path from this
    if (((SPElement)sourceSpTable.get(destNode)).predecessor != null)
      mappingPath = buildMappingPath(sourceSpTable, sourceNode, destNode, FINER);
    else
    {
      // otherwise, generate the path by performing by the V-path between the two
      //   granularities

      // (1) get minimum priority queue containing all reachable nodes from the
      //     destination, in order of closest to furthest (successors of dest.)
      MinPriorityQueue destSuccessors = getSuccessors(destNode, FINER);

      // (2) iterate through all successors of the dest to see if source node
      //     has a predecessor that can get to any dest. successor
      LatticeNode bottomNode = destNode;

      while (((SPElement)sourceSpTable.get(bottomNode)).predecessor == null
             &&  bottomNode != sourceNode)
      {
        if (destSuccessors.isEmpty())
        {
          // then we are out of successors and there is no possible V-path
          return null;
        }
        bottomNode = (LatticeNode) destSuccessors.dequeue().getValue();
      }

      // build v-path between the source node and the "nextSource" (finer) +
      //   path between "bottomNode" and destination (coarser - reversed)
      mappingPath = buildMappingPath(sourceSpTable, sourceNode,
                                     bottomNode, FINER);

      HashMap meetingPtPreds = getSpTable(bottomNode, COARSER);

      if (SP_DEBUG)
      {
        System.out.println("-- Coarser shortest path info. for " +
                           bottomNode.granularity.getLocalName() + ": --");
        printSpTable(meetingPtPreds);
      }

      mappingPath.addAll(buildMappingPath(meetingPtPreds, bottomNode, destNode,
                                          COARSER));
    }

    mappingCache.insert(mapKey, mappingPath);

    if (SP_DEBUG)  System.out.println("Mapping path = " + mappingPath);
    return mappingPath;
  }


  /**
    * Returns a table of shortest path information for the given node and type
    *   (i.e. coarser/finer).  First the caches are checked for a match then if
    *   no match is found in the cache Djikstra's algorithm is used to calculate
    *   the shortest path information.
    *
    * @param s            the node for which to get shortest path information
    * @param typeOfEdges  the type of edges (FINER/COARSER/CONGRUENT) to use to
    *                     get the shortest path information
    *
    * @return             a table that maps granularities -> (predecessors,
    *                     cost to reach source)
    */
  private HashMap getSpTable(LatticeNode s, int typeOfEdges)
  {
    Cache spTableCache = (typeOfEdges == FINER)
                          ? finerSpTableCache
                          : coarserSpTableCache;

    if (spTableCache.get(s.granularity) != null)
    {
      if (CACHE_DEBUG)
      {
        String debugStr = "Cache hit: "
            + ((typeOfEdges == FINER) ? "finer " : "coarser ") + "preds of " +
            s.granularity.getLocalName();

        System.out.println(debugStr);
      }

      return (HashMap) spTableCache.get(s.granularity);
    }
    else
    {
      if (CACHE_DEBUG)
      {
        String debugStr = "Cache miss: "
            + ((typeOfEdges == FINER) ? "finer " : "coarser ") + "preds of " +
            s.granularity.getLocalName();

        System.out.println(debugStr);
      }

      HashMap preds = dijkstraShortestPaths(s, typeOfEdges);
      spTableCache.insert(s.granularity, preds);
      return preds;
    }
  }


  /**
    * Returns a minimum priority queue of nodes that are reachable by the given
    *   node through the given type of edges (i.e. coarser/finer).  The minimum
    *   priority queue is organized according to closest --> furthest nodes away
    *   from the given node.  This serves as a "sorted" list of successors from
    *   the given node.
    *
    * @param node         the node for which to get successor information
    * @param typeOfEdges  the type of edges (FINER/COARSER/CONGRUENT) to use to
    *                     get the successor information
    *
    * @return             minimum priority queue that contains all reachable
    *                     nodes from closest --> furthest
    */
  private MinPriorityQueue getSuccessors(LatticeNode node, int typeOfEdges)
  {
    MinPriorityQueue successors = new MinPriorityQueue();

    HashMap preds         = getSpTable(node, typeOfEdges);
    Set predKeys          = preds.keySet();
    Iterator itr          = predKeys.iterator();

    while (itr.hasNext())
    {
      LatticeNode key = (LatticeNode)itr.next();
      SPElement spe   = (SPElement)preds.get(key);
      if (spe.predecessor != null)
        successors.enqueue(spe.estimate, key);
    }

    return successors;
  }


  /**
    * Helper method used to debug findPath by printing out predecessor lists.
    *
    * @param preds  the hash of predecessors for a particular granularity
    */
  private void printSpTable(HashMap preds)
  {
    Iterator fromItr = preds.keySet().iterator();

    while (fromItr.hasNext())
    {
      LatticeNode nextKey = (LatticeNode)fromItr.next();

      System.out.print("LN(" + nextKey.granularity.getLocalName() + ")");
      System.out.print(" --> ");

      SPElement nextValue = (SPElement)preds.get(nextKey);

      if (nextValue.predecessor == null)
        System.out.println("null");
      else
        System.out.println("SPE("
                           + ((LatticeNode)nextValue.predecessor).granularity.getLocalName()
                           + "; " + nextValue.estimate + ")");
    }
    System.out.println("-----------------------------------------------------");
  }


  /**
    * Uses the lattice to determine if one granularity is coarser than another.
    *
    * @param g1  the first granule to be compared
    * @param g2  the second granule to be compared
    *
    * @return <code>true</code> if g1 is coarser than g2, <code>false</code> if
    *         g1 is finer than g2 or if g1 and g2 cannot be compared
    *
    * @throws NoSuchGranularityException if either of the granularities passed
    *         in are not recognized by the lattice (i.e., there is no node in
    *         the lattice that has this granularity)
    */
  public boolean isCoarser(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    LatticeNode n1 = (LatticeNode)nodeTable.get(g1);
    LatticeNode n2 = (LatticeNode)nodeTable.get(g2);

    if (n1 == null)
      throw new CalendricSystemServiceException("Cannot do operation from non-existent granularity: "
                                                + g1.getLocalName());
    if (n2 == null)
      throw new CalendricSystemServiceException("Cannot do operation to non-existent granularity: "
                                                + g2.getLocalName());
    HashMap n2SpTable = getSpTable(n2, COARSER);

    return ((SPElement)n2SpTable.get(n1)).predecessor != null;
  }


  /**
    * Uses the lattice to determine if the two granularities are equivalent.
    *
    * @param g1  the first granule to be compared for equivalence
    * @param g2  the second granule to be compared for equivalence
    *
    * @return <code>true</code> if g1 is equivalent to g2, <code>false</code>
    *         if g1 is not equivalent to g2
    *
    * @throws NoSuchGranularityException if either of the granularities passed
    *         in are not recognized by the lattice (i.e., there is no node in
    *         the lattice that has this granularity)
    */
  public boolean isEquivalent(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    LatticeNode n1 = (LatticeNode)nodeTable.get(g1);
    LatticeNode n2 = (LatticeNode)nodeTable.get(g2);

    if (n1 == null)
      throw new CalendricSystemServiceException("Cannot do operation from non-existent granularity: "
                                                + g1.getLocalName());
    if (n2 == null)
      throw new CalendricSystemServiceException("Cannot do operation to non-existent granularity: "
                                                + g2.getLocalName());

    return n1.congruentEdges.get(g2) != null;
  }


  /**
    * Uses the lattice to determine if the two granularities are equivalent.
    *
    * @param g1  the first granule to be compared for equivalence
    * @param g2  the second granule to be compared for equivalence
    *
    * @return <code>true</code> if g1 is equivalent to g2, <code>false</code>
    *         if g1 is not equivalent to g2
    *
    * @throws NoSuchGranularityException if either of the granularities passed
    *         in are not recognized by the lattice (i.e., there is no node in
    *         the lattice that has this granularity)
    */
  public boolean isIncomparable(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    return !isCoarser(g1, g2)  &&  !isEquivalent(g1, g2)  &&  !isFiner(g1, g2);
  }


  /**
    * Uses the lattice to determine if one granularity is finer than another.
    *
    * @param g1  the first granule to be compared
    * @param g2  the second granule to be compared
    *
    * @return <code>true</code> if g1 is finer than g2, <code>false</code> if
    *         g1 is coarser than g2 or if g1 and g2 cannot be compared
    *
    * @throws NoSuchGranularityException if either of the granularities passed
    *         in are not recognized by the lattice (i.e., there is no node in
    *         the lattice that has this granularity)
    */
  public boolean isFiner(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    LatticeNode n1 = (LatticeNode)nodeTable.get(g1);
    LatticeNode n2 = (LatticeNode)nodeTable.get(g2);

    if (n1 == null)
      throw new CalendricSystemServiceException("Cannot do operation from non-existent granularity: "
                                                + g1.getLocalName());
    if (n2 == null)
      throw new CalendricSystemServiceException("Cannot do operation to non-existent granularity: "
                                                + g2.getLocalName());
    HashMap n2SpTable = getSpTable(n2, FINER);

    return ((SPElement)n2SpTable.get(n1)).predecessor != null;
  }


  /**
    * Resets all of the lattice's caches.
    */
  public void clearCaches()
  {
    finerSpTableCache        = new Cache(CACHE_SIZE);
    coarserSpTableCache      = new Cache(CACHE_SIZE);
    mappingCache             = new Cache(CACHE_SIZE);
  }


  /**
    * Returns a string representation of the lattice.
    *
    * @return a String representation of the lattice.
    */
  public String toString()
  {
    String indent1 = "";
    String nodeStr = "\n";

    for (int ii = 0; ii < SPACES; ii++)
        indent1 += " ";

    Vector nodeList = new Vector(nodeTable.values());
    Iterator i = nodeList.iterator();
    while (i.hasNext())
    {
      nodeStr += indent1 + i.next() + "\n";
    }

    return "Lattice:" + nodeStr + "\n";
  }


/* --------------------------------------------------------------------------
   INDUCE MAPPINGS WITH DFS
   -------------------------------------------------------------------------- */
 /**
   * Constant used in DFS part of induce mappings to indicate a node has not yet
   *   been explored.
   */
  private static final int WHITE = 0;

 /**
   * Constant used in DFS part of induce mappings to indicate a node is
   *   currently being explored.
   */
  private static final int GRAY = 1;

  /**
    * Constant used in DFS part of induce mappings to indicate a node has
    *   been fully explored.
    */
  private static final int BLACK = 2;


  /**
   * Induces all regular mappings in the sets of coarser, finer and congruent
   *   edges in the lattice.
   */
  private void induceMappings()
  {
    induceMappings(FINER);
    induceMappings(COARSER);
    induceMappings(CONGRUENT);
  }


  /**
   * Performs a DFS search on each node looking for regular mappings that can
   *   be induced.
   *
   * @param typeOfEdges  the type of edges (FINER/COARSER/CONGRUENT) from
   *                     which the regular mappings should be induced
   */
  private void induceMappings(int typeOfEdges)
  {
    HashMap colorTable = new HashMap(nodeList.size());

    // for each vertex u in V[G]
    for (int ii = 0; ii < nodeList.size(); ii++)
    {
      colorTable.clear();

      // for each vertex u in V[G]
      for (int jj = 0; jj < nodeList.size(); jj++)
      {
        // do color[u] <-- WHITE
        colorTable.put(nodeList.get(jj), new Integer(WHITE));
      }

      // then DFS-Visit(u)
      induceMappingsDFSVisit( (LatticeNode) nodeList.get(ii), null,
                              (LatticeNode) nodeList.get(ii), typeOfEdges,
                              colorTable);
    }
  }


  /**
    * Recursive helper to the DFS part of the mapping induction.  Visits all
    *   reachable nodes from the given node.
    *
    * @param typeOfEdges  the relationship we want to do the sort on (FINER/
    *                       COARSER/CONGRUENT)
    */
  private void induceMappingsDFSVisit(LatticeNode sourceNode,
                                      LatticeEdge sourceEdge,
                                      LatticeNode curNode, int typeOfEdges,
                                      HashMap colorTable)
  {
    // color[curNode] <-- GRAY
    colorTable.put(curNode, new Integer(GRAY));

    // for each nextNode in Adj[curNode], explore edge (curNode, nextNode)
    Vector adjEdgesToCurNode = (typeOfEdges == COARSER)
                               ? new Vector(curNode.coarserEdges.values())
                               : (typeOfEdges == FINER)
                                 ? new Vector(curNode.finerEdges.values())
                                 : new Vector(curNode.congruentEdges.values());

    Iterator itr = adjEdgesToCurNode.iterator();
    while (itr.hasNext())
    {
      LatticeEdge e        = (LatticeEdge)itr.next();
      LatticeEdge nse      = e;
      LatticeNode nextNode = e.node;

      // if this edge has a regular mapping and its period and group size are
      //   equal
      if (e.mapping instanceof RegularMapping
          && ((RegularMapping)e.mapping).getPeriodSize()
               == ((RegularMapping)e.mapping).getPeriodSize())
      {
        // if the given source edge is not equal to null and the source node
        //   doesn't already have a mapping between itself and the destNode of
        //   the edge, we can make a new edge with the source edge and this edge
        Hashtable sourceEdges = (typeOfEdges == COARSER)
                                ? sourceNode.coarserEdges
                                : (typeOfEdges == FINER)
                                  ? sourceNode.finerEdges
                                  : sourceNode.congruentEdges;

        if (sourceEdge != null  &&  sourceEdges.get(e.node.granularity) == null)
        {
          // anchor of new edge = period of this edge * anchor of source edge
          //                      + anchor of this edge
          int anchor = ((RegularMapping)e.mapping).getPeriodSize()
                       * ((RegularMapping)sourceEdge.mapping).getAnchor()
                       + ((RegularMapping)e.mapping).getAnchor();

          // period of new edge = period of this edge * period of source edge
          int period = ((RegularMapping)e.mapping).getPeriodSize()
                       * ((RegularMapping)sourceEdge.mapping).getPeriodSize();

          int relationship = (typeOfEdges == COARSER)
                             ? Mapping.FINER_TO_COARSER
                             : (typeOfEdges == FINER)
                               ? Mapping.COARSER_TO_FINER
                               : Mapping.CONGRUENT;

          nse = new LatticeEdge(e.node, new RegularMapping(e.node.granularity,
                                sourceNode.granularity, relationship, period,
                                period, anchor), LatticeEdge.INDUCED_WEIGHT);

          sourceEdges.put(e.node.granularity, nse);
        }

        // do if color[nextNode] = WHITE
        if (((Integer)colorTable.get(nextNode)).intValue() == WHITE)
        {
          // DFS-Visit(nextNode)
          induceMappingsDFSVisit(sourceNode, nse, nextNode, typeOfEdges, colorTable);
        }
      }
    }

    // color[curNode] <-- BLACK (blacken curNode; it is finished)
    colorTable.put(curNode, new Integer(BLACK));
  }


/* --------------------------------------------------------------------------
   DIJKSTRA'S SHORTEST PATHS
   -------------------------------------------------------------------------- */

  private static final Integer INFINITY      = new Integer(Integer.MAX_VALUE);


  /**
    * Using Djikstra's algorithm, computes all shortest paths starting from
    *   a given lattice node.
    *
    * @param s            the node from which we want all shortest paths in the
    *                     lattice
    * @param typeOfEdges  the relationship from which we want to find the
    *                     shortest path on (FINER/COARSER/CONGRUENT)
    *
    * @return             a table containing all shortest path information in
    *                     the form of [LatticeNode --> (predecessor, cost to
    *                     source)]
    */
  private HashMap dijkstraShortestPaths(LatticeNode s, int typeOfEdges)
  {
    HashMap spTable = new HashMap(nodeList.size());

    // Initialize single source
    initializeSingleSource(s, spTable);

    // Q <-- V[G]
    MinPriorityQueue Q = new MinPriorityQueue();
    for (int ii = 0;  ii < nodeList.size();  ii++)
      Q.enqueue(INFINITY, nodeList.get(ii));

    // change the sources priority to 0, since it is 0 away from itself
    Q.decreasePriority(s, new Integer(0));

    // while Q != empty
    while (!Q.isEmpty())
    {
      // do u <-- ExtractMin(Q)
      LatticeNode u = (LatticeNode)Q.dequeue().getValue();

        // for each vertex, v, in adj[u]
      Vector adjEdgesToU = (typeOfEdges == COARSER) ? new Vector(u.coarserEdges.values())
                             : new Vector(u.finerEdges.values());

      for (int ii = 0;  ii < adjEdgesToU.size();  ii++)
      {
        LatticeEdge edge = (LatticeEdge)adjEdgesToU.get(ii);
        LatticeNode v    = edge.node;

          // do relax(u, v, w)
        relax(u, v, edge.weight, spTable, Q);
      }
    }

    return spTable;
  }


  /**
    * Initializes the data structures used in Djiksta's algorithm.  The shortest
    *   path estimation table begins with all the estimates being infinity.
    *   The predecessor table begins with all entries' predecessors being null.
    *   Then the source node's estimation starts at 0.
    *
    * @param s                     the node from which we want all shortest
    *                              paths in the lattice
    * @param spTable               the table that maps [node --> sp est,
    *                              predecessor in shortest path to source]
    */
  private void initializeSingleSource(LatticeNode s, HashMap spTable)
  {
    // for each vertex, v, in G
    for (int ii = 0; ii < nodeList.size();  ii++)
    {
      // do d[v] <-- INFINITY; predecessor[v] <-- NIL
      spTable.put(nodeList.get(ii), new SPElement(INFINITY, null));
    }

    // d[s] <-- 0
    spTable.put(s, new SPElement(new Integer(0), null));
  }


  /**
    * Tests whether or not we can improve the shortest path to v found so far by
    *   going through u, and, if so, updating the shortest path estimation
    *   table and the predecessor table.
    *
    * @param u                     the new node that we can now use to get a
    *                              shortest path
    * @param v                     the destination node of relaxation process
    * @param w                     the weight of the edge between u and v
    * @param spTable               the table that maps [node --> sp est,
    *                              predecessor in shortest path to source]
    * @param Q                     the minimum priority queue that is used in
    *                              Djiksta's algorithm - needed in this method
    *                              to decrease priority of v if path is improved
    */
  private void relax(LatticeNode u, LatticeNode v, int w, HashMap spTable,
                     MinPriorityQueue Q)
  {
    int uEst = ((SPElement)spTable.get(u)).estimate.intValue();
    int vEst = ((SPElement)spTable.get(v)).estimate.intValue();

    // if d[v] > d[u] + w(u, v)
    if ((uEst != Integer.MAX_VALUE)  &&  (vEst > (uEst + w)))
    {
      // then d[v] <-- d[u] + w(u, v); predecessor[v] <-- u
      spTable.put(v, new SPElement(new Integer(uEst + w), u));
      Q.decreasePriority(v, new Integer(uEst + w));
    }
  }


  /**
    * Given a particular predecessor list, builds a mapping path between two
    *   nodes.
    *
    * @param spTable      the predecessor list composing ests. & shortest paths
    * @param s            the source node
    * @param v            the destination node
    * @param typeOfEdges  the relationship from which we want to build the path
    *                     (FINER/COARSER/CONGRUENT)
    *
    * @return             a list of mappings to follow to convert from s to v
    */
  private ArrayList buildMappingPath(HashMap spTable, LatticeNode s,
                                     LatticeNode v, int typeOfEdges)
  {
    ArrayList nodePath    = new ArrayList();
    ArrayList mappingPath = new ArrayList();
    buildPathHelper(spTable, s, v, nodePath);

    // iterate through the path of nodes to collect the actually path of
    //   that will need to be performed
    Iterator i = nodePath.iterator();
    LatticeNode thisNode = (LatticeNode)i.next();

    while (i.hasNext())
    {
      LatticeNode nextNode = (LatticeNode)i.next();
      if (typeOfEdges == COARSER)
        mappingPath.add(((LatticeEdge)thisNode.coarserEdges.get(nextNode.granularity)).mapping);
      else if (typeOfEdges == FINER)
        mappingPath.add(((LatticeEdge)thisNode.finerEdges.get(nextNode.granularity)).mapping);
      else
        mappingPath.add(((LatticeEdge)thisNode.congruentEdges.get(nextNode.granularity)).mapping);
      thisNode = nextNode;
    }

    return mappingPath;
  }


  /**
   * Recursive helper method used to build the mapping path.
   *
   * @param spTable      the predecessor list composing ests. & shortest paths
   * @param s            the source node
   * @param v            the destination node
   * @param path         the path being built
   */
 private void buildPathHelper(HashMap spTable, LatticeNode s, LatticeNode v,
                               ArrayList path)
  {
    if (BUILDPATH_DEBUG)
    {
      System.out.println("s = " + s);
      System.out.println("v = " + v);
    }

    if (s.equals(v))
      path.add(s);
    else
    {
      buildPathHelper(spTable, s,
                      (LatticeNode)((SPElement) spTable.get(v)).predecessor,
                      path);
      path.add(v);
    }
  }


/* --------------------------------------------------------------------------
   LATTICE NODE
   -------------------------------------------------------------------------- */

  /**
    * <code>LatticeNode</code> class is a private inner class to
    *   <code>GranularityLattice</code>.  This class encapsulates a node's
    *   granularity and its edges.
    *
    * @author  Jessica Miller
    * @version 0.1, March 2003
    * @see GranularityLattice
    * @see GranularityLattice.LatticeEdge
    * @status being implementation complete
    */
  private class LatticeNode {

    /**
      * The granularity in the calendric system that the LatticeNode is
      *   representing.
      */
    public Granularity granularity;

    /**
      * A hash between [destination granularity --> edge to dest. granularity].
      *   Represents the edges that connects this lattice node to other nodes
      *   which represent coarser granularities in the lattice.
      */
    public Hashtable coarserEdges;

    /**
      * A hash between [destination granularity --> edge to dest. granularity].
      *   Represents the edges that connects this lattice node to other nodes
      *   which represent congruent granularities in the lattice.
      */
    public Hashtable congruentEdges;

    /**
      * A hash between [destination granularity --> edge to dest. granularity].
      *   Represents the edges that connects this lattice node to other nodes
      *   which represent finer granularities in the lattice.
      */
     public Hashtable finerEdges;


    /**
      * Construct a <code>LatticeNode</code> by setting the node's granularity
      *   to the given granularity.
      *
      * @param g  the granularity this node is to represent
      */
    public LatticeNode(Granularity g) {
      granularity = g;
      coarserEdges   = new Hashtable();
      congruentEdges = new Hashtable();
      finerEdges     = new Hashtable();
    }


    /**
      * Construct a <code>LatticeNode</code> by setting the node's granularity
      *   to the given granularity and setting the node's edges to the given
      *   lists of edges.
      *
      * @param g          the granularity this node is to represent
      * @param cEdges     the list of edges that lead to granularities coarser
      *                   than this one
      * @param cgntEdges  the list of edges that lead to granularities
      *                   congruent to this one
      * @param fEdges     the list of edges that lead to granularities finer
      *                   than this one
      */
    public LatticeNode(Granularity g, Hashtable cEdges, Hashtable cgntEdges,
                       Hashtable fEdges)
    {
      granularity = g;
      coarserEdges   = cEdges;
      congruentEdges = cgntEdges;
      finerEdges     = fEdges;
    }


    /**
     * Indicates whether some other object is "equal to" this one.
     *
     * @param o  the object with which to compare this one
     *
     * @return   <code>true<\code> if this object is the same as the obj
     *           argument; <code>false<\code> otherwise
     */
    public boolean equals(Object o)
    {
      LatticeNode otherNode = (LatticeNode)o;
      return granularity.equals(otherNode.granularity);
    }


    /**
      * Returns a string representation of the lattice node.
      *
      * @return a String representation of the lattice node.
      */
    public String toString()
    {
      String indent2 = "";
      String indent3 = "";
      String coarserStr = "" , finerStr = "", congruentStr = "", edges = "";

      for (int ii = 0; ii < SPACES * 2; ii++)
        indent2 += " ";

      for (int ii = 0; ii < SPACES * 3; ii++)
        indent3 += " ";

      Iterator i = coarserEdges.values().iterator();
      while (i.hasNext())
        coarserStr += "\n" + indent3 + i.next();

      i = finerEdges.values().iterator();
      while (i.hasNext())
        finerStr += "\n" + indent3 + i.next();

      i = congruentEdges.values().iterator();
      while (i.hasNext())
        congruentStr += "\n" + indent3  + i.next();

      if (!coarserStr.equals(""))
        edges += "\n" + indent2 + "Coarser   Edges: " + coarserStr;
      if (!finerStr.equals(""))
        edges += "\n" + indent2 + "Finer     Edges: " + finerStr;
      if (!congruentStr.equals(""))
        edges += "\n" + indent2 + "Congruent Edges: " + congruentStr;
      return "LatticeNode: " + granularity.getLocalName() + edges;
    }
  }


/* --------------------------------------------------------------------------
   LATTICE EDGE
   -------------------------------------------------------------------------- */

  /**
    * <code>LatticeEdge</code> class is a private inner class to
    *   <code>GranularityLattice</code>.  This class encapsulates a reference
    *   to a node to which an edge connects as well as mapping between the node
    *   (granularity) whose edge this is and the node (granularity) that is
    *   given.
    *
    * @author  Jessica Miller
    * @version 0.1, March 2003
    * @see GranularityLattice
    * @see GranularityLattice.LatticeNode
    * @status implementation complete
    */
  private class LatticeEdge
  {
    // the weights for edges should reflect the following properties
    //   * since irregular mappings are more expensive to call, weight them
    //     significantly higher than regular or induced
    //   * in order that an induced edge be more desirable than > 1 regular,
    //     but less desirable than exactly 1 regular edge (for purposes of
    //     finding the correct greatest common granularity) weight the induced
    //     edge > regular edges but < 2 * weight of regular edge
    public static final int INDUCED_WEIGHT   = 15;
    public static final int IRREGULAR_WEIGHT = 100;
    public static final int REGULAR_WEIGHT   = 10;

    /**
      * The node to which this edge connects (the node from which this edge
      *   connects will be the node that has this edge in its list of edges).
      */
    public LatticeNode node;

    /**
      * The mapping that is a label of this edge and can be used to convert
      *   from the owning granularity (<code>LatticeNode</code>) to the
      *   granularity of the node to which this edge connects.
      */
    public Mapping mapping;

    /**
      * The weight of this edge.
      */
    private int weight;


    /**
      * Construct a <code>LatticeEdge</code> given a "to" node and a mapping.
      *
      * @param n  the node to which the edge connects
      * @param m  the label which is the mapping between the owning node and
      *           the given node
      */
    public LatticeEdge(LatticeNode n, Mapping m)
    {
      node    = n;
      mapping = m;
      weight = (m instanceof RegularMapping)? REGULAR_WEIGHT : IRREGULAR_WEIGHT;
    }


    /**
      * Construct a <code>LatticeEdge</code>.
      *
      * @param n  the node to which the edge connects
      * @param m  the label which is the mapping between the owning node and
      *           the given node
      * @param w  the weight of the edge
      */
    public LatticeEdge(LatticeNode n, Mapping m, int w)
    {
      node    = n;
      mapping = m;
      weight  = w;
    }


    /**
      * Returns a string representation of the lattice edge.
      *
      * @return a String representation of the lattice edge.
      */
    public String toString()
    {
      return "LatticeEdge [toNode = " + node.granularity.getLocalName()
             + "; " + mapping + "]";
    }
  }


  /**
    * <code>SPElement</code> class is a private inner class to
    *   <code>GranularityLattice</code>.  This class encapsulates a shortest
    *   path estimation and a predecessor object.  This is used to combine the
    *   two pieces of information needed from a shortest path algorithm.
    *
    * @author  Jessica Miller
    * @version 0.1, March 2003
    * @see GranularityLattice
    * @see GranularityLattice.LatticeNode
    * @status implementation complete
    */
  private class SPElement
  {
    /**
      * An estimation of the cost from the shortest path source to the node that
      *   owns this shortest path information.
      */
    public Integer estimate;

    /**
      * The predecessor of the node that owns this shortest path information.
      */
    public Object predecessor;

    /**
      * Constructs a shortest path element.
      *
      * @param estimate     the estimation of the cost from the shortest path
      *                     source to the node that owns this shortest path
      *                     information
      * @param predecessor  the predecessor of the node that owns this shortest
      *                     path information
      */
    public SPElement(Integer estimate, Object predecessor)
    {
      this.estimate    = estimate;
      this.predecessor = predecessor;
    }

    /**
      * Returns a string representation of this shortest path element.
      */
    public String toString()
    {
      return "SPElement [est = " + estimate + "; pred =  " + predecessor;
    }
  }
}

