package tauzaman.calendricsystem;

import java.net.*;
// import tauzaman.temporaldatatypes.*;
import tauzaman.calendar.*;
import java.io.*;

/**
* The Granularity class is for granularities in calendars.
* A granularity is a "system of measurement" within a calendar,
* for example, days, months, and years are common granularities
* within the Gregorian calendar.  A granularity is a modeled as
* a sequence of granules.  The granules are labeled with integers.
* Each integer is the distance of the granule from the granularity's
* anchor point (the zeroth granule).
*
* @author  Bedirhan Urgun
* @version 0.1, 10/10/02
* @see     tauzaman.timestamp.Granule
*
* @status design complete, implementation complete
*/
public class Granularity implements Serializable{

  /** local name of this <code>Granularity</code> **/
  private String name = null;

  /** URL of <code>Calendar</code> of which this <code>Granularity</code> is created **/
  private URL calendarUrl = null;

  /**
  * Constructs a <code>Granularity</code>.
  *
  * @param name String name of <code>Granularity</code>, such as "day"
  * @param calendarUrl URL of <code>Calendar</code> that contains this <code>Granularity</code> name
  */
  public Granularity(String name, URL calendarUrl) {
      this.name = name;
      this.calendarUrl = calendarUrl;
  }

  /**
  * Constructs a <code>Calendar</code>less <code>Granularity</code>.
  *
  * @param name String name of <code>Granularity</code>, such as "day"
  */
  public Granularity(String name) {
      this.name = name;
  }

  /**
  * Sets <code>Calendar</code> URL of this <code>Granularity</code>.
  *
  * @param calendarUrl URL of <code>Calendar</code> that contains this <code>Granularity</code> name
  */
  public void setCalendarUrl(URL calendarUrl){
      // we may have a check here, but would be expensive
      // and also system is the only one uses this.
      this.calendarUrl = calendarUrl;
  }

  /**
  * Returns name of this <code>Granularity</code>.
  *
  * @return String name of this <code>Granularity</code>
  */
  public String getLocalName(){
      return name;
  }


  /**
  * Returns calendar url part of this <code>Granularity</code>.
  * If this <code>Granularity</code> is calendar-less <code>Granularity</code>,
  * the null is returned.
  *
  * @return URL url of parent <code>Calendar</code> of this <code>Granularity</code>.
  */
  public URL getCalendarUrl(){
      return calendarUrl;
  }

  /**
  * Returns true if this <code>Granularity</code>'s global identity
  * is equal to the global identity of the given <code>Granularity</code>.
  *
  * There are three possibilities of this equality check:
  *
  * <ul>
  *  <li>
  *    Equality between <code>Calendar</code>-less <code>Granularities</code>
  *  is determined by checking their local names only,
  *  </li>
  *  <li>
  *    Equality between a <code>Calendar</code>-less <code>Granularity</code>
  * and <code>Calendar</code> <code>Granularity</code> always returns false,
  *  </li>
  *  <li>
  *    Equality between a <code>Calendar</code> <code>Granularity</code>
  * and <code>Calendar</code> <code>Granularity</code> is determined by checking
  * their parent <code>Calendar</code>s as well as their local names.
  *  </li>
  * </ul>
  *
  * @param other <code>Granularity</code> to be compared
  * @return boolean true if two Granularities are equal, false otherwise.
  */
  public boolean equals(Object obj){
      Granularity otherGranularity = (Granularity)obj;

      /* if both are calendar-less granularities */
      if (calendarUrl == null && otherGranularity.getCalendarUrl() == null)
      {
        if (name.equals(otherGranularity.getLocalName()))
          return true;
      }
      /* if none of them are calendar-less granularities */
      else if( calendarUrl != null && otherGranularity.getCalendarUrl() != null)
      {
        if (name.equals(otherGranularity.getLocalName()) &&
            calendarUrl.equals(otherGranularity.getCalendarUrl()))
          return true;
      }

      /* otherwise always return false */
      return false;
  }

  /**
  * Returns String representation of this <code>Granularity</code>.
  * return a String representing this <code>Granularity</code>
  */
  public String toString(){
      if(calendarUrl == null)
          return getLocalName();
      return getLocalName() + " calendar " + calendarUrl.toExternalForm();
  }

  /* *** methods add by jkm */
  public int hashCode()
  {
    String fullName = getLocalName() + calendarUrl.toExternalForm();
    return fullName.hashCode();
  }
}
