package tauzaman.calendricsystem;

import java.net.*;
import java.util.*;

import org.w3c.dom.*;

import tauzaman.*;

/* I have to put this explicitly */
import tauzaman.calendar.Calendar;
import tauzaman.calendar.*;
import tauzaman.calendar.mapping.*;

import tauzaman.property.*;
import tauzaman.field.*;

/* *** jkm */
import tauzaman.calendricsystem.granularitylattice.GranularityLattice;
import tauzaman.timestamp.Granule;


/*
  This class includes methods that should be public.
  I still beleive that most of them, being public, does not hurt anyone.
  On thing is user can't get handle to CalendricSystem object, that
  he/she created via a TauZamanService.
*/


/**
* <code>CalendricSystem</code> class represents a Calendric System, which contains
* <code>Calendar</code>s' <code>Epoch</code>s and provides an entrance point to
* all <code>Epoch</code> and <code>Calendar</code> services.
*
* @author  Bedirhan Urgun and Curtis Dyreson
* @version 0.1, 11/15/02
* @see tauzaman.calendar.Calendar
* @see tauzaman.property.PropertyManager
* @status still being designed (granularity lattice), implementation started
*/


public class CalendricSystem {

  /**
  * Url of the this <code>CalendricSystem</code>, which is enforced to be unique in
  * database
  */
  private URL url = null;

  /**
  * List of handle(s) of <code>Calendar</code>(s) that this calendric system contains, in
  * the order of default input order, which is defined in Calendar Specification
  * file.
  * @see tauzaman.calendar.Calendar
  */
  private Calendar [] calendars = null;

  /**
  * Keeps <code>Calendar</code> names as its keys and
  * <code>Calendar<code>s as values.
  */
  private Hashtable calendarNamesToCalendars = null;

  /**
  * Keeps <code>Calendar</code> urls as its keys and
  * <code>Calendar<code> names as values.
  */
  private Hashtable calendarUrlsToCalendarNames = null;

  /**
  * Granularity mappings belong to this <code>CalendricSystem</code>.
  */
  private Mapping mappings []= null;

  /**
  * Default regular expression for useful information in inputs.
  */
  String token = null;

  /**
  * Default <code>Granularity</code> of this <code>CalendricSystem</code>.
  */
 Granularity defaultGranularity = null;

  /**
  * a handle to unique <code>CalendarRepository</code> of, <code>CalendricSystemRepository</code>,
  * indirectly this <code>TauZamanSystem</code>
  */
  private CalendarRepository cr = null;

  /** *** jkm */
  private GranularityLattice lattice = null;

  /**
  * Constructs a <code>CalendricSystem</code> object;
  * Forms a <code>CalendricSystem</code> by parsing url.
  *
  * @param cr <code>CalendarRepository</code>, which this <code>CalendricSystem</code>
  * will use to load <code>Calendar</code>s
  *
  * @param tauzaman.url Url of <code>CalendricSystem</code>
  *
  * @throws CalendricSystemFormationException if any error occurs while forming
  * <code>CalendricSystem</code> from given URL
  *
  */
  public CalendricSystem(CalendarRepository cr, URL url)
                         throws CalendricSystemFormationException{

      /* gets url */
      this.url = url;

      /* get CalendarRepository */
      this.cr = cr;

      /* forms <code>CalendricSystem</code>'s state */
      formCalendricSystem();

      /* form lattice */
      lattice = new GranularityLattice(calendars, mappings);
  }

  /**
  * Returns url of this <code>CalendricSystem</code>'s specification file.
  *
  * @return URL url of this <code>CalendricSystem</code>
  */
  public URL getUrl(){
      return url;
  }

  /**
  * Returns name of a <code>Calendar</code> that is in this <code>CalendricSystem</code>.
  *
  * @return name of <code>Calendar</code> in this <code>CalendricSystem</code> given
  * its url
  *
  * @throws CalendricSystemFormationException if no <code>Calendar</code> can not be found
  * in this <code>CalendricSystem</code>
  */
//  public String getCalendarName(URL calendarUrl) throws CalendricSystemServiceException{
//      if(!calendarUrlsToCalendarNames.containsKey(calendarUrl))
//          throw new CalendricSystemServiceException("Exception no such Calendar: " + calendarUrl.toExternalForm() +
//                                                                 " is imported in CalendricSystem: " + url.toExternalForm());
//      return (String)calendarUrlsToCalendarNames.get(calendarUrl);
//  }

  /**
  * Returns all <code>Calendar</code>s in this <code>CalendricSystem</code>.
  *
  * @return array of <code>Calendar</code>s
  */
  public Calendar [] getAllCalendars(){
      return calendars;
  }

  /**
  * Returns default regular expression for this <code>CalendricSystem</code>
  */
  public String getDefaultRegex(){
      return token;
  }

  /**
  * Initializes this <code>CalendricSystem</code>'s states by
  * parsing file that its url points.
  *
  * @throws CalendricSystemFormationException if any error occurs while forming
  * <code>CalendricSystem</code> from given URL
  */
  private void formCalendricSystem() throws CalendricSystemFormationException{

      XMLParser parser = new XMLParser();
      Element root = null;

      try{
          root = parser.parseURL(url);
      }
      catch(XMLParseException xpe){
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " + url.toExternalForm(), xpe);
      }


      /*******************************************DEFAULT INPUT ORDER**********************************************/

      /* First get default input order to order calendars  */
      Vector defaultInputOrderNodes = parser.locator(root, "defaultInputOrder");
      if(defaultInputOrderNodes.size() != 1)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " + url.toExternalForm() + " no default input order is specified!");
      Element defaultInputOrderNode = (Element)defaultInputOrderNodes.elementAt(0);
      Node defaultInputOrderTextNode = defaultInputOrderNode.getFirstChild();
      if(defaultInputOrderTextNode.getNodeType() != Node.TEXT_NODE)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " + url.toExternalForm() + " default input order should have TEXT node!");
      String defaultInputOrder = defaultInputOrderTextNode.getNodeValue();

      /* check if it contains replicated names */
      /* remove the replicated names, ayikla */

      /*******************************************IMPORTING CALENDARS**********************************************/

      /* First, get imported Calendars */
      Vector calendarNodes = parser.locator(root, "importCalendar");

      if(calendarNodes.size() == 0)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " + url.toExternalForm() + " no Calendars specified!");

      /* A hashtable, which maps Calendar names to Calendars (for checking
         and efficiency purposes) only in the context of this method */
      Hashtable calendarNamesToCalendarsRaw = new Hashtable();

      calendarUrlsToCalendarNames = new Hashtable();

      /* load them */
      for(int i = 0; i < calendarNodes.size(); i++ ){
          Element calendarNode = (Element)(calendarNodes.elementAt(i));
          String calendarName = calendarNode.getAttribute("name");
          if(calendarNamesToCalendarsRaw.containsKey(calendarName)){
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                           url.toExternalForm() + " duplicate Calendar specified!: " + calendarName);
          }
          else{

              String calendarUrl = calendarNode.getAttribute("url");
              /* load the Calendar, I don't care if user loads two same Calendars
                 with different names */
              Calendar c = null;

              try{
                  c = cr.loadCalendar(new URL(calendarUrl));
              }
              catch(CalendarFormationException e){
                  throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                                     url.toExternalForm() + " cannot form Calendar!", e);
              }
              catch(MalformedURLException e){
                  throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                                                  url.toExternalForm() + " bad URL!", e);
              }

              /* mapping is from Calendar name to Calendar itself */
              calendarNamesToCalendarsRaw.put(calendarName, c);

              /* we don't allow to have two same Calendars with different local names */
              if(calendarUrlsToCalendarNames.containsKey(c.getUrl()))
                  throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                       url.toExternalForm() + " duplicate Calendar specified!: " + c.getUrl().toExternalForm());

              /* mapping is from Calendar url to Calendar name */
              calendarUrlsToCalendarNames.put(c.getUrl(), calendarName);
          }
      }

      /* allocate memory for Calendars and their names */
      calendars = new Calendar[calendarNodes.size()];
      calendarNamesToCalendars = new Hashtable();

      /* order calendars according to defaultInputOrder */
      StringTokenizer st = new StringTokenizer(defaultInputOrder);

      int calendarCount = 0;
      while (st.hasMoreTokens()) {

          String calendarName = st.nextToken();

          if(!calendarNamesToCalendarsRaw.containsKey(calendarName))
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                            url.toExternalForm() + " Calendar is not imported: " + calendarName);

          calendars[calendarCount] = (Calendar)calendarNamesToCalendarsRaw.remove(calendarName);
          calendarNamesToCalendars.put(calendarName, calendars[calendarCount]);

          calendarCount++;

      }

      /* handle the remaining Calendars */

      for (Enumeration remainingCalendarNames = calendarNamesToCalendarsRaw.keys(); remainingCalendarNames.hasMoreElements() ;) {

          String calendarName = (String)remainingCalendarNames.nextElement();

          calendars[calendarCount] = (Calendar)calendarNamesToCalendarsRaw.remove(calendarName);
          calendarNamesToCalendars.put(calendarName, calendars[calendarCount]);

          calendarCount++;
      }

      /* Underlying granularity of First Calendar in calendars will be default! Careful default is chosen
         after initial sort according to default input order */
      defaultGranularity = calendars[0].getUnderlyingGranularity();

      /*******************************************GRANULARITY MAPPINGS**********************************************/

      /* allocate memory for granularity mappings */
      Hashtable mappingTable = new Hashtable();

      Vector mappingNodes = parser.locator(root, "mappings");
      if(mappingNodes.size() != 1)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                    url.toExternalForm() + " there should be at least one mapping nodes between Calendars!");

      Element mappingNode = (Element)(mappingNodes.elementAt(0));

      /* first get regular mappings */
      Vector regularMappingNodes = parser.locator(mappingNode, "regularMapping");
      for(int j = 0; j < regularMappingNodes.size(); j++){
          Element regularMappingNode = (Element)(regularMappingNodes.elementAt(j));

          Granularity to = formGranularity(regularMappingNode.getAttribute("to"),
                                    regularMappingNode.getAttribute("toCalendar"));
          Granularity from = formGranularity(regularMappingNode.getAttribute("from"),
                                     regularMappingNode.getAttribute("fromCalendar"));

          int relationship = Mapping.fetchRelationship(regularMappingNode.getAttribute("relationship"));

          if(relationship == -1)
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                              url.toExternalForm() + " relationship undefined given as: " +
                                            regularMappingNode.getAttribute("relationship"));


          RegularMapping rm = new RegularMapping(to, from, relationship, regularMappingNode);

          if(mappingTable.containsKey(rm))
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                         url.toExternalForm() + " regular mapping between " +  to.getLocalName() +
                                             " and " + from.getLocalName() + " already defined!");

          mappingTable.put(rm, "dummy");
      }

      /* second get irregular mappings */
      Vector irregularMappingNodes = parser.locator(mappingNode, "irregularMapping");
      for(int j = 0; j < irregularMappingNodes.size(); j++){
          Element irregularMappingNode = (Element)(irregularMappingNodes.elementAt(j));

          Granularity to = formGranularity(irregularMappingNode.getAttribute("to"),
                                   irregularMappingNode.getAttribute("toCalendar"));
          Granularity from = formGranularity(irregularMappingNode.getAttribute("from"),
                                     irregularMappingNode.getAttribute("fromCalendar"));

          String irregularMappingUrl = (irregularMappingNode.getAttribute("url")).trim();

          /* get relationship between granularity and fromGranularity */
          int relationship = Mapping.fetchRelationship(irregularMappingNode.getAttribute("relationship"));

          if(relationship == -1)
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " relationship undefined given as: " +
                                                irregularMappingNode.getAttribute("relationship"));

          IrregularMapping irm = null;

          try{
              irm = new IrregularMapping(to, from, relationship, new URL(irregularMappingUrl), irregularMappingNode);
          }
          catch(MalformedURLException e){
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                                              url.toExternalForm() + " bad URL!", e);
          }
          catch(IrregularMappingFormationException ie){
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                        url.toExternalForm() + " cannot form IrregularMapping!", ie);
          }

          if(mappingTable.containsKey(irm))
              throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                         url.toExternalForm() + " irregular mapping between " +  to.getLocalName() +
                                             " and " + from.getLocalName() + " already defined!");

          mappingTable.put(irm, "dummy");

      }

      /* convert mappingTable Hashtable to mappings array */
      mappings = new Mapping [mappingTable.size()];
      int i = 0;
      for(Enumeration e = mappingTable.keys(); e.hasMoreElements();){
          mappings[i] = (Mapping)e.nextElement();
          i++;
      }

      /*******************************************DEFAULTS**********************************************/

      Vector defaultNodes = parser.locator(root, "defaults");
      if(defaultNodes.size() != 1)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                  url.toExternalForm() + " there should be one default element!");

      Element defaultNode = (Element)defaultNodes.elementAt(0);
      token = defaultNode.getAttribute("token");
      if(token == null)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                              url.toExternalForm() + " default regular expr. should be defined!");
  }

  /**
  * Returns <code>Mapping</code>s of this
  * <code>CalendricSystem</code>.
  * @return an array of <code>Mapping</code>s
  */
  public Mapping [] getMappings(){
      return mappings;
  }

  /**
  * Returns a <code>Granularity</code> produced from global identity of mapping elements. It also
  * checks validity of mapping names.
  *
  * @param localName String name of <code>Granularity</code> given as in <code>CalendricSystem</code> spec file
  * @param calendarName String name of <code>Calendar</code> given as in <code>CalendricSystem</code> spec file
  *
  * @return <code>Granularity</code>
  * @throws CalendricSystemFormationException if any abnormal condition occurs when forming
  * <code>Granularity</code>
  */
  private Granularity formGranularity(String localName, String calendarName) throws CalendricSystemFormationException{

      /* check validity of calendar in this calendric system */
      if(!calendarNamesToCalendars.containsKey(calendarName))
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                       url.toExternalForm() + " mappings Calendar name of \"to\" attribute is not defined: " + calendarName);

      Calendar calendar = (Calendar)calendarNamesToCalendars.get(calendarName);

      /* check if calendar has this granularity */
      Granularity granularity = calendar.getGranularityByLocalName(localName);

      if(granularity == null)
          throw new CalendricSystemFormationException("Exception when parsing xml file in: " +
                                     url.toExternalForm() + " granularity name " + localName +
                                     " is not defined in: " + calendarName);

      return granularity;
  }

  /**
  * Returns default <code>Granularity</code> of this <code>CalendricSystem</code>.
  *
  * @return <code>Granularity</code>, which is the default <code>Granularity</code>
  */
  public Granularity getDefaultGranularity(){
      return defaultGranularity;
  }

  /**
  * Returns an array of ordered list of <code>Calendar</code>s for converting
  * a temporal data type to corresponding output string.
  *
  * @param overrideInputOrderProperty <code>Property</code> that will be used
  * when forming an ordered input calendar list
  *
  * @throws CalendricSystemServiceException if any problem occurs when forming
  * an ordered <code>Calendar</code> list
  *
  * @return array of ordered <code>Calendar</code>s
  */
  public Calendar [] getOrderedCalendars(Property overrideInputOrderProperty) throws CalendricSystemServiceException{

     String value = null;

     try{
         value = new URL(overrideInputOrderProperty.getContent()).toExternalForm();
     }
     catch(MalformedURLException mue){
         throw new CalendricSystemServiceException("Exception when forming ordered calendars: override input calendar is invalid");
     }
 
     if(value == null)
         throw new CalendricSystemServiceException("Exception when forming ordered calendars: override input calendar is invalid");

     Calendar calendar = null;

     /* check if overrideInputOrderProperty is valid in this CalendricSystem context */
     for(int i = 0; i < calendars.length; i++){
         if(value.equals((calendars[i].getUrl()).toExternalForm())){
             calendar = calendars[i];
             break;
         }
     }

     if(calendar == null)
         throw new CalendricSystemServiceException("Exception when forming ordered calendars: override "
                               + "input calendar does not correspond to valid Calendar in CalendricSystem");

     Vector orderedCalendarListRaw = new Vector();

     /* re-order Calendars according to property's value */
     for(int i = 0; i < calendars.length; i++)
         if(calendar.equals(calendars[i])) /* we use a repository, so we keep only one */
             orderedCalendarListRaw.addElement(calendars[i]);

     /* get the remaining Calendars in order */
     for(int i = 0; i < calendars.length; i++)
         if(!calendar.equals(calendars[i]))
             orderedCalendarListRaw.addElement(calendars[i]);

     Calendar orderedCalendarList [] = new Calendar [orderedCalendarListRaw.size()];
     for(int i=0;i<orderedCalendarListRaw.size();i++)
         orderedCalendarList[i] = (Calendar)orderedCalendarListRaw.elementAt(i);

     return orderedCalendarList;
  }

  /**
  * Returns url of this <code>CalendricSystem</code>.
  *
  * @return String representation of a URL
  */
  public String toString(){
      StringBuffer rep  = new StringBuffer();
      rep.append("\nCalendricSystem: " + url.toExternalForm());
      for(int i = 0; i < calendars.length; i++)
          rep.append(calendars[i].toString());
      for(int i = 0; i < mappings.length; i++)
          rep.append(mappings[i].toString());
      rep.append("\nDefault Granularity: " + defaultGranularity.toString() + "\n");
      return rep.toString();
  }

  /* *** methods added by jkm */

  public void printGranularityLattice()
  {
    System.out.println(lattice);
  }

  /*
  public GranularityLattice getGranularityLattice()
  {
    return lattice;
  }
  */

  public Calendar getCalendar(String calendarName)
  {
    return (Calendar)calendarNamesToCalendars.get(calendarName);
  }

  public Granule castDeterminantInstant(Granule granule,
                                        Granularity toGranularity)
      throws CalendricSystemServiceException
  {
    return lattice.castAnchored(granule, toGranularity);
  }

  public Granule castDeterminantInterval(Granule granule,
                                         Granularity toGranularity)
      throws CalendricSystemServiceException
  {
    return lattice.castUnanchored(granule, toGranularity);
  }

  public boolean isCoarser(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    return lattice.isCoarser(g1, g2);
  }

  public boolean isEquivalent(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    return lattice.isEquivalent(g1, g2);
  }

  public boolean isFiner(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    return lattice.isFiner(g1, g2);
  }

  public boolean isIncomparable(Granularity g1, Granularity g2)
      throws CalendricSystemServiceException
  {
    return lattice.isIncomparable(g1, g2);
  }

  public void clearCaches()
  {
    lattice.clearCaches();
  }
}
