package tauzaman.calendar;

import org.w3c.dom.*;
import tauzaman.calendar.mapping.Mapping;
import tauzaman.calendricsystem.Granularity;

/**
* <code>RegularMapping</code> class represents the granularity and granularity
* mapping information of a <code>Calendar</code>.
* For example;
* <pre>
*   &lt; granularity name = &quot;dayHundreth&quot; &gt;
*      &lt; regularMapping from = &quot;second&quot; groupSize = &quot;864&quot; anchor = &quot;0&quot;
*           anchorGranularity = &quot;second&quot; /&gt;
*   &lt; /granularity &gt;
* </pre>
*
* @author  Bedirhan Urgun
* @version 0.1, 10/10/02
* @see     tauzaman.calendar.Calendar
* @status design complete, implementation complete
*/


public class RegularMapping extends Mapping{

  /**
  * Group size of from <code>Granularity</code> in to <code>Granularity</code>.
  */
  private int groupSize;

  /**
  * Period size of from <code>Granularity</code> in to <code>Granularity</code>.
  */
  private int periodSize;

  /**
  * anchor <code>Granularity</code>
  */
  private Granularity anchorGranularity = null;

  /**
  * Value of anchor
  */
  private int anchor;


  /* *** added by jkm */
  public RegularMapping(Granularity to, Granularity from, int relationship,
                        int groupSize, int periodSize, int anchor)
  {
    super(to, from, relationship);
    this.groupSize         = groupSize;
    this.periodSize        = periodSize;
    this.anchor            = anchor;
    this.anchorGranularity = from;
  }

  /**
  * Constructs a <code>RegularMapping</code> object, which stores a Regular Mapping information
  * for a <code>Calendar</code>.
  *
  * @param to <code>Granularity</code>, which mapping is declared to
  * @param from <code>Granularity</code>, which mapping is declated from
  * @param root DOM Element, which points to <code>RegularMapping</code> information of
  * corresponding xml specification file.
  *
  * @see tauzaman.calendar.IrregularMapping
  */
  public RegularMapping(Granularity to, Granularity from, int relationship, Element root){
      super(to, from, relationship);
      formRegularMapping(root);
  }

  /**
  * Forms an <code>RegularMapping</code> object by parsing a xml specification file.
  *
  * @param root DOM Element, which points to <code>RegularMapping</code> information of
  * corresponding xml specification file.
  */
  private void formRegularMapping(Element root){

      String groupSizeString = root.getAttribute("groupSize");
      String periodSizeString = root.getAttribute("periodSize");
      String anchorString = root.getAttribute("anchor");
      String anchorGranularityString = root.getAttribute("anchorGranularity");
      String sourceGranuleString = root.getAttribute("sourceGranule");
      String anchoredAtGranuleString = root.getAttribute("anchoredAtGranule");


      /* handle all default cases */

      if(groupSizeString.equals("") && periodSizeString.equals("")){
          groupSize = 0; periodSize = 0;
      }
      else if(!groupSizeString.equals("") && periodSizeString.equals("")){
          groupSize = Integer.parseInt(groupSizeString);
          periodSize = groupSize;
      }
      else if(groupSizeString.equals("") && !periodSizeString.equals("")){
          periodSize = Integer.parseInt(periodSizeString);
          groupSize = periodSize;
      }
      else{
          groupSize = Integer.parseInt(groupSizeString);
          periodSize = Integer.parseInt(periodSizeString);
      }

      /*
        There are three ways of getting anchor;
          1. anchor is directly given
          2. anchor should be derived from "sourceGranule" and "anchoredAtGranule"
          3. anchor is 0, which is default value.
      */
      if(!anchorString.equals(""))
      {
        anchor = Integer.parseInt(anchorString);
      }
      else if( !( sourceGranuleString.equals("") || anchoredAtGranuleString.equals("") ) ){
          /* evaluate the anchor */
          int sourceGranule = Integer.parseInt(sourceGranuleString);
          int anchoredAtGranule = Integer.parseInt(anchoredAtGranuleString);
          anchor = anchoredAtGranule - sourceGranule * periodSize;
      }
      else
          anchor = 0;

      if(anchorGranularityString.equals(""))
          anchorGranularity = from;
      else
          anchorGranularity = new Granularity(anchorGranularityString,
                                  from.getCalendarUrl()); // calendar url should be same

  }


   /**
   * Casts a given anchored point.
   *
   * @param point point in time to be casted
   * @return long casted point, if mapping can not be performed -1 is returned
   */
   public long performAnchoredCast(long point){
       /* according to the relationship between granularities, choose a formula */
       switch(relationship){
           case FINER_TO_COARSER:
                  if( new Double(Math.IEEEremainder( (point - anchor) , periodSize) ).intValue() < groupSize  )
                      return new Double(Math.floor( (point - anchor) / periodSize) ).longValue();
                  break;
           case COARSER_TO_FINER:
                  return point*periodSize + anchor;
       }
       return -1;
   }

   /**
   * Scales a given anchored point.
   *
   * @param point point in time to be scaled
   * @return long value(s) of scaled point, , if mapping can not be performed -1 is returned
   */
   public long [] performAnchoredScale(long point){
       long result[] = new long [2];
       result[0] = -1; result[1] = -1;

       /* according to the relationship between granularities, choose a formula */
       switch(relationship){
           case FINER_TO_COARSER:
                  if( new Double(Math.IEEEremainder( (point - anchor) , periodSize) ).intValue() < groupSize  )
                      result[0] = new Double(Math.floor( (point - anchor) / periodSize) ).longValue();
                  break;
           case COARSER_TO_FINER:
                  result[0] = point*periodSize + anchor;
                  result[1] = point*periodSize + anchor + groupSize - 1;

       }
       return result;
   }

   /**
   * Casts a given unanchored point.
   *
   * @param point point in time to be casted
   * @return long casted point, if mapping can not be performed -1 is returned
   */
   public long performUnanchoredCast(long point){
       return performAnchoredCast(point);
   }

   /**
   * Scales a given unanchored point.
   *
   * @param point point in time to be scaled
   * @return long value(s) of scaled point, if mapping can not be performed -1 is returned
   */
   public long [] performUnanchoredScale(long point){
       long result[] = new long [2];
       result [0] = point;
       result [1] = point;
       return result;
   }


  /* Accessor Methods (addition to methods of Mapping) */

  /**
  * Returns the period size of from <code>Granularity</code> with respect to
  * to <code>Granularity</code>.
  *
  * @return int period size
  */
  public int getPeriodSize(){
      return periodSize;
  }

  /**
  * Returns the group size of from <code>Granularity</code> with respect to
  * to <code>Granularity</code>.
  *
  * @return int group size
  */
  public int getGroupSize(){
      return groupSize;
  }

  /**
  * Returns the value anchor
  *
  * @return int value
  */
  public int getAnchor(){
      return anchor;
  }

  /**
  * Returns anchor <code>Granularity</code>
  *
  * @return Granularity
  */
  public Granularity getAnchorGranularity(){
     return anchorGranularity;
  }

  /**
  * toString method
  *
  * @return string representation of this object
  */
  public String toString(){
      return super.toString() + " group: " + groupSize + " period: " + periodSize
        + " anchor " + anchor + " anchorGranularity: " + anchorGranularity.getLocalName();
  }

}
