package tauzaman.calendar;

import tauzaman.calendar.mapping.Mapping;
import tauzaman.calendricsystem.Granularity;
import java.net.*;
import java.util.*;
import tauzaman.*;
import org.w3c.dom.*;

/**
* <code>IrregularMapping</code> class represents the irregular 
* granularity mapping information of a <code>Calendar</code>.
* For example;
* <pre>
*   &lt; granularity name = &quot;month&quot; &gt;
*      &lt; irregularMapping from = &quot;day&quot; url = &quot;.../classname.class&quot; &gt;
*         &lt; method name = &quot;gregCastMonthToDay&quot; /&gt;
*         &lt; method name = &quot;gregCastDayToMonth&quot; /&gt;
*         &lt; method name = &quot;gregScaleMonthToDay&quot; /&gt;
*      &lt; /irregularmapping &gt;
*   &lt; /granularity &gt;
* </pre>
*
* @author  Bedirhan Urgun
* @version 0.1, 10/10/02

* @see     tauzaman.calendar.Calendar
* @see     tauzaman.calendar.mapping.Mapping
* @see     tauzaman.calendricsystem.Granularity
* @status design complete, implementation complete
*/

public class IrregularMapping extends Mapping{

  /**
  * A Hashtable relates type and behavior of methods to method names.
  * There two possible types and behaviors for methods; cast, scale as types 
  * and  anchored, unanchored for bahaviors. A method for
  * <code>IrregularMapping</code> can only belong to one of these types.
  * Keys are behavior+type names, "anchoredscale", "unachoredscale", "anchoredcast", 
  * "unanchoredcast" and values are method names. 
  */
  private Hashtable methodBehaviorTypeToNames = null;
  
  /** 
  * URL that points either to a (.jar file | directory) or a class file  
  */  
  private URL url = null;

  /**
  * Instance of <code>ClassLoaderMethodCaller</code>, which keeps
  * dynamically loaded classes by this <code>IrregularMapping</code>
  */
  private ClassLoaderMethodCaller clmc = null;

  /**
  * Constructs an <code>IrregularMapping</code> object, which allocates memory
  * for method names and urls.
  *  
  * @param to <code>Granularity</code>, which  <code>IrregularMapping</code> is declared to
  * @param from <code>Granularity</code>, which  <code>IrregularMapping</code> is declared from 
  * @param relationship int between the <code>Granularities</code> of this <code>IrregularMapping</code> 
  * @param url URL of irregular method implementations
  * @param root DOM Element, which points to <code>IrregularMapping</code> information of
  * corresponding xml specification file.
  *
  * @throws IrregularMappingFormationException if any abnormal condition occurs when 
  * parsing a <code>IrregularMapping</code>
  */
  public IrregularMapping(Granularity to, Granularity from, int relationship, URL url, Element root) 
                                                  throws IrregularMappingFormationException{
      super(to, from, relationship);

      this.url = url;

      /* allocates memory for method names */
      methodBehaviorTypeToNames = new Hashtable();

      /* forms this irregular mapping */
      formIrregularMapping(root);

      /* this is a test and should be removed later on */
      //testPerformMappings();      


  }

  /**
  * Forms an <code>IrregularMapping</code> object by parsing a xml specification file.
  * 
  * @param root DOM Element, which points to <code>IrregularMapping</code> information of
  * corresponding xml specification file.
  *
  * @throws IrregularMappingFormationException if any abnormal condition occurs when 
  * parsing a <code>IrregularMapping</code>
  */
  private void formIrregularMapping(Element root) throws IrregularMappingFormationException{
      
      XMLParser parser = new XMLParser();

      String irregularMappingUrl = url.toExternalForm();
              
      /***************CLMC FORMATION*****************/

      // form an instance of CLMC here
      URL urls[] = new URL[1];

      String classUrl = null;
      //get the useful part of URL
      if(irregularMappingUrl.endsWith(".class"))
          classUrl = irregularMappingUrl.substring(0, irregularMappingUrl.lastIndexOf('/') + 1);
      else
          classUrl = irregularMappingUrl;

      /* form the url array */
      try{ urls[0] = new URL(classUrl); }
      catch(MalformedURLException e){
          throw new IrregularMappingFormationException("Exception when forming URL from: " + classUrl + " bad URL!", e);
      }
              
      /* form instance of CLMC */
      clmc = new ClassLoaderMethodCaller(urls);

      /***************CLMC FORMATION*****************/


      Vector methodNameNodes = parser.locator(root, "method");   
      for(int i = 0; i < methodNameNodes.size(); i++){
          Element methodNameNode = (Element)(methodNameNodes.elementAt(i));

          String methodName = methodNameNode.getAttribute("name");
          String className = methodNameNode.getAttribute("className");
          String type = methodNameNode.getAttribute("type").toLowerCase();
          String behavior = methodNameNode.getAttribute("behavior").toLowerCase();
              
          /* a class contains all the methods */
          if(className.equals("") && irregularMappingUrl.endsWith(".class")){
              className = irregularMappingUrl.substring(irregularMappingUrl.lastIndexOf('/') + 1, irregularMappingUrl.length());
              className = className.substring(0, className.lastIndexOf(".class"));
          }
          /* a directory or .jar file contains all the classes */
          else if(!(className.equals("")) && !(irregularMappingUrl.endsWith(".class"))){
              /* do nothing */
          }
          /* no other option, throw an error */
          else{
              throw new IrregularMappingFormationException("Exception when parsing xml file in: " + 
                                url.toExternalForm() +" bad formed irregular mapping specification!");
          }

          /* default irregular mapping type is cast */
          if(type.equals(""))
              type = "cast";

          /* check on type names */
          if( !type.equals("cast") && !type.equals("scale") )
              throw new IrregularMappingFormationException("Exception when parsing xml file in: " + 
                    url.toExternalForm() + " method type is undefined for irregular mapping between: " +
                                             to.getLocalName() + " and " + from.getLocalName());

          /* default irregular mapping behavior is anchored */

          /* A simple note here: I could merge type and behavior, but for user this might be more
             appropriate. particularly he/she does not have to remember default values of both */

          if(behavior.equals(""))
              behavior = "anchored";

          /* check on type names */
          if( !behavior.equals("anchored") && !behavior.equals("unanchored") )
              throw new IrregularMappingFormationException("Exception when parsing xml file in: " + 
                    url.toExternalForm() + " method behavior is undefined for irregular mapping between: " +
                                             to.getLocalName() + " and " + from.getLocalName());
          
          String methodKey = behavior + type;

          if(methodBehaviorTypeToNames.containsKey(methodKey))
              throw new IrregularMappingFormationException("Exception when parsing xml file in: " + 
                    url.toExternalForm() + " method type and behavior are already defined for irregular mapping between: " +
                                             to.getLocalName() + " and " + from.getLocalName());

          /* add the behavior+type and corresponding method name to hashtable */
          methodBehaviorTypeToNames.put(methodKey, methodName);

          /* load the class in CLMC */
          try{
              clmc.loadClass(className, methodName);
          }
          catch(DynamicLoadException dle){
              throw new IrregularMappingFormationException("Exception when loading class:" + className +
                                                                        " in: " + url.toExternalForm(), dle);
          }

      }

  }


   /**
   * Tests mappings when constructed...
   */
   private void testPerformMappings(){
       
       System.out.println();
       System.out.println(".......MAPPINGS ARE BEING TEST......");
       System.out.println(toString());
      
       URL ADGregorian, UofA, UofACalendricSystem;
       try{
           ADGregorian = new URL("http://www.eecs.wsu.edu/~cdyreson/pub/tauZaman/calendars/ADGregorianCalendar.class");
           UofA = new URL("http://www.eecs.wsu.edu/~cdyreson/pub/tauZaman/calendars/UofACalendar.class");
           UofACalendricSystem = new URL("http://www.eecs.wsu.edu/~cdyreson/pub/tauZaman/calendars/UofALimitedCalendricSystem.class");
       }
       catch(Exception e){
           System.out.println("test failed");
           return;
       }

       if(url.equals(ADGregorian)){
           if("second".equals(to.getLocalName()) && "minute".equals(from.getLocalName())){
              System.out.println("cast(1036906562 minute(s), seconds)   = " + performAnchoredCast(1036906562)
                   + " second(s)");    //  62214393662 [July 1, 1972 00:01
 
           }
       }
       else if(url.equals(UofA)){
           if("week".equals(to.getLocalName()) && "term".equals(from.getLocalName())){
               System.out.println("cast(16 term(s), weeks)  = " + performAnchoredCast(15)  + " week(s) [should be 191]");
           }
       }
       else if(url.equals(UofACalendricSystem)){
           if("hour".equals(to.getLocalName()) && "hour".equals(from.getLocalName())){

           }
       }


   }

   /**
   * Casts a given anchored point.
   *
   * @param point point in time to be casted
   * @return long casted point
   */ 
   private long performCast(long point, String castType){

      long result = -1;

      Class parameterTypes [] = new Class [1];
      
      try{
          parameterTypes[0] = Class.forName("java.lang.Long");
      }
      catch(ClassNotFoundException cnfe){
          // a severe exception. either we throw a new exception or 
          return result;
      }

      try{
  
          Object args[] = new Object[1];
          args[0] = new Long(point);

          /* check if method behavior+type exists */
          if(!methodBehaviorTypeToNames.containsKey(castType)){
              // either throw a new exception or
              return result;
          }
          /* get method name */
          String methodName = (String)methodBehaviorTypeToNames.get(castType);

          Long returnValue = (Long) (clmc.callMethod(methodName, parameterTypes, args));
  
          result = returnValue.longValue();
  
      }
      catch(DynamicLoadException dle){
          // either throw a new exception or
          return result;
      }
      
      return result;
      

   }
   

   /**
   * Casts a given anchored point.
   *
   * @param point point in time to be casted
   * @return long casted point
   */
   public long performAnchoredCast(long point)
   {
     /* *** jkm */
     return performCast(point, "anchoredcast");
   }


   /**
   * Scales a given anchored point.
   *
   * @param point point in time to be scaled
   * @return long value(s) of scaled point
   */ 
   public long [] performAnchoredScale(long point){

      long result[] = new long [2];
      result[0] = -1; result[1] = -1;

      Class parameterTypes [] = new Class [1];
      
      try{
          parameterTypes[0] = Class.forName("java.lang.Long");
      }
      catch(ClassNotFoundException cnfe){
          // a severe exception. either we throw a new exception or 
          return result;
      }

      try{
  
          Object args[] = new Object[1];
          args[0] = new Long(point);

          /* check if method type exists */
          if(!methodBehaviorTypeToNames.containsKey("anchoredscale")){
              // either throw a new exception or
              return result;
          }

          /* get method name */
          String methodName = (String)methodBehaviorTypeToNames.get("anchoredscale");

          result = (long []) (clmc.callMethod(methodName, parameterTypes, args));
    
      }
      catch(DynamicLoadException dle){
          // either throw a new exception or
          return result;
      }
      
      return result;
      
   }


   /**
   * Casts a given unanchored point.
   *
   * @param point point in time to be casted
   * @return long casted point
   */ 
   public long performUnanchoredCast(long point){
       return performCast(point, "unanchoredcast");
   }
     
   /**
   * Scales a given unanchored point.
   *   
   * @param point point in time to be scaled
   * @return long value(s) of scaled point
   */ 
   public long [] performUnanchoredScale(long point){
       long result[] = new long [2];
       result [0] = point;
       result [1] = point;
       return result;
   }



  /**
  * Returns irregular mapping implementation url.
  *
  * @return Url of this irregular mapping implementation
  */
  public URL getUrl(){
      return url;
  }

  /**
  * Returns a String representing this <code>IrregularMapping</code> object.
  *
  * @return string URL of this <code>IrregularMapping</code>
  */
  public String toString(){
      return super.toString() + " url " + url.toExternalForm() + 
                             " methods " + methodBehaviorTypeToNames.toString();
  }

}
