package tauzaman;

import java.net.*;
import org.w3c.dom.*;

import java.util.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.*;
import java.io.StringReader;
import java.io.File;
import java.io.IOException;


/* do we want this class to be public?   
   do we want this class to be in this package, namely, directly under tauzaman package?   
   This is a general class, it does not contain any specific behavour to tauzaman!
*/


/**
* <code>XMLParser</code> class parses a given url, which contains an XML file,
* and returns the root of resulting DOM. It uses DOM to parse the xml file. The 
* classes, which utilize this class are;
* 
* <ul>
* <li> <code>CalendricSystem</code> </li>
* <li> <code>Calendar</code> </li>
* <li> <code>Property</code> </li>
* <li> <code>FVTable</code> </li>
* </ul>
*
* This class is basically an XML DOM Parser.
*
* @see tauzaman.calendricsystem.CalendricSystem
* @see tauzaman.calendar.Calendar
* @see tauzaman.property.Property
* @see tauzaman.field.FVTable
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1. 01/27/02
* @status design complete, implementation complete
*/

public class XMLParser{

  private DocumentBuilderFactory factory;

  /**
  * Constructs an <code>XMLParser</code> object, which parses an xml file, 
  * which given url points to. 
  */
  public XMLParser(){
    
    // Form a new DOM factory
    factory = DocumentBuilderFactory.newInstance();

    // Configure it to ignore comments
    factory.setIgnoringComments(true);
    // Prevent expansion of entity references
    factory.setExpandEntityReferences(false);
    // Configure it to coalesce CDATA nodes  
    factory.setCoalescing(true);

  }

  /**
  * Parses a given URL, which points to an xml file. 
  * Higher level exceptions will be handled in callees.
  *
  * @param url URL , which points to an xml file to be parsed
  *
  * @return a DOM Element object, which is the root element of parsed xml file 
  *
  * @throws XMLParseException if any abnormal condition occurs when forming a DOM
  * object
  */
  public Element parseURL(URL url) throws XMLParseException{

      Document document = null;

      try {
          DocumentBuilder builder = factory.newDocumentBuilder();
          document = builder.parse(url.toExternalForm());
      } 
      catch (SAXException sxe) {
          // Error generated during parsing
          throw new XMLParseException("Exception when parsing XML", sxe);
      } 
      catch (ParserConfigurationException pce) {
          // Parser with specified options can't be built
          throw new XMLParseException("Exception when parsing XML", pce);
      }
      catch (IOException ioe) {
          // I/O error
          throw new XMLParseException("Exception when parsing XML", ioe);
      }

      return document.getDocumentElement();
  }

  
  /**
  * Parses a given String XML fragment. Higher level exceptions 
  * will be handled in callees.
  *
  * @param source String XML fragment to be parsed
  *
  * @return a DOM Element object, which is the root element of parsed xml file 
  *
  * @throws XMLParseException if any abnormal condition occurs when forming a DOM
  * object
  */
  public Element parseFragment(String source) throws XMLParseException{
       
      Document document = null;

      try {
          DocumentBuilder builder = factory.newDocumentBuilder();
          document = builder.parse(new InputSource(new StringReader(source)));
      } 
      catch (SAXException sxe) {
          // Error generated during parsing
          throw new XMLParseException("Exception when parsing XML", sxe);
      } 
      catch (ParserConfigurationException pce) {
          // Parser with specified options can't be built
          throw new XMLParseException("Exception when parsing XML", pce);
      }
      catch (IOException ioe) {
          // I/O error
          throw new XMLParseException("Exception when parsing XML", ioe);
      }
      
      return document.getDocumentElement();
  }


  /**
  * Locator. This method takes an element and returns all its "children" elements
  * having given name as a Vector.
  */
  public Vector locator(Element parent, String name){

      Vector result = new Vector(); 
      NodeList children = parent.getChildNodes();
      for(int i = 0; i < children.getLength(); i++){
          Node child = children.item(i);
          if( (child.getNodeType() == Node.ELEMENT_NODE ) && ( ((Element)child).getTagName().equals(name) )){
              result.addElement(child);
          }
      }
      return result;
  }

  /**
  * Returns content of a DOM Element node as a string. It recursively traverses, 
  * Element, Text and Attributes and prints them as xml string. 
  *
  * @param e Element, whose content will be returned
  * @return a String, which represents recursive content of an Element
  */
  public String getElementContent(Element e){
       StringBuffer result = new StringBuffer();
        for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
            switch (n.getNodeType()) {
            case Node.ELEMENT_NODE:
                
                result.append("<" + ((Element)n).getTagName());
            
                NamedNodeMap attributes = n.getAttributes();
                for(int i = 0; i < attributes.getLength() ; i++){
                    Attr attr = (Attr)attributes.item(i);
                    result.append( " " + attr.getName() + "=\"" + attr.getValue() + "\"");
                }   
      
                if(n.hasChildNodes())
                    result.append(">");
      
                result.append(getElementContent((Element)n));
                
                if(n.hasChildNodes())
                    result.append("</" + ((Element)n).getTagName() + ">");
                else
                    result.append("/>");
                
                break;
            case Node.TEXT_NODE:
                result.append(n.getNodeValue());
            }
        }
        return result.toString();
  }

}
