package tauzaman;

import java.rmi.*;
import java.rmi.server.*;

import java.net.*;
import java.util.*;


import tauzaman.calendricsystem.*;
import tauzaman.calendar.*;
import tauzaman.property.*;
import tauzaman.field.*;
import tauzaman.temporaldatatypes.*;
import tauzaman.timestamp.*;
import tauzaman.io.*;

/**
* <code>TauZamanRemoteServiceImpl</code> class implements <code>TauZamanRemoteService</code>
* remote interface and provides remote <code>Calendar</code> related services.
* It is instantiated via <code>TauZamanSystem</code>.
*
* <pre>
* // assume americanCalendricSystemUrl and defaultPropertyListUrl are alread declared and initialized
* TauZamanSystem tzs = new TauZamanSystem();
* // this gets a default remote service if there is a local server. See TauZamanSystem class for detailed remote service
* // acquisition.
* TauZamanRemoteService tzrs = tzs.getTauZamanRemoteService("american", americanCalendricSystemUrl, defaultPropertyListUrl);
* </pre>
*
* @see tauzaman.TauZamanSystem
*
* @author Curtis Dyreson and Bedirhan Urgun
* @version 0.1. 01/27/02
* @status design complete, implementation complete
*/
public class TauZamanRemoteServiceImpl extends UnicastRemoteObject implements TauZamanRemoteService{

  /** <code>TauZamanRemoteService</code> "has a" <code>TauZamanLocalService</code> **/
  private TauZamanLocalService tzls = null;

  
  /**
  * Constructs a <code>TauZamanRemoteService</code> object ready to serve.
  *
  * @param parentSystem parent <code>TauZamanSystem</code> of this <code>TauZamanLocalService</code>
  * @param calendricSystemName String name of defaultly loaded <code>CalendricSystem</code>
  * @param calendricSystemUrl URL of defaultly loaded <code>CalendricSystem</code>
  * @param defaultPropertyListUrl URL of defaultly loaded <code>Property</code> list
  *
  * @throws RemoteException if an abnormal condition occurs during remote connection
  *
  * @throws CalendricSystemFormationException if any abnormal condition occurs when
  * loading default <code>CalendricSystem</code>
  *
  * @throws PropertyFormationException if any abnormal condition occurs when
  * loading default <code>Property</code> list
  */
  protected TauZamanRemoteServiceImpl(TauZamanSystem parentSystem, String calendricSystemName, 
                                      URL calendricSystemUrl, URL defaultPropertyListUrl) 
                       throws RemoteException, CalendricSystemFormationException, PropertyFormationException{

      super();
      tzls = new TauZamanLocalService(parentSystem, calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);
  }

  /**
  * Test method for calculating RTT independent of any operation, but
  * depending on Java RMI.
  */ 
  public void dummy()throws RemoteException{}


  /**
  * Loads the <code>CalendricSystem</code>, whose specification file and default
  * <code>Property</code> list URLs are given.
  * 
  * @param calendricSystemName String name of loaded <code>CalendricSystem</code>
  * @param calendricSystemUrl URL of loaded <code>CalendricSystem</code>
  * @param defaultPropertyListUrl URL of loaded <code>Property</code> list
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * 
  * @throws CalendricSystemFormationException if any abnormal condition occurs when
  * loading <code>CalendricSystem</code>
  *
  * @throws PropertyFormationException if any abnormal condition occurs when
  * loading <code>Property</code> list
  */ 
  public void loadCalendricSystem(String calendricSystemName, URL calendricSystemUrl, URL defaultPropertyListUrl)
                            throws RemoteException, CalendricSystemFormationException, PropertyFormationException{
      tzls.loadCalendricSystem(calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);
  }
  

  /**
  * Sets given <code>CalendricSystem</code> name as active <code>CalendricSystem</code> of
  * this service.
  * 
  * @param calendricSystemName String name of <code>CalendricSystem</code> to be activated
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * 
  * @throws CalendricSystemServiceException if any abnormal condition occurs when setting active
  * <code>CalendricSystem</code>. Specifically it is thrown when provided a <code>CalendricSystem</code>
  * name, which is not already loaded.
  */
  public void setActiveCalendricSystem(String calendricSystemName) throws RemoteException, CalendricSystemServiceException{
      tzls.setActiveCalendricSystem(calendricSystemName);
  }
 

  /**
  * Returns name of active <code>CalendricSystem</code> of this service.
  *
  * @throws RemoteException if an abnormal condition occurs during remote connection
  *  
  * @return String name of active <code>CalendricSystem</code> of this service
  */
  public String getActiveCalendricSystemName() throws RemoteException{
      return tzls.getActiveCalendricSystemName();
  }
                                                       
  /**
  * Returns default <code>Granularity</code> of active <code>CalendricSystem</code>.
  * 
  * @return default <code>Granularity</code>
  * 
  * @throws RemoteException if any abnormal condition occurs during remote connection
  */
  public Granularity getActiveCalendricSystemDefaultGranularity() throws RemoteException{
      return tzls.getActiveCalendricSystemDefaultGranularity();
  }

  /**
  * Returns true if given <code>CalendricSystem</code> name is actually mapped
  * to a loaded <code>CalendricSystem</code> object. False otherwise.
  *
  * @param calendricSystemName String name of <code>CalendricSystem</code> that validity is checked.
  *
  * @throws RemoteException if an abnormal condition occurs during remote connection
  *
  * @return true if given <code>CalendricSystem</code> name is actually mapped
  * to a loaded <code>CalendricSystem</code> object. False otherwise.
  */ 
  public boolean isValidCalendricSystemName(String calendricSystemName) throws RemoteException{   
      return tzls.isValidCalendricSystemName(calendricSystemName);
  }


  /**
  * Returns an array of <code>Granule</code> objects formed by using input temporal constant and
  * given <code>Property</code> name.
  * 
  * @param input String that represents input temporal constant
  * @param propertyName name of the <code>Property</code> to be used when parsing input
  * 
  * @return array of <code>Granule</code> objects
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  *
  * @throws IOException if any error occurs while parsing an input
  * temporal constant
  */
  public Granule [] parseInput(String inputStr, String propertyName) throws RemoteException, IOException{
      return tzls.parseInput(inputStr, propertyName);
  }
  

  /**
  * Converts a timestamp (in terms of <code>Granule</code>(s)) into temporal
  * constant as string. Format of the output string, given <code>Property</code>'s
  * format template is used.
  * 
  * @param granules <code>Calendar</code> parsed timestamp
  * @param propertyName name of the <code>Property</code>, which is used to
  * format of the output string, temporal constant
  * @param cachedCalendricSystemName String name of the <code>CalendricSystem</code>,
  * which was used during the formation of this timestamp
  * 
  * @return formatted String, which is a temporal constant
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  *
  * @throws IOException if any error occurs 
  * while converting a timestamp into a temporal constant
  */
  public String parseOutput(Granule granules[], String propertyName, String cachedCalendricSystemName) 
                                  throws RemoteException, IOException{
      return tzls.parseOutput(granules, propertyName, cachedCalendricSystemName);
  }

  /* will be changed to package when we add temporaldatatypes */
  
  /**
  * Activates a <code>Property</code> from given Property specification file URL.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @param url URL of Property specification file
  * @param focusPropertyNames names of <code>Properties</code> that are wanted to be activated
  * residing in given file
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * 
  * @throws PropertyFormationException if any problem occurs when activating the <code>Property</code>
  * or if maximum number of allowable <code>Property</code>ies exceeded (not implemented yet).
  */
  public void propertyActivate(URL propertyUrl, String focusPropertyNames[]) throws RemoteException, PropertyFormationException{
      tzls.propertyActivate(propertyUrl, focusPropertyNames);
  }
  
  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  */
  public void propertyDeactivateAll() throws RemoteException{
      tzls.propertyDeactivateAll();
  }
  
  /**
  * Deactivates a <code>Property</code> whose has the given name.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>  
  * if calendricSystemName is null. Else it'll try to update <code>CalendricSystem</code>,
  * which corresponds to calendricSystemName.
  *
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param focusPropertyName String name of <code>Property</code> to be deactivated
  *  
  * @throws RemoteException if an abnormal condition occurs during remote connection  
  * @throws PropertyServiceException if the name is an invalid <code>Property</code> name
  */
  public void propertyDeactivate(String propertyName) throws RemoteException, PropertyServiceException{
      tzls.propertyDeactivate(propertyName);
  }
  
  /**
  * Sets all active <code>Properties</code> to their default values.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  */
  public void propertySetDefaultAll() throws RemoteException{
      tzls. propertySetDefaultAll();
  }
 
  /**
  * Sets <code>Property</code> (given by its name) to its default value. If the
  * <code>Property</code> already has its default value active, then keep silent.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  *  
  * @param name name of the <code>Property</code>
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyServiceException if the name is an invalid <code>Property</code> name
  */
  public void propertySetDefault(String propertyName) throws RemoteException, PropertyServiceException{
      tzls.propertySetDefault(propertyName);
  }
 
  /**
  * Gets all the current <code>Property</code> information of a specific
  * Property name in a String. Format of the return string is;
  * 
  * {urlOfProperty}%{urOfProperty}%....%{urlOfProperty}
  *
  * order is same as of Property stack from bottom to top.
  * 
  * @param name String name of a <code>Property</code>
  * @return a String containing current Properties activated for a
  * specific Property name
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyServiceException if any abnormal condition occurs when
  * getting the image of <code>Property</code> stack given its name
  */
  public String getPropertyStackImage(String propertyName) throws RemoteException, PropertyServiceException{
      return tzls.getPropertyStackImage(propertyName);
  }
 
  /**
  * Activates a <code>Property</code> from given URL of Property specification file.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param url URL of Property specification file
  * @param focusPropertyNames names of <code>Properties</code> that are wanted to be activated
  * residing in given file
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyFormationException if any problem occurs when activating the <code>Property</code>
  * or if maximum number of allowable <code>Property</code>ies exceeded (not implemented yet)
  * 
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName
  * is invalid
  */
  public void propertyActivate(String calendricSystemName, URL propertyUrl, String focusPropertyNames[])
                           throws RemoteException, PropertyFormationException, CalendricSystemServiceException{ 
      tzls.propertyActivate(calendricSystemName, propertyUrl,focusPropertyNames);
  }
  
  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to 
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */
  public void propertyDeactivateAll(String calendricSystemName) throws RemoteException, CalendricSystemServiceException{
      tzls.propertyDeactivateAll(calendricSystemName);
  }

  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  *
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName
  * is invalid
  */
  public void propertyDeactivate(String calendricSystemName, String propertyName) 
                  throws RemoteException, PropertyServiceException, CalendricSystemServiceException{
      tzls.propertyDeactivate(calendricSystemName, propertyName);
  }
  
  /**
  * Sets all active <code>Properties</code> to their default values.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  *  
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName
  * is invalid
  */
  public void propertySetDefaultAll(String calendricSystemName) throws RemoteException, CalendricSystemServiceException{
      tzls.propertySetDefaultAll(calendricSystemName);
  }
  
  /**
  * Sets <code>Property</code> (given by its name) to its default value. If the
  * <code>Property</code> already has its default value active, then keep silent.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  *
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @param focusPropertyName String name of the <code>Property</code>
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  * 
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName
  * is invalid
  */
  public void propertySetDefault(String calendricSystemName, String propertyName) 
                           throws RemoteException, PropertyServiceException, CalendricSystemServiceException{
      tzls.propertySetDefault(calendricSystemName, propertyName);
  }
  
  /**
  * Gets all the current <code>Property</code> information of a specific
  * Property name in a String. Format of the return string is;
  * 
  * {urlOfProperty}%{urOfProperty}%....%{urlOfProperty}
  * 
  * order is same as of Property stack from bottom to top.
  *
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param name String name of a <code>Property</code>
  * @return a String containing current Properties activated for a   
  * specific Property name
  * 
  * @throws RemoteException if an abnormal condition occurs during remote connection
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  * 
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName
  * is invalid
  */
  public String getPropertyStackImage(String calendricSystemName, String propertyName) 
                           throws RemoteException, PropertyServiceException, CalendricSystemServiceException{
      return tzls.getPropertyStackImage(calendricSystemName, propertyName);
  }

}
