package tauzaman;

import java.util.*;
import java.net.*;

import tauzaman.calendricsystem.*;
import tauzaman.calendar.*;
import tauzaman.property.*;
import tauzaman.field.*;
import tauzaman.temporaldatatypes.*;
import tauzaman.timestamp.*;
import tauzaman.io.*;

/**
* <code>TauZamanLocalService</code> class provides local <code>Calendar</code>
* related services. It is instantiated via <code>TauZamanSystem</code>.
* 
* <pre>
* // assume americanCalendricSystemUrl and defaultPropertyListUrl are alread declared and initialized
* TauZamanSystem tzs = new TauZamanSystem();
* TauZamanLocalService tzls = tzs.getTauZamanLocalService("american", americanCalendricSystemUrl, defaultPropertyListUrl);
* </pre>
*
* @see tauzaman.TauZamanSystem
*/
public class TauZamanLocalService{

  /* active CS is not kept as reference to CalendricSystem since
     there might be two same CS used twice with different names. 
     So, when we set one of CS as active and want to get the name
     of it, which one it will return is ambiguous  */

  /** user given name of active <code>CalendricSystem</code> **/
  private String activeCalendricSystemName = null;
  
  /** A mapping from user given <code>CalendricSystem</code> names to actual 
      <code>CalendricSystem</code> objects **/
  private Hashtable calendricSystemNamespace = null;

  /** A handle to <code>CalendricSystemRepository</code> of parent <code>TauZamanSystem</code> **/
  private CalendricSystemRepository csr = null;

  /** A handle to <code>PropertyRepository</code> of parent <code>TauZamanSystem</code> **/
  private PropertyRepository pr = null;

  /** A handle to <code>FVSupportRepository</code> of parent <code>TauZamanSystem</code> **/
  private FVSupportRepository fvsr = null;


  /* it would be better to use "package" access modifier in constructor */

  /**
  * Constructs a <code>TauZamanLocalService</code> object ready to serve. 
  *
  * @param parentSystem parent <code>TauZamanSystem</code> of this <code>TauZamanLocalService</code>
  * @param calendricSystemName String name of defaultly loaded <code>CalendricSystem</code>
  * @param calendricSystemUrl URL of defaultly loaded <code>CalendricSystem</code>
  * @param defaultPropertyListUrl URL of defaultly loaded <code>Property</code> list 
  *
  * @throws CalendricSystemFormationException if any abnormal condition occurs when 
  * loading default <code>CalendricSystem</code> 
  *
  * @throws PropertyFormationException if any abnormal condition occurs when 
  * loading default <code>Property</code> list
  */

  protected TauZamanLocalService(TauZamanSystem parentSystem, String calendricSystemName, 
                                      URL calendricSystemUrl, URL defaultPropertyListUrl) 
                                 throws CalendricSystemFormationException, PropertyFormationException{

      /* allocate memory for CalendricSystems */
      calendricSystemNamespace = new Hashtable();

      /* get the unique CalendricSystemRepository of this TauZaman system */
      csr = parentSystem.getCalendricSystemRepository();

      /* get the unique CalendricSystemRepository of this TauZaman system */
      pr = parentSystem.getPropertyRepository();

      /* get the unique FVSupportRepository of this TauZaman system */
      fvsr = parentSystem.getFVSupportRepository();
   
      /* load initial Calendric System */
      loadCalendricSystem(calendricSystemName, calendricSystemUrl, defaultPropertyListUrl);
  }


  /* this is for Granules, created by the user directly without providing
     a granularity. So, system should assign a default granularity to those
     granules. For now, those Granule constructors have the code, but they
     are commented out. */


  /**
  * Returns default <code>Granularity</code> of active <code>CalendricSystem</code>.
  *
  * @return <code>Granularity</code>
  */
  public Granularity getActiveCalendricSystemDefaultGranularity(){

      /* get the active Calendric System */
      CalendricSystem activeCalendricSystem = (CalendricSystem)calendricSystemNamespace.get(activeCalendricSystemName);

      /* return its default Granularity */
      return activeCalendricSystem.getDefaultGranularity();
  }



  /**
  * Loads the <code>CalendricSystem</code>, whose specification file and default 
  * <code>Property</code> list URLs are given.
  *  
  * @param calendricSystemName String name of loaded <code>CalendricSystem</code>
  * @param calendricSystemUrl URL of loaded <code>CalendricSystem</code>
  * @param defaultPropertyListUrl URL of loaded <code>Property</code> list 
  *
  * @throws CalendricSystemFormationException if any abnormal condition occurs when 
  * loading <code>CalendricSystem</code> 
  *
  * @throws PropertyFormationException if any abnormal condition occurs when 
  * loading <code>Property</code> list
  */
  public void loadCalendricSystem(String calendricSystemName, URL calendricSystemUrl, 
                                   URL defaultPropertyListUrl) 
                                 throws CalendricSystemFormationException, PropertyFormationException{

      /* check the name space first against duplicate names */
      if(calendricSystemNamespace.containsKey(calendricSystemName))
          throw new CalendricSystemFormationException("Exception when adding a new Calendric System name to namespace: " + 
                                                                calendricSystemName + "is already defined");

          /* if repository already has it, it'll return existing one */
      CalendricSystem cs = csr.loadCalendricSystem(calendricSystemUrl);
     
      /* load default properties for this calendric system */
      PropertyManager pm = new PropertyManager(pr, defaultPropertyListUrl);

      /* if everthing went well, form a calendricsystem-propertymanager pair by making use of a Vector 
         and update calendric system namespace */
      Vector v = new Vector();
      v.addElement(cs);
      v.addElement(pm);
      calendricSystemNamespace.put(calendricSystemName, v);

      /* this is the side effect of this method
         it'll set loaded Calendric System as active one */
      activeCalendricSystemName = calendricSystemName;
  }


  /**
  * Sets given <code>CalendricSystem</code> name as active <code>CalendricSystem</code> of
  * this service.
  * 
  * @param calendricSystemName String name of <code>CalendricSystem</code> to be activated
  * 
  * @throws CalendricSystemServiceException if any abnormal condition occurs when setting active 
  * <code>CalendricSystem</code>. Specifically it is thrown when provided a <code>CalendricSystem</code>
  * name, which is not already loaded.
  */
  public void setActiveCalendricSystem(String calendricSystemName) throws CalendricSystemServiceException{

      if(!calendricSystemNamespace.containsKey(calendricSystemName))
          throw new CalendricSystemServiceException("Exception when setting active calendric system: " + 
                               "No such calendric system name defined: " + calendricSystemName);

      activeCalendricSystemName = calendricSystemName;
  }

  /**
  * Returns name of active <code>CalendricSystem</code> of this service.
  *
  * @return String name of active <code>CalendricSystem</code> of this service
  */
  public String getActiveCalendricSystemName(){
      return activeCalendricSystemName;
  }

  /* For now we don't use these (getCalendricSystem(), getPropertyManager(), 
     getFVSupportRepository() ) methods */

  /**
  * Returns a handle to <code>CalendricSystem</code> object given its name.
  *
  * @param calendricSystemName String name of requested <code>CalendricSystem</code> object
  * 
  * @return a handle to <code>CalendricSystem</code> object
  * @see tauzaman.calendricsystem.CalendricSystem
  */
  protected CalendricSystem getCalendricSystem(String calendricSystemName){
      Vector v = (Vector)calendricSystemNamespace.get(calendricSystemName);
      return (CalendricSystem)v.elementAt(0);
  }

  /**
  * Returns a handle to <code>PropertyManager</code> object given its fellow
  * <code>CalendricSystem</code> name.
  * 
  * @param calendricSystemName String name of <code>CalendricSystem</code> object
  * 
  * @return a handle to <code>PropertyManager</code> object
  * @see tauzaman.property.PropertyManager
  */
  protected PropertyManager getPropertyManager(String calendricSystemName){
      Vector v = (Vector)calendricSystemNamespace.get(calendricSystemName);
      return (PropertyManager)v.elementAt(1);
  }

  /**
  * Returns a handle to <code>FVSupportRepository</code> of this service.
  *
  * @see tauzaman.field.FVSupportRepository
  */
  protected FVSupportRepository getFVSupportRepository(){
      return fvsr;
  }


  /* parseInput() and parseOutput() should have smt. like "package"
     access modifier. We say "something like" since "package" 
     does not provide what we want. We want an access modifier,
     which can include sub-packages. The problem is temporal data type
     package calls these methods. So, we want these to be visible
     to temporal data type package but not to user, who is outside
     of tauzaman package. */

  /* but having these public does not hurt anyone, I beleive. */

  /**
  * Returns an array of <code>Granule</code> objects formed by using input temporal constant and
  * given <code>Property</code> name.
  *
  * @param input String that represents input temporal constant
  * @param propertyName name of the <code>Property</code> to be used when parsing input
  *
  * @return array of <code>Granule</code> objects
  *
  * @throws IOException if any error occurs while parsing an input
  * temporal constant
  */    
  public Granule [] parseInput(String inputStr, String propertyName) throws IOException{

      CalendricSystem activeCalendricSystem = getCalendricSystem(getActiveCalendricSystemName());
      PropertyManager activePropertyManager = getPropertyManager(getActiveCalendricSystemName());

      Input input = new Input(activeCalendricSystem, activePropertyManager, fvsr);
      
      Granule granules [] = input.parseInput(inputStr, propertyName);

      return granules;
  }

  /**
  * Converts a timestamp (in terms of <code>Granule</code>(s)) into temporal
  * constant as string. Format of the output string, given <code>Property</code>'s
  * format template is used.
  *
  * @param granules <code>Calendar</code> parsed timestamp
  * @param propertyName name of the <code>Property</code>, which is used to
  * format of the output string, temporal constant
  *
  * @return formatted String, which is a temporal constant
  *
  * @throws IOException if any error occurs
  * while converting a timestamp into a temporal constant
  */
  public String parseOutput(Granule granules[], String propertyName, String cachedCalendricSystemName) throws IOException{

      /* if cachedCalendricSystemName is null, that means a user formed a TauZamanLocalService |
         TauZamanLocalService and without forming any temporal data type, wants to parse given
         Output. In this case, for outputting, we'll use currently active calendric system name, 
         instead of throwing an exception. */

      if(cachedCalendricSystemName == null)
          cachedCalendricSystemName = getActiveCalendricSystemName();

      CalendricSystem activeCalendricSystem = getCalendricSystem(cachedCalendricSystemName);
      PropertyManager activePropertyManager = getPropertyManager(cachedCalendricSystemName);

      Output output = new Output(activeCalendricSystem, activePropertyManager, fvsr);

      String outputStr = output.parseOutput(granules, propertyName);

      return outputStr;
  }

  /**
  * Returns true if given <code>CalendricSystem</code> name is actually mapped
  * to a loaded <code>CalendricSystem</code> object. False otherwise.
  *
  * @param calendricSystemName String name of <code>CalendricSystem</code> that validity is checked.
  *
  * @return true if given <code>CalendricSystem</code> name is actually mapped
  * to a loaded <code>CalendricSystem</code> object. False otherwise.
  */
  public boolean isValidCalendricSystemName(String calendricSystemName){
      if(calendricSystemNamespace.containsKey(calendricSystemName))
          return true;
      return false;
  }



  /***PROPERTY OPERATIONS***/

        /****PROPERTY OPERATIONS WITH ACTIVE CALENDRIC SYSTEM****/

  /**
  * Activates a <code>Property</code> from given Property specification file URL.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @param url URL of Property specification file
  * @param focusPropertyNames names of <code>Properties</code> that are wanted to be activated
  * residing in given file. If null, all of the <code>Properties</code> in the file, pointed by url,
  * will be activated.
  * 
  * @throws PropertyFormationException if any problem occurs when activating the <code>Property</code>
  * or if maximum number of allowable <code>Property</code>ies exceeded (not implemented yet).
  */
  public void propertyActivate(URL propertyUrl, String focusPropertyNames[]) throws PropertyFormationException{
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      propertyActivateInternal(pm, propertyUrl, focusPropertyNames);
  }

  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  */ 
  public void propertyDeactivateAll(){
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      propertyDeactivateAllInternal(pm);
  }
 
  /**
  * Deactivates a <code>Property</code> whose has the given name.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code> 
  * if calendricSystemName is null. Else it'll try to update <code>CalendricSystem</code>,
  * which corresponds to calendricSystemName.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param focusPropertyName String name of <code>Property</code> to be deactivated
  * 
  * @throws PropertyServiceException if the name is an invalid <code>Property</code> name
  */
  public void propertyDeactivate(String propertyName) throws PropertyServiceException{
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      propertyDeactivateInternal(pm, propertyName);
  }

  /**
  * Sets all active <code>Properties</code> to their default values.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  */  
  public void propertySetDefaultAll(){
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      propertySetDefaultAllInternal(pm);
  }

  /**
  * Sets <code>Property</code> (given by its name) to its default value. If the
  * <code>Property</code> already has its default value active, then keep silent.   
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  *
  * @param name name of the <code>Property</code>
  * 
  * @throws PropertyServiceException if the name is an invalid <code>Property</code> name
  */ 
  public void propertySetDefault(String propertyName) throws PropertyServiceException{
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      propertySetDefaultInternal(pm, propertyName);
  }


  /**
  * Gets all the current <code>Property</code> information of a specific
  * Property name in a String. Format of the return string is;
  *
  * {urlOfProperty}%{urOfProperty}%....%{urlOfProperty}
  *
  * order is same as of Property stack from bottom to top.
  *
  * @param name String name of a <code>Property</code>
  * @return a String containing current Properties activated for a
  * specific Property name
  *
  * @throws PropertyServiceException if any abnormal condition occurs when
  * getting the image of <code>Property</code> stack given its name
  */
  public String getPropertyStackImage(String propertyName) throws PropertyServiceException{
      PropertyManager pm = getPropertyManager(activeCalendricSystemName);
      return getPropertyStackImageInternal(pm, propertyName);
  }



        /****PROPERTY OPERATIONS WITH A PARTICULAR CALENDRIC SYSTEM****/

  /**
  * Activates a <code>Property</code> from given URL of Property specification file.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param url URL of Property specification file
  * @param focusPropertyNames names of <code>Properties</code> that are wanted to be activated
  * residing in given file
  * 
  * @throws PropertyFormationException if any problem occurs when activating the <code>Property</code>
  * or if maximum number of allowable <code>Property</code>ies exceeded (not implemented yet)
  *
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */ 
  public void propertyActivate(String calendricSystemName, URL propertyUrl, String focusPropertyNames[]) 
                                 throws PropertyFormationException, CalendricSystemServiceException{

      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                     calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      propertyActivateInternal(pm, propertyUrl, focusPropertyNames);
  }


  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of active <code>CalendricSystem</code>.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */ 
  public void propertyDeactivateAll(String calendricSystemName) throws CalendricSystemServiceException{

      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                 calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      propertyDeactivateAllInternal(pm);
  }

  /**
  * Deactivates and removes any active <code>Properties</code>. It will not deactivate
  * or remove any default <code>Property</code>, so any attempt will only succeed to
  * deactivate and remove only non-default active <code>Properties</code>.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  *
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */ 
  public void propertyDeactivate(String calendricSystemName, String propertyName) 
                              throws PropertyServiceException, CalendricSystemServiceException{

      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                     calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      propertyDeactivateInternal(pm, propertyName);
  }


  /**
  * Sets all active <code>Properties</code> to their default values.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */
  public void propertySetDefaultAll(String calendricSystemName) throws CalendricSystemServiceException{

      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                     calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      propertySetDefaultAllInternal(pm);
  }

  /**
  * Sets <code>Property</code> (given by its name) to its default value. If the
  * <code>Property</code> already has its default value active, then keep silent.
  * This method updates <code>PropertyManager</code> of <code>CalendricSystem</code>,
  * given its name.
  * 
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * 
  * @param focusPropertyName String name of the <code>Property</code>
  * 
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  *
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */
  public void propertySetDefault(String calendricSystemName, String propertyName)
                              throws PropertyServiceException, CalendricSystemServiceException{
      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                     calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      propertySetDefaultInternal(pm, propertyName);
  }

  /**
  * Gets all the current <code>Property</code> information of a specific
  * Property name in a String. Format of the return string is;
  *
  * {urlOfProperty}%{urOfProperty}%....%{urlOfProperty}
  *
  * order is same as of Property stack from bottom to top.
  *
  * @param calendricSystemName String name, which should point to a URL of <code>CalendricSystem</code>
  * @param name String name of a <code>Property</code>
  * @return a String containing current Properties activated for a
  * specific Property name
  *
  * @throws PropertyServiceException if given name is not a valid <code>Property</code> name
  *
  * @throws CalendricSystemServiceException if given <code>CalendricSystem</code> calendricSystemName 
  * is invalid
  */
  public String getPropertyStackImage(String calendricSystemName, String propertyName)
                              throws PropertyServiceException, CalendricSystemServiceException{
      if(!isValidCalendricSystemName(calendricSystemName))
          throw new CalendricSystemServiceException("Exception: no such calendric system name defined: " + 
                                                                                     calendricSystemName);

      PropertyManager pm = getPropertyManager(calendricSystemName);
      return getPropertyStackImageInternal(pm, propertyName);
  }

        /****PROPERTY OPERATIONS****/

  /**
  * Internal use...
  */ 
  private void propertyActivateInternal(PropertyManager pm, URL propertyUrl, String focusPropertyNames[]) 
                                                 throws PropertyFormationException{
      pm.propertyActivate(propertyUrl, focusPropertyNames);
  }

  /**
  * Internal use...
  */ 
  private void propertyDeactivateAllInternal(PropertyManager pm){
      pm.propertyDeactivateAll();      
  }

  /**
  * Internal use...
  */ 
  private void propertyDeactivateInternal(PropertyManager pm, String propertyName) throws PropertyServiceException{
      pm.propertyDeactivate(propertyName);
  }

  /**
  * Internal use...
  */ 
  private void propertySetDefaultAllInternal(PropertyManager pm){
      pm.propertySetDefaultAll();  
  }

  /**
  * Internal use...
  */ 
  private void propertySetDefaultInternal(PropertyManager pm, String propertyName) throws PropertyServiceException{
      pm.propertySetDefault(propertyName);
  }

  /**
  * Internal use...
  */ 
  private String getPropertyStackImageInternal(PropertyManager pm, String propertyName) throws PropertyServiceException{
      return pm.getPropertyImage(propertyName);
  }

  /***PROPERTY OPERATIONS***/

  /**
  * Returns String representation of this <code>TauZamanLocalService</code>.
  * @return string representation of active <code>CalendricSystem</code> of this 
  * <code>TauZamanLocalService</code>
  */
  public String toString(){
      return "\nTauZamanLocalService: " + getCalendricSystem(activeCalendricSystemName).toString() ;
  }
}
