/**
 * 
 */
package cs.arizona.tau.xml;

import org.w3c.dom.*;
import org.w3c.dom.ls.*;

import cs.arizona.util.*;

import org.w3c.dom.xpath.XPathEvaluator;
import org.w3c.dom.xpath.XPathNSResolver;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.Hashtable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cs.arizona.tau.time.ITimePeriod;
import cs.arizona.tau.time.TimePeriod;
import cs.arizona.tau.time.ITemporalRegion;
import cs.arizona.tau.time.TemporalRegion;
import cs.arizona.tau.time.ITemporalElement;
import cs.arizona.tau.time.TemporalElement;
import cs.arizona.tau.time.ITime;
import cs.arizona.util.ConventionalParser;
import cs.arizona.util.SchemaPathEvaluator;
import cs.arizona.util.TauLogger;
import cs.arizona.util.ValidatorProperties;


/**
 * @author sthomas
 * Static Class to perform functions common to all tools.
 * TODO: Add comments.
 * TODO: Create standardized list of exit codes (instead of hard-coded).
 */


public class Common{
	
	public static boolean validateAnnotations(LogicalAnnotationValidator lav, PhysicalAnnotationValidator pav){
		if (lav.validateLogicalAnnotation() == false) {
			TauLogger.logger.fatal("Temporal Annotations did not validate.");
			System.exit(3); //TODO: something smarter
		}
		
		if (pav.validatePhysicalAnnotation() == false) {
			TauLogger.logger.fatal("Physical Annotations did not validate.");
			System.exit(3); //TODO: something smarter
		}
		
		// TODO: Can we also check to see if all physical annotations are above all temporal annotations?
		
		
		// If we've made it here, then we can return true.
		return true;
	}
	
	public static void checkFileExists(String path){
		File f = new File(path);
	    if (!f.exists() || !f.canRead()){
	    	TauLogger.logger.fatal("Cannot access file \"" + path + "\"");
	    	System.exit(4); //TODO: something smarter
	    }
	}
	
	public static Element getNextElement(Node a){
		NodeList nl = a.getChildNodes();
		for(int i=0 ; i<nl.getLength() ; i++){
			if (isElement(nl.item(i))){
				return (Element)nl.item(i);
			}
		}
		return null;
	}
	
	public static boolean isElement(Node a){
		return (a.getNodeType() == Node.ELEMENT_NODE);
	}
	
	public static boolean isNamed(Node a, String s){
		return a.getLocalName().equals(s);
	}
	
	public static void deleteFile(String path){
		File f = new File(path);

	    // Make sure the file or directory exists and isn't write protected
	    if (!f.exists()){
	      TauLogger.logger.fatal("Delete: no such file or directory: " + path);
	      return;
		}
	    
	    if (!f.canWrite()){
	    	TauLogger.logger.fatal("Delete: write protected: " + path);
	    	return;
	    }

	    boolean success = f.delete();

	    if (!success){
	    	TauLogger.logger.fatal("Delete: deletion failed");
	    	return;
	    }
	}

	
	
	/**
	 * @author sthomas
	 * @param path
	 * Adds <tXnl> tags for each newline in a given file.
	 */
	public static void filterEditInput(String path, String newPath){
		String command = "";
		int count = getNumberOfLines(path);
		
		// Make a copy of the file
		command  = "cp " + path + " " + newPath + "\n";
		
		// Run filter on new file to append <tXnl/> elements in the document.
		command += "perl -pi -e '$. > 1 && $. < " + count + " &&  s/\\n/<tXnl\\/>/g' " + newPath;

		outputStringToFile(command, "tmpScript.sh");
        runCommand("sh tmpScript.sh");
        deleteFile("tmpScript.sh");
	}
	
	
	/**
	 * @author sthomas
	 * @param path
	 * Adds newlines from <tXnl> tags to a given file.
	 */
	public static void filterEditOutput(String path){
		String command = "";

		// First need to remove all newlines from file that might have been output by DOM.
	    command  = "perl -pi -e 's/\n//g' " + path + "\n";
	    
	    // Then we need to make sure that the <?xml> tag is all by itself on the first line.
	    command += "perl -pi -e '$. == 1 && s/<\\?xml.*?>/<\\?xml version=\\\"1.0\\\"\\?>\\n/g' " + path + "\n";
	    
	    // Now replace all tags with newlines.
		command += "perl -pi -e '$. > 1 &&  s/<tXnl\\/>/\\n/g' " + path;
								
		outputStringToFile(command, "tmpScript.sh");
        runCommand("sh tmpScript.sh");
        deleteFile("tmpScript.sh");
	}
	

	/**
	 * @author sthomas
	 * @param path
	 * @return Number of lines in the file pointed to by path.
	 */
	public static int getNumberOfLines(String path){
		int count = 0;
		try{
			FileReader fr = new FileReader(path);
		    LineNumberReader ln = new LineNumberReader(fr);
		  
		    while (ln.readLine() != null){
		    	count++;
		    }
		} catch (IOException e) {
            System.out.println("Exception reading number of lines: ");
            e.printStackTrace();
            System.exit(-1);
        }
		
		return count;
		
	}
	
	
	/**
	 * @author sthomas
	 * @param path
	 * Adds newlines from <tXnl> tags to a given file.
	 */
	public static void runPatchCommand(Element a, String path1, String path2){
		
		TauLogger.logger.debug("Patching " + path2 + " from " + path1);	
		
		// Copy last file
		runCommand("cp " + path1 + " " + path2);
		
		String editScript = "";
		Node b;
		NodeList nl2= a.getChildNodes();
		for (int j=0 ; j<nl2.getLength() ; j++){
			b = nl2.item(j);
			if(Common.isElement(b)){	
				// Convert "change" => "c", "add" => "a", etc
				String commandType = b.getNodeName();
				if (commandType.equals("change")){
					commandType = "c";
				} else if (commandType.equals("add")){
					commandType = "a";
				} else if (commandType.equals("delete")){
					commandType = "d";
				}
			
			
				// Need to replace ASCII codes to newlines in diff script
				String changeText = ((Element)b).getTextContent() + "\n";
				changeText = changeText.replaceAll("&&#10;", "\n");
						
				// Build the patch edit script (duplicate diff -e output)
				editScript += ((Element)b).getAttribute("lines");
				editScript += commandType + "\n";
				editScript += changeText;
				editScript += ".\n";
			}
		}
		
		
		outputStringToFile(editScript, "tmpPatch");
	
		String command = "patch " + path2 + " < tmpPatch";
		
		outputStringToFile(command, "tmpScript.sh");
        runCommand("sh tmpScript.sh");
        deleteFile("tmpScript.sh");
        deleteFile("tmpPatch");
	}
	
	
	/**
	 * @author sthomas
	 * @param path1
	 * @param path2
	 */
	public static void runDiffCommand(
			String path1, String path2, 
			Document temporalDoc, Element root2, 
			String beginDate, String endDate)
	{
		
        // For edits, need to run diff command and parse its output. 
        // Output looks something like:
        // 4c
        //    <age>24</age>
        // .
        //
        // So, need to look for three types of lines: beginning (to get line number and command
        // type), middle (the actual changes), and end marker.
		
		String command;		
		String s = null, lines = "", action = "", changeText = "";
		
		// Matching diff output
    	Pattern pEnd      = Pattern.compile("^.$");
    	Pattern pBegin    = Pattern.compile("(^[0-9]+,?[0-9]*)([cad])");
    	Matcher mBegin, mEnd;
    	
    	ITimePeriod transactionPeriod = new TimePeriod(beginDate, endDate);
		Element time 	= transactionPeriod.toXML(ITimePeriod.TRANSACTION_TIME, ITimePeriod.EXTENT_REP, temporalDoc);
		
		
		try{
			command = "diff -e " + path1 + " " + path2;
			TauLogger.logger.debug("Executing command " + command);
            Process pr = Runtime.getRuntime().exec(command);
            BufferedReader stdInput  = new BufferedReader(new InputStreamReader(pr.getInputStream()));
            BufferedReader stdError  = new BufferedReader(new InputStreamReader(pr.getErrorStream()));
            Element newEdit = null;
            Boolean first = false;
            
            while ((s = stdInput.readLine()) != null) {
            	//System.out.println(s);
            	mBegin  = pBegin.matcher(s);
            	mEnd 	= pEnd.matcher(s);
            	
            	// Edit begin line
            	if (mBegin.matches()){
            		lines  = mBegin.group(1); 
            		action = mBegin.group(2);
            		
            		if (action.equals("c")){
            			newEdit = temporalDoc.createElement("change");
            		} else if (action.equals("a")){ 
            			newEdit = temporalDoc.createElement("add");
            		} else if (action.equals("d")){ 
            			newEdit = temporalDoc.createElement("delete");
            		} else{
            			TauLogger.logger.fatal("Unknown diff output."); 
            			System.exit(5);
            		}

            		newEdit.setAttribute("lines", lines);
            		//newEdit.setAttribute("at", beginDate);
            		first = true;
            		changeText = "";
            	}
            	
            	// Termination line: build add edit element to timestamp node
            	else if (mEnd.matches()){ 
            		 newEdit.setTextContent(changeText);
            		 time.appendChild(newEdit);
            	}
            	
            	// Regular line
            	else {
            		// If this isn't the first line of input, append a newline in there.
            		String newSt = s;
            		if (first){
            			first = false;
            		}
            		else{
            			changeText += "&&#10;";
            		}
            		changeText += newSt;
            	}
            }
            while ((s = stdError.readLine()) != null) {
            	System.out.println(s);
            }

        
		} catch (IOException e) {
            System.out.println("Exception happened - here's what I know: ");
            e.printStackTrace();
            System.exit(-1);
        }
		
		root2.appendChild(time);
		
		
	}
	
	
	/**
	 * @author sthomas
	 * @param command
	 * Runs the command and outputs stdout, stderr.
	 */
	public static void runCommand(String command){
		try{
		    Process pr = Runtime.getRuntime().exec(command);
	        
			BufferedReader stdInput  = new BufferedReader(new InputStreamReader(pr.getInputStream()));
	        BufferedReader stdError  = new BufferedReader(new InputStreamReader(pr.getErrorStream()));
	        String s;
			while ((s = stdInput.readLine()) != null) {
				TauLogger.logger.info(s);
			}while ((s = stdError.readLine()) != null) {
				TauLogger.logger.info(s);
			}
		} catch (IOException e) {
            System.out.println("Exception running system command: ");
            e.printStackTrace();
            System.exit(-1);
        }
	}
	
	
	/**
	 * @author sthomas
	 * @param str
	 * @param path
	 */
	public static void outputStringToFile(String str, String path){
		try{
			BufferedWriter out = new BufferedWriter(new FileWriter(path));
			out.write(str);
			out.close();
		} catch (IOException e) {
			System.out.println("Exception outputting file: ");
			e.printStackTrace();
			System.exit(-1);
		}
	}
	
	
	/*
	 * @author sthomas
	 */
	public static String getOriginalRootName(Document d){
		Element tr    	= d.getDocumentElement();	
		
		String result = tr.getAttribute("originalRoot");
		return result;
	}
	
	/*
	 * @author sthomas
	 * Formats a Date object into the following format; 20081122 for Nov 22, 2008.
	 */
	public static String formatDate2(Date a){
		
		//TODO: Fix calls to deprecated methods.
		String result = String.format("%04d-%02d-%02d",   (a.getYear() + 1900), (a.getMonth() + 1), a.getDate());
		return result;
	}
	
	
	/*
	 * @author sthomas
	 * Formats a Date object into the following format; 20081122 for Nov 22, 2008.
	 */
	public static String formatDate(Date a){
		
		//TODO: Fix calls to deprecated methods.
		String result = String.format("%04d%02d%02d",   (a.getYear() + 1900), (a.getMonth() + 1), a.getDate());
		return result;
	}
	
	
	/**
	 * @author sthomas
	 * @param path
	 * @return
	 */
	public static String getDirectory(String path){
		try{
			File myFile = new File( path );
			String ans = myFile.getCanonicalPath();
			return ans.substring(0,ans.lastIndexOf("/")) + "/";
			//myFile.getPath()
		} catch (IOException e){
			e.printStackTrace();
			System.out.println(e);
		}
		return null;
	}
	
	
	
	/**
	 * @author sthomas
	 * @param path
	 * @return
	 */
	public static String getFileName(String path){
		File myFile = new File( path );
		return myFile.getName();
	}
	
	
	/**
	 * @author sthomas
	 * @param path
	 * @return
	 */
	public static Element getFirstElementNamed(Document d, String name){
		Element e;
		Element root = (Element)d.getFirstChild();
		
		if (isNamed(root, name)){
			return root;
		}
		
		NodeList nl1 = root.getElementsByTagName(name);
		return (Element)nl1.item(0);

	}
	
	
	/**
	 * @author sthomas
	 * @param path
	 * @return
	 */
	public static String getLastElementNameFromString(String n){
		return n.substring(n.lastIndexOf("/")+1, n.length());
	}
	
	
	
	/**
	 *	The function creates a Default Physical annotation, in which timestamp is represented only at the top level.
	 *  Initially when different documents are merged initially, only top level element is shown to be time varying
	 *  This is a kind of most basic physical annotation. It is then passed to the function physicalToTemporalConversion
	 *  of Primitives and then the timestamps are pushed down till the elements in temporal annotation.    
	 */
	public static Document createDefaultAnnotationDoc(ConventionalParser cp, String topLevelElementName){

		Document defaultAnnotationDoc 	= cp.createDocument(null, "annotationSet", null); 
		
        Element rootElement 			= defaultAnnotationDoc.getDocumentElement();
        Element physicalElement			= defaultAnnotationDoc.createElement("physical");
        Element stampElement 			= defaultAnnotationDoc.createElement("stamp");				/* create stampElement */
        Element stampKindElement 		= defaultAnnotationDoc.createElement("stampKind");		/* create stampKindElement */
        
        stampKindElement.setAttribute("timeDimension","transactionTime");						/* Set attributes of stampKindElement */
        stampKindElement.setAttribute("stampBounds","extent");
        
        stampElement.appendChild(stampKindElement);												/* Append stampKindElement to stampElement */
        stampElement.setAttribute("target",topLevelElementName);							/* Set attributes of stampElement */	
        stampElement.setAttribute("dataInclusion","expandedVersion");
        
        physicalElement.appendChild(stampElement);
        
        rootElement.appendChild(physicalElement);				/* Append stampElement as child of rootElement */
    
        return defaultAnnotationDoc;
	} 
	
	/**
	 * The function needs to be implemented. It should analyze the snapshotSchemaDoc and find out the top level element.
	 * The method used in the old implementation of SchemaMapper (by earlier group) could also be used.
	 */
	public static String getTopLevelSchemaElementName(TargetIdentifier ti){
		String name = ti.getTopLevelElement().getAttribute("name");
		return name;
	}

	/**
	 */
	public static void printElaspedTime(String s, long t1, long t2){
	    double e = (t2 - t1)/1000000000F;	
        //System.out.printf("%l %l\n", t1, t2);
        //System.out.printf("%s: elapsed time: %.8f\n", s, e);
	}
	
	
	
	
	
}
