package cs.arizona.tau.docs;

import java.io.File;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.ls.LSException;

import cs.arizona.tau.xml.Common;
import cs.arizona.util.ConventionalParser;
import cs.arizona.util.TauLogger;
import cs.arizona.util.ValidatorProperties;

public class TemporalSchema {
  /**
   * The Temporal Schema's internal representation	
   * @param vp The ValidatorProperties instance
   * @param cp The ConventionalParser instance which parses an XML document
   * @param temporal_schema_name
   */
  public TemporalSchema(ValidatorProperties vp,
                        ConventionalParser cp,
                        String temporal_schema_name) {
    ParseTemporalSchema(vp, cp, temporal_schema_name);
  }
  
  
  /**
   * Parse the temporal schema document. This is called directly by the
   * constructor.
   * @param vp The ValidatorProperties instance
   * @param cp The ConventionalParser instance which parses an XML document
   * @param temporal_schema_name The name of the temporal schema file
   * @return True if the temporal schema document is properly parsed
   */
  private boolean ParseTemporalSchema(ValidatorProperties vp,
                                      ConventionalParser cp,
                                      String temporal_schema_name){
    NodeList conventionalSchemaNode;
    NodeList conventionalSchemas;
    NodeList annotationNode;
    temporal_schema_name_ = temporal_schema_name;
    if (!temporal_schema_name.contains("/")) {
      temporal_schema_dir_ = "./";
    } else {
      temporal_schema_dir_ =
    	  temporal_schema_name.substring(
              0, temporal_schema_name.lastIndexOf("/")) + "/";
    }
    long t1 = System.nanoTime();
    try {
      temporal_schema_doc_ =
          cp.parseDocument(temporal_schema_name, null,
        		           vp.getProperty("TSSchema"));
      //TODO: Should we require a namespace? For now, allow any.
      conventionalSchemaNode =
          temporal_schema_doc_.getElementsByTagNameNS(
              "*", "conventionalSchema");
      if (conventionalSchemaNode.getLength() > 0){
        conventionalSchemas = conventionalSchemaNode.item(0).getChildNodes();
        if (conventionalSchemas.getLength() > 0) {
          for(int i= 0; i < conventionalSchemas.getLength(); i++) {
            if(Common.isElement(conventionalSchemas.item(i))
               && ((Element) conventionalSchemas.item(i)).getTagName().
                   equals("include")) {
              //Get name of conventionalSchema documents.
              conventional_schema_name_ =
            	  temporal_schema_dir_ + conventionalSchemas.item(i).
            	      getAttributes().getNamedItem("schemaLocation").
            	          getNodeValue();
              conventional_schema_doc_ =
                  cp.parseDocument(conventional_schema_name_, null,
                                   vp.getProperty("XMLSchema"), false);
            }
          }
        } else{
          System.out.println("conventionalSchemas length = 0");
        }
      } else{
        TauLogger.logger.info(
            "No \"conventionalSchema\" elements in temporal schema!");
        //TODO: Err in some way
      }
      annotationNode =
          temporal_schema_doc_.getElementsByTagNameNS("*", "annotationSet");
      if(annotationNode.getLength() > 0) {
        annotationNode = annotationNode.item(0).getChildNodes();
        if(annotationNode.getLength() > 0) {
          for(int i = 0; i < annotationNode.getLength(); i++) {
            if(Common.isElement(annotationNode.item(i))
               && ((Element) annotationNode.item(i)).getTagName().
                   equals("include") ) {
              annotation_name_ =
            	  temporal_schema_dir_ + annotationNode.item(i).getAttributes().
                      getNamedItem("location").getNodeValue();
              annotation_doc_ = cp.parseDocument(
                  annotation_name_, null, vp.getProperty("ASchema"));
            }
          }
        } else {
          // TODO: Err in some way
        }
      } else {
        TauLogger.logger.info(
            "No \"annotationSet\" element in temporal schema");
      }
    } catch(DOMException de){
      System.out.println(
          "Error occurred while parsing Temporal Bundle Document.");
      return false;
    } catch(LSException le){
      System.out.println("Unable to load Temporal Bundle Document.");
      le.printStackTrace();
      return false;
    } catch(Exception e){
      System.out.println("Some other error occurred.");
      e.printStackTrace();
      return false;
    }
    long t2 = System.nanoTime();
    Common.printElaspedTime("parseSchema", t1, t2);
    return true;
  }
  
  
  /**
   * 
   * @return the DOM document of the temporal schema
   */
  public Document getTemporalSchemaDoc() {
    return temporal_schema_doc_;
  }
  
  
  /**
   * 
   * @return the DOM document of the conventional schema referenced by this
   * temporal schema document
   */
  public Document getConventionalSchemaDoc() {
    return conventional_schema_doc_;
  }
  
  
  /**
   * 
   * @return the DOM document of the annotation document referenced by this
   * temporal schema document
   */
  public Document getAnnotationDoc() {
    return annotation_doc_;
  }
  
  
  /**
   * 
   * @return the name of the annotation file
   */
  public String getAnnotationName() {
    return annotation_name_;
  }
  
  
  /**
   * 
   * @return the name of the conventional schema file
   */
  public String getConventionalSchemaName() {
    return conventional_schema_name_;
  }
  
  
  /**
   * 
   * @return the directory where this temporal schema file is located
   */
  public String getTemporalSchemaDir() {
    return temporal_schema_dir_;
  }
  
  
  public String getTemporalSchemaName() {
	    return temporal_schema_name_;
  }
  
  private Document temporal_schema_doc_;
  private Document annotation_doc_;
  private Document conventional_schema_doc_;
  private String conventional_schema_name_;
  private String annotation_name_;
  private String temporal_schema_dir_;
  private String temporal_schema_name_;
}
