/*
 *	tGenerator - Simulates a temporal document, based on XBench.
 *  Copyright (C) 2010  Stephen W. Thomas
 *
 *   This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/*
 *
 *  Last updated November 2010
 *  
 *  Stephen W. Thomas
 *  sthomas@cs.queensu.ca
 *  School of Computing, Queen's University
 *
 *
 *	For more information, see the accompanying tech report:
 *  Stephen W. Thomas. "tauXencha: Extending Xdench with Time",
 *  TimeCENTER TR-92. October 2010.
 *  
 *
 *  Instructions for use:
 *  1. Download Xerces Java version 2.2.1; place the jar in the current directory.
 *     Download tDom; place the jar in the current directory.
 *     Run XBench to create a catalog.xml file.
 *     Set the parameters in this file (see below).
 *
 *  2. Compile the program with:
 *      javac tGenerator.java -cp ./xerces-2.2.1.jar
 *
 *	3. Run the program with:
 *		java tGenerator settings.txt
 * 
 */

import org.apache.xerces.parsers.DOMParser;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;

import org.w3c.dom.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringWriter;

import java.text.SimpleDateFormat;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;



public class tGenerator
{
	///////////////////////////////////////////////////////////////////////
	///  BEGIN SIMULATION PARAMETERS (with defaults)
	//////////////////////////////////////////////////////////////////////
	static double INITIAL_PERCENTAGE	    = 10;  				// Percentage of catalog.xml that is used in initial snapshot
	static int OUTPUT_XML_SLICES	    = 0;  				// Output each slice?
	static int OUTPUT_RELATION_SLICES	= 0;  				// Output each slice?
	static int OUTPUT_XML    			= 0;  				// Output each slice?
	static int OUTPUT_RELATIONS	    	= 0;  				// Output each slice?
	static String START_TIME  		    = "2010-01-01";		// 
	static String FOREVER_TIME		    = "2099-12-31";		// 
	static int TIME_STEP   			    = 5;				// Days in a time step
	static int REQUIRED_CHANGES 		= 600000;		    // Sum of all changes to all elements
	static String OUTPUT_DIR       	    = "output";			// Directory for output files
	static String INPUT_FILE       	    = "catalog.xml";	// Name of input file
	static String NULL_VALUE			= "";				// DBMS-specific null value (for PSM)
	static String SELECTION_TYPE		= "uniform";		// Distribution type.
	static double SELECTION_STDDEV		= 10.0;				// Standard distribution if SELECTION_TYPE is "gaussian"
	
	static int[] ITEM_PER_TIME_STEP 	= new int[3];		// Number of items to {insert, delete, update}
	static int[] AUTHOR_PER_TIME_STEP 	= new int[3];		// Number of items to {insert, delete, update}
	static int[] PUBLISHER_PER_TIME_STEP= new int[4];		// Number of items to {empty, empty, update, delete/insert}
	static int[] RELATED_PER_TIME_STEP 	= new int[4];		// Number of items to {empty, empty, update, delete/insert}

	static int DEBUG_OUTPUT	    	    = 0;  				// 
	static int NUMBER_SLICES	   	    = 100000;  			// 
	///////////////////////////////////////////////////////////////////////
	///  END SIMULATION PARAMETERS 
	//////////////////////////////////////////////////////////////////////	
	
	static ArrayList<Node> 	pool	 		= new ArrayList<Node>();
	static ArrayList<Node> 	poolAuthor	 	= new ArrayList<Node>();
	static ArrayList<Node> 	poolPublisher	= new ArrayList<Node>();
	static ArrayList<Node> 	poolRelated	 	= new ArrayList<Node>();
	
	static Random generator 				= new Random(1);
	static final int CHANGE_INSERT 			= 0;
	static final int CHANGE_DELETE 			= 1;
	static final int CHANGE_UPDATE 			= 2;
	static final int CHANGE_DELETEINSERT	= 3;
	
	static ArrayList<String> cols 			= new ArrayList<String>();
	static ArrayList<String> colsAuthor 	= new ArrayList<String>();
	static ArrayList<String> colsPublisher 	= new ArrayList<String>();
	static ArrayList<String> colsRelated 	= new ArrayList<String>();

	// Just counters for debugging output
	static int numChanges 	= 0;
	static int numEdits 	= 0;
	static int numInserts 	= 0;
	static int numDeletes 	= 0;
	
	static final int TYPE_ITEM 		= 1;
	static final int TYPE_AUTHOR 	= 2;
	static final int TYPE_PUBLISHER = 3;
	static final int TYPE_RELATED 	= 4;
	
	// For getIDToChange
	static String lastTime = "";
	static HashMap<String, Integer> usedIDs = new HashMap<String, Integer>();
	static HashMap<String, String> idMap = new HashMap<String, String>();
	static ArrayList<String> activeIDs = new ArrayList<String>();
	
    static int aCounter = 0;
    static int pCounter = 0;
    static int rCounter = 0;
	
	// DCSD.xsd sequenced constraints
	static final int minAuthors = 1;
	static final int maxAuthors = 4;
	static HashMap<String, Integer> numberAuthors = new HashMap<String, Integer>();

    // Non-sequenced constraints for TKDE paper
    //static final int minAuthorsNS = 1; 
    static final int maxAuthorsNS = 6; 
    // Key will be id+year; value will be number of authors
	static HashMap<String, Integer> numberAuthorsNS = new HashMap<String, Integer>();
	
	static Document inputDocument;
	static Document outputDocument;

    static XPath xpath = XPathFactory.newInstance().newXPath();
    
	
	// For slice listings
	static HashMap<String, ArrayList<String>> sliceListings = new HashMap<String, ArrayList<String>>();	
	/**
	 * MAIN FUNCTION
	 */
	public static void main(String[] args) throws Exception{

        ////////////////////////////////////////////////////////////////////////
        // Read input file (settings)
        ///////////////////////////////////////////////////////////////////////
		System.out.println("tGenerator: Reading parameters file...");
		String fileName = "parameters.txt";
        if (args.length > 0) {
            fileName = args[0];
        }
        readSettings(fileName);

        // Read in additional things from the command line
        for (int i = 1; i < args.length; i+=2){
            setSetting(args[i], args[i+1]);
        } 

        ////////////////////////////////////////////////////////////////////////
        // Build output directory
        ///////////////////////////////////////////////////////////////////////
        boolean success = (new File(OUTPUT_DIR)).mkdir();
		
		////////////////////////////////////////////////////////////////////////
		// Build output document 
		/////////////////////////////////////////////////////////////////////////
		DocumentBuilderFactory factory;
		DocumentBuilder loader;
		outputDocument = null;
		try {
		      factory = DocumentBuilderFactory.newInstance();
		      loader  = factory.newDocumentBuilder();
		      outputDocument = loader.newDocument();
		      Element root1 = outputDocument.createElement("catalog");
		      outputDocument.appendChild(root1);
		      
		    } catch (ParserConfigurationException ex) {
		      handleError(ex);
		    } catch (FactoryConfigurationError ex) {
		      handleError(ex);
		 }
		 Element outputRoot = outputDocument.getDocumentElement();
	
		 
		////////////////////////////////////////////////////////////////////////
		// Read Input document
		/////////////////////////////////////////////////////////////////////////
		if(!(new File(INPUT_FILE)).exists()){
			System.out.printf("tGenerator: Error! File \""+INPUT_FILE+"\" doesn't exist.");
			System.exit(2);
		}
		System.out.println("tGenerator: Reading catalog file...");
		DOMParser parser = new DOMParser();
		parser.parse(INPUT_FILE);
		inputDocument = parser.getDocument();
		
		Node inputRoot = inputDocument.getDocumentElement();

		List<Element> expectedChildren = findAllElementsByTagName((Element)inputRoot, "item");
        int numChildren = expectedChildren.size();
		
        //int numInitial = (numChildren*INITIAL_PERCENTAGE/100);
		//System.out.println("tGenerator: XBench file has "+numChildren+" children. Taking "+numInitial+" for first snapshot.");
		
		
		//////////////////////////////////////////////////////////////////////////
		// Initialize column arrays
		//////////////////////////////////////////////////////////////////////////
		cols.add("ISBN"); cols.add("title"); cols.add("subject"); cols.add("number_of_pages"); cols.add("type_of_book"); cols.add("length");
		cols.add("width"); cols.add("height"); cols.add("suggested_retail_price"); cols.add("cost");
		cols.add("when_is_available"); cols.add("quantity_in_stock"); cols.add("date_of_release"); cols.add("description");
		
		colsAuthor.add("first_name"); colsAuthor.add("middle_name"); colsAuthor.add("last_name"); colsAuthor.add("date_of_birth"); colsAuthor.add("biography");
		colsAuthor.add("street_address"); colsAuthor.add("name_of_city"); colsAuthor.add("name_of_state"); colsAuthor.add("zip_code"); colsAuthor.add("name_of_country");
		colsAuthor.add("phone_number"); colsAuthor.add("email_address");
		
		colsPublisher.add("street_address"); colsPublisher.add("name_of_city"); colsPublisher.add("name_of_state"); colsPublisher.add("zip_code");
		colsPublisher.add("phone_number"); colsPublisher.add("web_site"); 
		
		colsRelated.add("item_id");
		
		String last_ct = START_TIME;
		
		
		// Now, add IDs to publishers and authors
		System.out.println("tGenerator: Adding IDs to authors and publishers...");
		addIDs((Element)inputRoot);
        
		
		//////////////////////////////////////////////////////////////////////////
		// Loop through the items, build initial snapshot and pools
		/////////////////////////////////////////////////////////////////////////
        int ratio = (int)(((INITIAL_PERCENTAGE)/100.0)*((double)numChildren));
        int takeEvery = (int)((((float)numChildren))/ratio);
		System.out.println("tGenerator: Building initial snapshot and pools...");
		//System.out.println("tGenerator: (taking every "+takeEvery+"th item)");
		int counter=0;
		int counterI=0;
		for(Node childNode = inputRoot.getFirstChild(); childNode!=null;){
			Node nextChild = childNode.getNextSibling();
			
			if (childNode.getNodeType() == Document.ELEMENT_NODE){
				counter++;
				
				// Have to import the node from the input to the output; it's a weird DOM-ism
				Node n = outputDocument.importNode(childNode, true);
				Element wrappedItem = wrapItem(outputDocument, n);
				
				Node n1 = wrappedItem.getElementsByTagName("author").item(0);
				Element wrappedAuthor = (Element) wrapAllElements(outputDocument, n1, "author", true);
				
				Node n2 = wrappedItem.getElementsByTagName("publisher").item(0);
				Element wrappedPublisher = (Element) wrapAllElements(outputDocument, n2, "publisher", true);
				
				Node n3 = wrappedItem.getElementsByTagName("related_items").item(0);
				Element wrappedRelated = (Element) wrapAllElements(outputDocument, n3, "related_items", true);
				
				// Only take one tenth (or whatever INIITIAL_PERCENTAGE is) of elements;
				// want them to be uniform over the collection.
				if (counter % takeEvery == 0){
				    counterI++;
					outputRoot.appendChild(wrappedItem);
					activeIDs.add(getMyID(wrappedItem, TYPE_ITEM));
					
				} else {
                    // sometimes put the item, sometimes put everything else
                    // NO DUPLICATION!
                    if ((generator.nextInt(2)+1) == 1){
                        pool.add(wrappedItem);
                    }else{
                        poolAuthor.add(wrappedAuthor);
                        poolPublisher.add(wrappedPublisher);
                        poolRelated.add(wrappedRelated);

                    }
				}
			}
			childNode = nextChild;
		}
		
		// Need to fix integrity of related_items
		// We need to select an IDs that are currently valid
		ensureRelatedIntegrity((Element) outputRoot);

		System.out.println("");
		System.out.println("tGenerator: Number of <item>s read from input:        " + counter);
		System.out.println("tGenerator: Number of <item>s put into initial slice: " + counterI);
		System.out.println("tGenerator: Size of <item>  pool:                     " + pool.size());
		System.out.println("tGenerator: Size of <author>  pool:                   " + poolAuthor.size());
		System.out.println("tGenerator: Size of <publisher>  pool:                " + poolPublisher.size());
		System.out.println("tGenerator: Size of <related_item>  pool:             " + poolRelated.size()+"\n");
		
		///////////////////////////////////////////////////////////////////////////////////
		// Main time loop
		///////////////////////////////////////////////////////////////////////////////////
		String ct = START_TIME;
		int count = 1, numChangesInStep = 0;
		
		// Output first slice
		if (OUTPUT_XML_SLICES == 1){	
        	if (DEBUG_OUTPUT==1){
            	System.out.println("Printing XML slice...");
        	}
        	String name = "output."+ct+".xml";
		    outputXmlSlice(outputDocument, ct, name);
		    ArrayList<String> a = new ArrayList<String>();
		    a.add(last_ct);
		    a.add(ct);
		    sliceListings.put(name, a);
		    last_ct = ct;
        }
        if (OUTPUT_RELATION_SLICES == 1){	
		    shredXmlSlice(outputDocument, ct);
        }
		
		
		
		
		// count is just an emergency counter to avoid infinite loops (or if NUMBER_SLICES is set)
		while (numChanges < REQUIRED_CHANGES && count++ < NUMBER_SLICES){
			numChangesInStep = 0;
			
			// Increment date from 1 to TIME_STEP days
			ct= getDateString(ct, TIME_STEP);
		    if (DEBUG_OUTPUT==1){
                System.out.println("Starting time step "+ct);
            }
		    
		    updateActiveIDs(outputRoot, ct);		    
			
			// ... some <author>s
			ArrayList<Node> curItems = getAllCurrentItems(outputDocument, "author", ct);
			for (int k = 0; k < 3; ++k){ // For each edit type...
				for (int j = 0; j < AUTHOR_PER_TIME_STEP[k]; ++j){
					++numChangesInStep;
					int idToChange = getIdToChange(curItems.size()-1, ct, TYPE_AUTHOR, k, curItems);
					Node curItem = curItems.get(idToChange);

		            if (DEBUG_OUTPUT==1){
                        String id = getMyID(curItem, TYPE_AUTHOR);
                        String itemID = idMap.get(id);
                    }

			        makeChanges(outputDocument, ct, curItem, TYPE_AUTHOR, k);
				}
			}
			
			// ... some <publisher>s
			curItems = getAllCurrentItems(outputDocument, "publisher", ct); //all items that are active
			for (int k = 2; k < 4; ++k){ // For each edit type...
				for (int j = 0; j < PUBLISHER_PER_TIME_STEP[k]; ++j){
					++numChangesInStep;
					int idToChange = getIdToChange(curItems.size()-1, ct, TYPE_PUBLISHER, k, curItems);
					Node curItem = curItems.get(idToChange);
			        makeChanges(outputDocument, ct, curItem, TYPE_PUBLISHER, k);
				}
			}
			
			// Randomly change some <item>s
			curItems = getAllCurrentItems(outputDocument, "item", ct); //all items that are active
			ArrayList<Integer> order = new ArrayList<Integer>();
			ArrayList<String> deletedIDs = new ArrayList<String>();
			order.add(1); order.add(0); order.add(2);
			for (int h = 0; h < 2; ++h){ // For each edit type...
				int k = order.get(h);
				int idToChange=0;
				for (int j = 0; j < ITEM_PER_TIME_STEP[k]; ++j){ // For all the number of edits...
					++numChangesInStep;
					idToChange 	  = getIdToChange(curItems.size()-1, ct, TYPE_ITEM, k, curItems);
					Node curItem  = curItems.get(idToChange);
					makeChanges(outputDocument, ct, curItem, TYPE_ITEM, k);
					if (h==0){
						String myID = getMyID(curItem, TYPE_ITEM);
						deletedIDs.add(myID);
					}
				}
				
				if (h==0){
					updateActiveIDs(outputRoot, ct);
				}
			}
			updateActiveIDs(outputRoot, ct);
			
			// ... some <related_items>s
			curItems = getAllCurrentItems(outputDocument, "related_items", ct); //all items that are active
			
			// debug printing:
			/*System.out.println("Currently active related_items: ");
			for (int i=0; i < curItems.size();++i){
				Node cur = curItems.get(i);
				String id = getMyID(cur, TYPE_RELATED);
				System.out.print(id + " ");
			}
			System.out.println("");*/
			
			//TODO: for all ids that were changed, make sure the ones that point to them are changed
			// dirty dirty hack: change all dangling pointers: 
			/*System.out.println("Fixing dangles");
			System.out.println("Deleted ids are: " + deletedIDs.toString());
			System.out.println("Active ids are: " + activeIDs.toString());*/
			ArrayList<Integer> related_itemsToChange = whichRelatedItemsPointTo(curItems, deletedIDs, ct);
			int k = 2;
			for (int h = 0; h < related_itemsToChange.size(); ++h){
				Node curItem = curItems.get(related_itemsToChange.get(h));
				makeChanges(outputDocument, ct, curItem, TYPE_RELATED, k);
			}
			
			for (k = 2; k < 4; ++k){ // For each edit type...
				for (int j = 0; j < RELATED_PER_TIME_STEP[k]; ++j){
					++numChangesInStep;
					int idToChange = getIdToChange(curItems.size()-1, ct, TYPE_RELATED, k, curItems);
					Node curItem = curItems.get(idToChange);
					makeChanges(outputDocument, ct, curItem, TYPE_RELATED, k);
				}
			}
			
			numChanges+=numChangesInStep;
			System.out.printf("tGenerator: %3d changes at %s (%4d total, %5d required); %d slices\n", numChangesInStep, ct, numChanges, REQUIRED_CHANGES, count);
			
			
            if (OUTPUT_XML_SLICES == 1){	
            	if (DEBUG_OUTPUT==1){
                	System.out.println("Printing XML slice...");
            	}
            	String name = "output."+ct+".xml";
			    outputXmlSlice(outputDocument, ct, name);
			    ArrayList<String> a = new ArrayList<String>();
			    a.add(last_ct);
			    a.add(ct);
			    sliceListings.put(name, a);
			    last_ct = ct;
            }
            if (OUTPUT_RELATION_SLICES == 1){	
			    shredXmlSlice(outputDocument, ct);
            }

            if (DEBUG_OUTPUT==1){
            	System.out.println("At end of time loop:");
            	
            	// Useful for debugging, when you know what item id is messing up and 
            	// you want to find out *when*
            	//System.out.println("Item has "+getNumAuthorsNS("I1390", "2002"));
            	//printNodeXPath("//item[@id='I2210']", outputDocument);
            }

		} // End time loop
		
		System.out.println("\ntGenerator: Number of elements changed:  "+numEdits);
		System.out.println("tGenerator: Number of elements inserted: "+numInserts);
		System.out.println("tGenerator: Number of elements deleted:  "+numDeletes);
		
		System.out.println("\ntGenerator: Simulation complete.");
		
        if (OUTPUT_XML == 1){	
        	System.out.println("tGenerator: Writing final XML output file.");
        	
        	// Wrap Catalog
        	Element cr = outputDocument.createElement("catalog_RepItem");
        	cr.setAttribute("isItem", "yes");
        	cr.setAttribute("originalElement", "catalog");
        	Element cv = outputDocument.createElement("catalog_Version");
        	cv.setAttribute("begin", START_TIME);
        	cv.setAttribute("end", ct);
        	Element curRoot = outputDocument.getDocumentElement();
        	
        	cv.appendChild(curRoot);
        	cr.appendChild(cv);
        	outputDocument.appendChild(cr);
        	
			writeXmlFile(outputDocument, OUTPUT_DIR+"/output.final.xml", true, ct);
			
        }	
        if (OUTPUT_XML_SLICES == 1){
        	writeXmlSliceListingsFile(OUTPUT_DIR+"/output.final.slices.xml", ct);
          
            int num = sliceListings.size();  
        	System.out.println("tGenerator: Wrote "+num+" slices.");
        }
		
        if (OUTPUT_RELATIONS ==1){
        	System.out.println("tGenerator: Shredding XML output file into 5 tables.");
			shredXmlFile(outputDocument, "final");
        }
		
		System.out.println("\ntGenerator: Done.");
	}
	
	
	
	
	
	
	
	
	
	

	
	private static ArrayList<Integer> whichRelatedItemsPointTo(ArrayList<Node> curItems, ArrayList<String> deletedIDs, String ct) {
		
		ArrayList<Integer> result = new ArrayList<Integer>();
		
		// loop through all related items. If any of the item_ids points to any of the deltedids, add to reuslt
		
		for (int i = 0; i < curItems.size(); i++){
			Node cur = curItems.get(i);
			String end = findAttribute(cur, "end");
			String begin = findAttribute(cur, "begin");
			String itemID = getItemID(TYPE_RELATED, cur);
			//if (!begin.equals(ct)){
			List<Element> dd = findAllElementsByTagName((Element) cur, "item_id");
			for (int j = 0; j < dd.size(); j++){
				Node cur1 = dd.get(j);
				Node parent1 = cur1.getParentNode();
				Node parent2 = parent1.getParentNode();
				String related_id = ((Element) parent2).getAttribute("related_items_id");
				
				String thisID = cur1.getTextContent();
				if (deletedIDs.contains(thisID)){
					//System.out.println("Marking " + related_id + " for change ("+itemID+")");
					result.add(i);
					break; //only need one per related_items
				} else {
					//System.out.println("Skipping " + related_id + " for change ("+itemID+")");
				}
			
			}
		//}
			
		}
		
		return result;
	}


	private static void updateActiveIDs(Element outputRoot, String ct) {
		 // First, build list of all current item ids (without duplication)	
        if (DEBUG_OUTPUT == 1){
            System.out.println("entering updateActiveIDs\n");
        }
	    activeIDs.clear();
   		//List<Element> d = findAllElementsByTagName(outputRoot, "item_Version");
	    NodeList d = getAllNodesXPath(".//item_Version", outputRoot);
        int l = d.getLength();
   		for (int j=0; j< l; ++j){
   			  Element cur1 = (Element)d.item(j);
   			  String e = cur1.getAttribute("end");
   			  if (e.compareTo(ct) >  0){
   				  Element item = (Element) cur1.getFirstChild();
   				  String myID = item.getAttribute("id");
	    				  
   				  if (! activeIDs.contains(myID)){
   					  activeIDs.add(myID);
   				  }
   			  }
   		}
        if (DEBUG_OUTPUT == 1){
            System.out.println("exiting updateActiveIDs\n");
        }
		
    }


	/**
	 * 
	 * @param fileName
	 */
	private static void writeXmlSliceListingsFile(String fileName, String lastTime) {
		
		Document out = null;
		DocumentBuilderFactory factory;
		DocumentBuilder loader;
		try {
		      factory = DocumentBuilderFactory.newInstance();
		      loader  = factory.newDocumentBuilder();
		      out     = loader.newDocument();
        }catch(Exception e){}
        
        Element sliceSequence = out.createElement("sliceSequence");
        
        ArrayList<String> keys = new ArrayList<String>(); 
        keys.addAll(sliceListings.keySet()); 
        Collections.sort(keys); 
        Iterator<String> it2 = keys.iterator(); 
        while (it2.hasNext()) {
        	String curKey = it2.next();
          	// System.out.println(it2.next()); 
          	 
          	 ArrayList<String> a = sliceListings.get(curKey);
          	 
          	 Element curSlice = out.createElement("slice");
          	 curSlice.setAttribute("location", curKey);
          	 curSlice.setAttribute("begin", a.get(0));
          	 curSlice.setAttribute("end", a.get(1));
          	 
          	 sliceSequence.appendChild(curSlice);
        }
        
        out.appendChild(sliceSequence);
        
        writeXmlFile(out, fileName, true, lastTime);
		
	}


	/**
	 * 
	 * @param type
	 * @param curItem
	 * @return
	 */
	private static String getItemID(int type, Node curItem){
		
		if (type == TYPE_ITEM){
			String itemID = findAttribute(curItem, "id");
			return itemID;
		} else {
	
		  Element cur = (Element)curItem;
		  if (cur.hasAttribute("id")) {
		      return cur.getAttribute("id");
		    }
		  Node parent = curItem.getParentNode();
		  if (parent != null){
			  return getItemID(type, parent);
		  }
		return null;
		}
		

		
	}


	/**
	 * HELPER FUNCTION
	 * Has the actual logic for making the changes to a tuple.
	 */
	private static void makeChanges(Document outputDocument, String ct, Node curItem, int type,  int changeType){
		ArrayList<String> c;
		String typeName = "";
		
		if (type==TYPE_ITEM){
			c = cols;
			typeName = "item";
		} else if (type == TYPE_AUTHOR){
			c = colsAuthor;
			typeName = "author";
		} else if (type == TYPE_PUBLISHER){
			c = colsPublisher;
			typeName = "publisher";
		} else {
			c = colsRelated;
			typeName = "related_item";
		}
			
		
		/////////////////////////////////////////////////////////
		if (changeType==CHANGE_UPDATE){
			
            if (DEBUG_OUTPUT==1){	
                String id = getMyID(curItem, type);
                String itemID = idMap.get(id);
			    System.out.printf("   %s: Updating %s with associated itemID (%s) (%s)\n", ct, typeName, itemID, id);
            }
            

            
			numEdits++;
			
			// Select node to steal data from (has version already)
			Node poolChild = getFromPool(type, ct);
			Node lastChild = getLastVersion(curItem);
			Node newChild = lastChild.cloneNode(true);

			// Randomly choose some fields from current, update with new
			// (this function adds a new <version>, and sets timestamps appropriately)
			makeEdits(outputDocument, (Element)poolChild, (Element)newChild, c);
			
			
			
			// Remove "expired" sub elements (only applies to <item>)
			if (type==TYPE_ITEM){
				removeExpiredSubelements(newChild, ct);
			}
			if (type == TYPE_RELATED){
				//ensureRelatedIntegrity((Element) newChild);
				newChild = getFirstElement(poolChild); //now is version
				Element curRelated = getFirstElement(newChild); //now is related_item
				Element curLast  = getFirstElement(lastChild);
				curRelated.setAttribute("related_items_id", curLast.getAttribute("related_items_id"));
			
				
				//debug print
				/*NodeList dd = ((Element) newChild).getElementsByTagName("item_id");
				System.out.print("          Related items are: ");
				for (int i=0;i<dd.getLength();++i){
					System.out.print(dd.item(i).getTextContent()+ " ");
				}
				System.out.println("");*/
			}

			
			// Set the end times
			setEndTime((Element)lastChild, ct);
            setAllChildrenEndTime((Element)lastChild, ""+ct);

            // Set the begin times
			setBeginTime((Element)newChild, ct);
			setAllChildrenBeginTime((Element)newChild, ct);
			
			// Add it as a child to the root node
			curItem.appendChild(newChild);
			
			String id = getMyID((Element)newChild, type);
            String itemID= getItemID(type, (Element)newChild);
            idMap.put(id, itemID);
            usedIDs.put(itemID, 1);
			

		////////////////////////////////////////////////////////////
		} else if (changeType == CHANGE_INSERT){
			numInserts++;
			
			// Select node to insert
			Node newChild = getFromPool(type, ct);
			
			// Set the begin time to now (it already has end time = forever)
			setBeginTime((Element)newChild.getFirstChild(), ct);
			setAllChildrenBeginTime((Element)newChild.getFirstChild(), ""+ct);
			
			// Add it as a child to the root node
			curItem.getParentNode().appendChild(newChild);
			
			//ensureRelatedIntegrity((Element) newChild);
            String id = getMyID((Element)newChild, type);
            String itemID= getItemID(type, (Element)newChild);
            idMap.put(id, itemID);
            usedIDs.put(itemID, 1);
            activeIDs.add(itemID);
            
            if (type == TYPE_AUTHOR){
            	addNumAuthors(itemID);
            	addNumAuthorsNS(itemID, getYearFromString(ct));
            }
            
		    if (DEBUG_OUTPUT==1){
			    System.out.printf("   %s: Inserting %s with associated itemID (%s) (%s)\n",  ct, typeName, itemID, id);
		    }

			
	    ////////////////////////////////////////////////////////////
		} else if (changeType == CHANGE_DELETE){
			numDeletes++;
		

			// Just set the end time to the current time.
			//Element v = (Element)curItem.getFirstChild();
			Element v = (Element)getLastVersion(curItem);
			setEndTime(v, ""+ct);
            setAllChildrenEndTime(v, ""+ct);

            String id = getMyID(curItem, type);
            String itemID = idMap.get(id);

            usedIDs.put(itemID, 1);
            activeIDs.remove(itemID);
            
            if (type == TYPE_AUTHOR){
            	deleteNumAuthors(itemID);
            	
            	//Don't actually need to call this, because its NS!
            	//deleteNumAuthorsNS(itemID, getYearFromString(ct));
            }
            
            if (DEBUG_OUTPUT==1){	
                System.out.printf("   %s: Deleting %s with associated itemID (%s) (%s)\n", ct, typeName, itemID, id);
            }
        
        ////////////////////////////////////////////////////////////
		} else if (changeType == CHANGE_DELETEINSERT){

			//First, delete someone
			// Just set the end time to the current time.
			//Element v = (Element)curItem.getFirstChild();
			Element v = (Element)getLastVersion(curItem);
			setEndTime(v, ""+ct);
            setAllChildrenEndTime(v, ""+ct);
            
            //Then, insert someone
			// Select node to insert
			Node newChild = getFromPool(type, ct);
			// Set the begin time to now (it already has end time = forever)
			setBeginTime((Element)newChild.getFirstChild(), ct);
			setAllChildrenBeginTime((Element)newChild.getFirstChild(), ct);
			// Add it as a child to the root node
			//ensureRelatedIntegrity((Element) newChild);
			curItem.getParentNode().appendChild(newChild);
			


            String idDelete = getMyID(curItem, type);
            String idInsert = getMyID(newChild, type);
			String itemID= getItemID(type, (Element)newChild);
	        idMap.put(idInsert, itemID);
            usedIDs.put(itemID, 1);
            if (DEBUG_OUTPUT==1){	
			    System.out.printf("   %s: Delete/Inserting %s with associated itemID (%s) (%s) (%s)\n", ct, typeName, itemID, idDelete, idInsert);
            }
		}
		
		// Ensure integrity of related_item
		// For related, we need to select an ID that's currently valid
		//ensureRelatedIntegrity((Element) curItem);
		
		return;
	}
	
	private static void ensureRelatedIntegrityAtTime(Node curItem){
		// Which ids does this related_items already point to? Can't point to the same twice!
		HashMap<String, Integer> alreadyHas = new HashMap<String, Integer>();
		
		List<Element> r = findAllElementsByTagName((Element) curItem, "item_id");
	  	for (int i=0; i< r.size(); i++){
	  		Element cur = (Element)r.get(i);
	  		String thisID = cur.getFirstChild().getNodeValue();
	  		
	  		// If something is wrong with thisID..
	  		if ((!activeIDs.contains(thisID)) || alreadyHas.containsKey(thisID)){
		  		do {
					int idx = generator.nextInt(activeIDs.size());
					String id = activeIDs.get(idx);
					if (!alreadyHas.containsKey(id)){
						cur.getFirstChild().setNodeValue(id);
						alreadyHas.put(id, 1);
			            usedIDs.put(id, 1);
						break;
					} 
		  		} while (true);
	  		
		  	// If everything is OK with thisID, then just add it to what we already have
	  		} else {
				alreadyHas.put(thisID, 1);
			}
	  	}	
  }

		
		
	
	
	
	
	
	
	
	
	
	
	
	
	
	// Gets rid of duplication!
	private static void ensureRelatedIntegrity(Element curItem){
		

		List<Element> d = findAllElementsByTagName(curItem, "related_items");
		
		for (int j=0; j< d.size(); j++){
			  Element cur1 = (Element)d.get(j);
			  HashMap<String, Integer> alreadyHas = new HashMap<String, Integer>();
		
			List<Element> r = findAllElementsByTagName(cur1, "item_id");
			for (int i=0; i< r.size(); i++){
				  Element cur = (Element)r.get(i);
				  
				  String myID = cur.getFirstChild().getNodeValue();
				  
				if (! activeIDs.contains(myID) || alreadyHas.containsKey(myID)){
					
					do {
						int idx = generator.nextInt(activeIDs.size());
						String id = activeIDs.get(idx);
						if (! alreadyHas.containsKey(id)){
							alreadyHas.put(id, 1);
							cur.getFirstChild().setNodeValue(id);
							break;
						}
					} while (true);
				} else {
					alreadyHas.put(myID, 1);
				}
			}	
		}
		
	}

	
	private static void removeExpiredSubelements(Node newChild, String ct) {
		int result = 0;
		do{
			Node remover = ((Element) newChild).getElementsByTagName("authors").item(0);
			NodeList a = ((Element) newChild).getElementsByTagName("author_Version");
			result = removeExpiredSubelements2(remover, a, ct);
		}while (result > 0);
		
		result = 0;
		do{
			NodeList a = ((Element) newChild).getElementsByTagName("publisher_Version");
			result = removeExpiredSubelements2(newChild.getFirstChild(), a, ct);
		}while (result > 0);

		result = 0;
		do{
			NodeList a = ((Element) newChild).getElementsByTagName("related_items_Version");
			result = removeExpiredSubelements2(newChild.getFirstChild(), a, ct);
		}while (result > 0);
		return;
	}
	
	private static int removeExpiredSubelements2(Node newChild, NodeList nodeList, String ct) {
		for(int i=0; i<nodeList.getLength(); i++){
		  Node childNode = nodeList.item(i);
		  if (childNode.getNodeType() == Node.ELEMENT_NODE){
            Element e = (Element)childNode;
            
            // Poor-man's time comparison
            if (e.hasAttribute("end") && e.getAttribute("end").compareTo(ct) < 0){
            	// Since we're at <_version>, remove the parent (<_item>)
            	Node parentToRemove = childNode.getParentNode();
            	
            	//System.out.println("\nBefore remove:");
            	//printItem(parentToRemove, "");
            	//System.out.println();
            	
                parentToRemove.removeChild(childNode);
                
            	
            	// If this was the last Version in the RepItem, get rid of the RepItem as well
            	if (parentToRemove.getChildNodes().getLength() == 0){
            		Node Parent2 = parentToRemove.getParentNode();
            		Parent2.removeChild(parentToRemove);
            	}
                return 1;
            } 
		  }
		}
		return 0;
	}
	
    // Assumes you pass in a <_version>
    private static String getMyID(Node n, int type){
		String id = "";

        if (type == TYPE_AUTHOR){
            //id = ((Element)n.getFirstChild().getFirstChild()).getAttribute("author_id");
            id = findAttribute(n, "author_id");
         } else if (type == TYPE_PUBLISHER){
           // id = ((Element)n.getFirstChild().getFirstChild()).getAttribute("publisher_id");
            id = findAttribute(n, "publisher_id");
         } else if (type == TYPE_RELATED){
            //id = ((Element)n.getFirstChild().getFirstChild()).getAttribute("related_items_id");
            id = findAttribute(n, "related_items_id");
         } else if (type == TYPE_ITEM){
            //id = ((Element)n.getFirstChild().getFirstChild()).getAttribute("id");
            id = findAttribute(n, "id");
         }

        if (id.equals("") || id==null){
            System.out.println("Warning: getMyID is funky!\n");
        }

         return id;
    }

	private static String findAttribute(Node n, String string) {
		Element cur = (Element)n;
		  if (cur.hasAttribute(string)) {
		      return cur.getAttribute(string);
		    }
		  Node child = n.getFirstChild();
		  if (child != null){
			  return findAttribute(child, string);
		  }
		return null;
	}


	/**
	 * HELPER FUNCTION
	 */
	private static Node getLastVersion(Node curItem) {
		return curItem.getLastChild();
	}

	/**
	 * HELPER FUNCTION
	 */
	private static void makeEdits(Document outputDocument, Element poolChild, Element newChild, ArrayList<String> c) {
		int columnToChange = generator.nextInt(c.size());
		Node tmp1 = null, tmp2 = null;

		tmp1 = poolChild.getElementsByTagName(c.get(columnToChange)).item(0);
		tmp2 = newChild.getElementsByTagName(c.get(columnToChange)).item(0);
		
		//System.out.println("Looking for "+c.get(columnToChange));
		
		if (tmp1!=null && tmp2 != null){
			tmp2.getFirstChild().setNodeValue(tmp1.getFirstChild().getNodeValue());
		}
	}

	/**
	 * HELPER FUNCTION
	 */
	private static Element wrapItem(Document outputDocument, Node n) {
		// Insert standard tXMLSchema items, versions *before* the following nodes.
		Element n1 = (Element) wrapAllElements(outputDocument, n, "author", false);
		Element n2 = (Element) wrapAllElements(outputDocument, n1, "publisher", false);
		Element n3 = (Element) wrapAllElements(outputDocument, n2, "related_items", false);
		Element wrapped = (Element) wrapAllElements(outputDocument, n3, "item", true);
		return wrapped;
	}
	
	/**
	 * HELPER FUNCTION
	 */
	private static void addIDs(Element n) {
        Node node = (Node)n;
        String id = "";
        String year = getYearFromString(START_TIME);

        for(Node childNode = node.getFirstChild(); childNode!=null;){
            Node nextChild = childNode.getNextSibling();

            if (childNode.getNodeType() == Node.ELEMENT_NODE){
                Element e = (Element)childNode;
                if (e.getNodeName().equals("author")){
                    id = "A"+aCounter;
	                e.setAttribute("author_id", id);
	                ++aCounter;
	                
	                // Keep track of how many authors each item has
	                String itemID = getItemID(TYPE_AUTHOR, e);
	                if (numberAuthors.containsKey(itemID)){
	                	numberAuthors.put(itemID, numberAuthors.get(itemID)+1);
	                } else {
	                	numberAuthors.put(itemID, 1);
	                }
	                if (numberAuthorsNS.containsKey(itemID+year)){
	                	numberAuthorsNS.put(itemID+year, numberAuthorsNS.get(itemID+year)+1);
	                } else {
	                	numberAuthorsNS.put(itemID+year, 1);
	                }
                    idMap.put(id, itemID);
                } else if (e.getNodeName().equals("publisher")){
                    id = "P"+pCounter;
	                e.setAttribute("publisher_id", id);
	                

	                String itemID = getItemID(TYPE_PUBLISHER, e);
                    idMap.put(id, itemID);
	                ++pCounter;
                } else if (e.getNodeName().equals("related_items")){
                    id = "R"+rCounter;

	                String itemID = getItemID(TYPE_RELATED, e);
	                e.setAttribute("related_items_id", id);
                    idMap.put(id, itemID);
	                ++rCounter;
                } else if (e.getNodeName().equals("item")){
                	id = e.getAttribute("id");
                	idMap.put(id, id);
                }
                addIDs(e);
            }
            childNode = nextChild;
        }
		return;
	}

	/**
	 * HELPER FUNCTION
	 */
	private static void setAllChildrenEndTime(Element n, String t) {
        // Loop through, looking for children with "end" attribute
        Node node = (Node)n;
        for(Node childNode = node.getFirstChild(); childNode!=null;){
            Node nextChild = childNode.getNextSibling();

            if (childNode.getNodeType() == Node.ELEMENT_NODE){
                Element e = (Element)childNode;
                if (e.hasAttribute("end") && e.getAttribute("end").equals(FOREVER_TIME)){
	                e.setAttribute("end", t);
                }
                setAllChildrenEndTime(e, t);
            }

            childNode = nextChild;
        }
        
		return;
	}
	
	/**
	 * HELPER FUNCTION
	 */
	private static void setAllChildrenBeginTime(Element n, String t) {
        // Loop through, looking for children with "end" attribute
        Node node = (Node)n;
        for(Node childNode = node.getFirstChild(); childNode!=null;){
            Node nextChild = childNode.getNextSibling();

            if (childNode.getNodeType() == Node.ELEMENT_NODE){
                Element e = (Element)childNode;
                if (e.hasAttribute("begin")){
                	//String id = ((Element)e.getFirstChild()).getAttribute("author_id");
                	//System.out.printf("Setting begin time for id (%s)\n", id);
	                e.setAttribute("begin", t);
                }
                setAllChildrenBeginTime(e, t);
            }

            childNode = nextChild;
        }
        
		return;
	}

	/**
	 * HELPER FUNCTION
	 */
	private static void setEndTime(Element n, String t) {
		  n.setAttribute("end", t);
		  return;
	}
	
	/**
	 * HELPER FUNCTION
	 */
	private static void setBeginTime(Element n, String t) {
		  n.setAttribute("begin", t);
		  return;
	}

	/**
	 * HELPER FUNCTION
	 * Wraps all children of n named string with <string_item> and <string_version> elements.
	 */
	private static Node wrapAllElements(Document d, Node n, String string, Boolean root) {
		String string1 = string+"_RepItem";
		String string2 = string+"_Version";

		List<Element> l = findAllElementsByTagName((Element)n, string);
		
		  for (int i=0; i< l.size(); i++)
		  {
			  Element cur = (Element)l.get(i);
			  
			  Element e  = d.createElement(string1);
              e.setAttribute("isItem", "y");
              e.setAttribute("originalElement", string);
			  Element e2 = d.createElement(string2);
			  e2.setAttribute("begin", START_TIME);

              //int numDays = generator.nextInt(NUM_DAYS_VALID)+1;
              //String endTime = getDateString(START_TIME, numDays);
			  //e2.setAttribute("end", endTime);
			  e2.setAttribute("end", FOREVER_TIME);
			  
			  e2.appendChild(cur.cloneNode(true));
			  e.appendChild(e2);
			  
			  if (!root){
				  cur.getParentNode().replaceChild((Node)e, (Node)cur);
			  } else{
				  return e;
			  }
		  }
		return n;
	}

	/**
	 * HELPER FUNCTION
	 */
	private static ArrayList<Node> getAllCurrentItems(Document d, String name, String ct) {
		ArrayList<Node> answer = new ArrayList<Node>();
		
		NodeList r = d.getElementsByTagName(name+"_RepItem");
		
		// Loop through, only finding current "forever"
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  
			  // Loop through each version:
			  NodeList rr = cur.getElementsByTagName(name+"_Version");
			  for (int j=0; j< rr.getLength(); j++){
				  Element ve = (Element)(Element)rr.item(j);
	
				  if (ve == null){
					  System.err.println("Error: something bad happened.");
					  System.exit(1);
				  }
	  
				  if (ve.getAttribute("begin").compareTo(ct) <= 0 && ve.getAttribute("end").equals(FOREVER_TIME)){
					  answer.add((Node)cur);
				  } 
			  }
		  }		
		return answer;
	}

	/** 
	 * HELPER FUNCTION
	 * Return an integer code otherwise  
	 **/
	static int rollDice(){
		return generator.nextInt(3)+1;
	}
	
	/** 
	 * HELPER FUNCTION
	 * Return 0 if no change, and an integer code otherwise  
	 **/
	static Node getFromPool(int type, String ct){
		if (type==TYPE_ITEM){
			if (pool.size()==0){
				System.out.println("\ntGenerator: Error! Item pool empty. Exiting.");
				System.exit(1);
			}
			int id = getIdToChange(pool.size()-1, ct, TYPE_ITEM, 0, null);
			
			// Make sure related_items are up-to-date!
			Node thisID = pool.remove(id);
			Node thisRelated = findAllElementsByTagName((Element) thisID, "related_items_RepItem").get(0);
			ensureRelatedIntegrityAtTime(thisRelated);
			
			return thisID;
		} else if (type == TYPE_AUTHOR){
			if (poolAuthor.size()==0){
				System.out.println("\ntGenerator: Error! Author pool empty. Exiting.");
				System.exit(1);
			}
			return poolAuthor.remove(0);
		} else if (type == TYPE_PUBLISHER){
			if (poolPublisher.size()==0){
				System.out.println("\ntGenerator: Error! Publisher pool empty. Exiting.");
				System.exit(1);
			}
			return poolPublisher.remove(0);
		} else {
			if (poolRelated.size()==0){
				System.out.println("\ntGenerator: Error! Related pool empty. Exiting.");
				System.exit(1);
			}
			Node thisRelated = poolRelated.remove(0);
			ensureRelatedIntegrityAtTime(thisRelated);
			return thisRelated;
		}
		
	}
	
	/**
	 * HELPER FUNCTION
	 */
	private static int getIdToChange(int max, String ct, int type, int changeType, ArrayList<Node> curItems){
		int id = 1;
		
		// Are we in the same timestep as last call to this function?
		if (!ct.equals(lastTime)){
        	lastTime = ct;
        	usedIDs.clear();
        	
        	//TODO: output list of IDs selected for each timestep, to verify their distribution
		}
		//System.out.println("curItems size: " + curItems.size());
		//System.out.println("max: " + max);
		//System.out.println("usedIDs ("+usedIDs.size()+"): " + usedIDs.toString());
		
		// Keep selecting IDs until we've found one we haven't already used
		// (or we are allowed to, based on constraints)
		boolean go = true; int counter = 0;
		while (go == true && counter < 5000){
			if (SELECTION_TYPE.equalsIgnoreCase("gaussian")){
				id = (int)(generator.nextGaussian()*SELECTION_STDDEV+(max/2));
				if (id>max){ id = max;}
				if (id<0)  { id = 0;}
			} else if (SELECTION_TYPE.equalsIgnoreCase("uniform")){
				id = generator.nextInt(max);
			} else {
				System.err.println("tGenerator: Unknown Selection Type: "+SELECTION_TYPE);
				System.exit(1);
			}
			
			// curItems is null if drawing from the pool (can never draw twice), so we don't need to do this.
			if (curItems != null){
				
				// Now: a bunch of hullabaloo to get the id of the associated <item>
                String myid = getMyID(curItems.get(id), type);
				//String itemID= getItemID(type, curItems.get(id));
				String itemID= idMap.get(myid);
				
				if (type == TYPE_AUTHOR){
					int numAuthorsInItem = getNumAuthors(itemID);
					int numAuthorsInItemNS = getNumAuthorsNS(itemID, getYearFromString(ct));
                    //System.out.printf("Author %s (item %s) has %d authors\n", myid, itemID, numAuthorsInItem);
					if (changeType == CHANGE_DELETE && (numAuthorsInItem == minAuthors)){
						//BAD; select a new id
						//usedIDs.put(itemID, 1);
					    ++counter;
						continue;
					} else if(changeType == CHANGE_INSERT && (numAuthorsInItem == maxAuthors || numAuthorsInItemNS == maxAuthorsNS)){
						//BAD
						//usedIDs.put(itemID, 1);
					    ++counter;
						continue;
					}
				}
				
				if (!usedIDs.containsKey(itemID)){
					// Is this id free (for the current timestep)?
					usedIDs.put(itemID, 1);
					go = false;
				} else {
					//System.out.println("Warning: ID "+id+" ("+itemID+") already selected for this timestep; re-selecting.");
					++counter;
				}
			} else {
				go = false;
			}
		}
		
		if (counter>=5000){

            if (DEBUG_OUTPUT==1){	
                System.out.println(" getIdToChange: Last resort. max = "+max);
            }
			
			boolean found=false;
			for (int i=0; i<max;++i){
				String myid = getMyID(curItems.get(i), type);
				String itemID= idMap.get(myid);
				if (!usedIDs.containsKey(itemID)){
                    id = i;
					usedIDs.put(itemID, 1);
					go = false;
					found = true;
                    if (DEBUG_OUTPUT==1){	
                        System.out.printf(" getIdToChange: Selected id (%d) (%s) (%s)\n", i, itemID, myid);
                    }
                    break;
				}
			}
			if (found==false){
				System.err.println("Warning: couldn't find ID to select (all used). Try reducing number of changes, or increasingg the initial slice size.");
				System.exit(1);
			}
		}
		return id;
	}

	

	
	private static void addNumAuthorsNS(String itemID, String year) {
		String key = itemID + year;
		int val = 0;
		if (numberAuthorsNS.containsKey(key)){
			val = numberAuthorsNS.get(key);
		} 
		numberAuthorsNS.put(key, val+1);
    }
	
	// returns the number of author subelements in an item
	private static int getNumAuthorsNS(String itemID, String year) {
		String key = itemID + year;
		if (numberAuthorsNS.containsKey(key)){
			return numberAuthorsNS.get(key);
		}
		//System.err.printf("tGenerator: Warning: item %s, year %s has no authors!\n", itemID, year);
		return 0;
	}
	
	
	private static void deleteNumAuthors(String itemID) {
		if (numberAuthors.containsKey(itemID)){
	       	numberAuthors.put(itemID, numberAuthors.get(itemID)-1);
		} else {
			//System.err.printf("tGenerator: Warning: item %s has no authors!\n", itemID);
		}
    }
	
	private static void addNumAuthors(String itemID) {
		int val = 0;
		if (numberAuthors.containsKey(itemID)){
	       	val = numberAuthors.get(itemID);
		} 
		numberAuthors.put(itemID, val+1);
    }
	
	// returns the number of author subelements in an item
	private static int getNumAuthors(String itemID) {
		if (numberAuthors.containsKey(itemID)){
			return numberAuthors.get(itemID);
		}
		//System.err.printf("tGenerator: Warning: item %s has no authors!\n", itemID);
		return 0;
	}

	/**
	 * HELPER FUNCTION
	 */
	private static final void handleError(Throwable ex) {
	    System.err.println("tGenerator: Error with DOM: "+ ex.getMessage());
	    ex.printStackTrace();
	    System.exit(2);
	 }
	
	/**
	 * HELPER FUNCTION
	 *  This method writes a DOM document to a file
	 * Copied from http://www.exampledepot.com/egs/javax.xml.transform/WriteDom.html
	 */
	public static void writeXmlFile(Document doc, String filename, boolean wrapRoot, String lastTime) {

        // First, need to prepend temporal root
        Document output = null;

        if (wrapRoot){
		    DocumentBuilderFactory factory;
		    DocumentBuilder loader;
		    Document outputDocument = null;
		    try {
		      factory = DocumentBuilderFactory.newInstance();
              factory.setNamespaceAware(true);
		      loader  = factory.newDocumentBuilder();
              output = loader.newDocument();
            } catch(Exception e) {System.out.println(e);}

            Element oldRoot = doc.getDocumentElement();
            
            Element root = output.createElement("temporalRoot");
            Element tss = output.createElement("temporalSchemaSet");
            Element ts = output.createElement("temporalSchema");
            ts.setAttribute("schemaLocation", "ts.xml"); // TODO: make input
            //root.setAttribute("xmlns", "http://www.cs.arizona.edu/tau/TDSchema");
            //root.setAttribute("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
            //root.setAttribute("xsi:schemaLocation", "http://www.cs.arizona.edu/tau/TDSchema etc/TDSchema.xsd");
            root.setAttribute("begin", START_TIME);
            root.setAttribute("end", lastTime);

            tss.appendChild(ts);
            root.appendChild(tss);
            Node n = output.importNode((Node)oldRoot, true); 
            root.appendChild(n);

            output.appendChild(root); 
        } else {
            output = doc;
        }
        
		try{
		FileOutputStream fos = new FileOutputStream(filename);
		// XERCES 1 or 2 additionnal classes.
		OutputFormat of = new OutputFormat("XML","ISO-8859-1",true);
		of.setIndent(1);
		of.setIndenting(true);
		XMLSerializer serializer = new XMLSerializer(fos,of);
		// As a DOM Serializer
		serializer.asDOMSerializer();
		serializer.serialize( output.getDocumentElement() );
		fos.close();
		} catch (Exception e){handleError(e);}
	}
	
	/**
	 * HELPER FUNCTION
	 * stolen from http://www.java2s.com/Code/Java/XML/FindAllElementsByTagName.htm 
	 */
	public static List<Element> findAllElementsByTagName(Element elem, String tagName) {
	    List<Element> ret = new LinkedList<Element>();
	    findAllElementsByTagName(elem, tagName, ret);
	    return ret;
	}
	
	
	/**
	 * HELPER FUNCTION
	 */
	  private static void findAllElementsByTagName(Element el, String tagName, List<Element> elementList) {

	    if (tagName.equals(el.getTagName())) {
	      elementList.add(el);
	    }
	    Element elem = getFirstElement(el);
	    while (elem != null) {
	      findAllElementsByTagName(elem, tagName, elementList);
	      elem = getNextElement(elem);
	    }
	  }
	  
		/**
		 * HELPER FUNCTION
		 */
	  public static Element getFirstElement(Node parent) {
		    Node n = parent.getFirstChild();
		    while (n != null && Node.ELEMENT_NODE != n.getNodeType()) {
		      n = n.getNextSibling();
		    }
		    if (n == null) {
		      return null;
		    }
		    return (Element) n;
		  }
	  
	/**
	 * HELPER FUNCTION
	 */
  public static Element getNextElement(Element el) {
    Node nd = el.getNextSibling();
    while (nd != null) {
      if (nd.getNodeType() == Node.ELEMENT_NODE) {
        return (Element) nd;
      }
      nd = nd.getNextSibling();
    }
    return null;
  }

		/**
		 * HELPER FUNCTION
		 */
private static String getDateString(String dateString, int add) {
	String DATE_FORMAT = "yyyy-MM-dd";
	java.text.SimpleDateFormat sdf =  new SimpleDateFormat(DATE_FORMAT);
	Calendar c1 = Calendar.getInstance(); 

	int year= Integer.parseInt(dateString.substring(0,4));
	int month= Integer.parseInt(dateString.substring(5,7));
	int day= Integer.parseInt(dateString.substring(8,10));
	
	
	c1.set(Calendar.YEAR, year);
	c1.set(Calendar.MONTH, month-1);
	c1.set(Calendar.DATE, day);
	
	c1.add(Calendar.DATE, add);
	String result = sdf.format(c1.getTime());
	return result;
}


private static void outputXmlSlice(Document doc, String ct, String name){
	Document slice = getXmlSlice(doc, ct);
	writeXmlFile(slice, OUTPUT_DIR+"/"+name, false, null);
	
}


private static Document getXmlSlice(Document doc, String ct) {

        Document d = copyDocument(doc); // the copy of doc

        Document slice= null;
        Element root1 =null;

		DocumentBuilderFactory factory;
		DocumentBuilder loader;
		try {
		      factory = DocumentBuilderFactory.newInstance();
		      loader  = factory.newDocumentBuilder();
		      slice = loader.newDocument();
		      root1 = slice.createElement("catalog");
		      slice.appendChild(root1);
		      
		    } catch (ParserConfigurationException ex) {
		      handleError(ex);
		    } catch (FactoryConfigurationError ex) {
		      handleError(ex);
		 }

	    NodeList dd = d.getElementsByTagName("item_Version");
		for (int j=0; j< dd.getLength(); j++){
			  Element cur = (Element)dd.item(j);
			  String e = cur.getAttribute("end");

              // If this item is alive... 
              if (e.compareTo(ct) > 0){
                  Node curItem1 = ((Node) cur).getFirstChild();

                  Node curItem = slice.importNode(curItem1, true);


                  // Should only be one authors
			      Element curAuthors = null;
	              //NodeList dd1 = ((Element) curItem).getElementsByTagName("authors");
	              NodeList dd1 = getAllNodesXPath(".//authors", curItem);
		          for (int i=0; i< dd1.getLength(); i++){
			        curAuthors = (Element)dd1.item(i);

    	            //NodeList dd2 = curAuthors.getElementsByTagName(".author_Version");
	                NodeList dd2 = getAllNodesXPath(".//author_Version", curAuthors);
		            for (int k=0; k< dd2.getLength(); k++){
			            Element curAuthorVersion = (Element)dd2.item(k);
			            String e1 = curAuthorVersion.getAttribute("end");
                        Node curAuthor = ((Node) curAuthorVersion).getFirstChild();
                        if (e1.compareTo(ct) > 0){
                            curAuthors.appendChild(curAuthor);
                        } else {
                            curAuthorVersion.removeChild(curAuthor);
                        }
                    }
                }



  	            //NodeList dd2 = ((Element)curItem).getElementsByTagName("publisher_Version");
	            NodeList dd2 = getAllNodesXPath(".//publisher_Version", curItem);
		            for (int k=0; k< dd2.getLength(); k++){
		            Element curPV = (Element)dd2.item(k);
			            String e1 = curPV.getAttribute("end");
                        Node curPublisher = ((Node) curPV).getFirstChild();
                        if (e1.compareTo(ct) > 0){
                            curItem.appendChild(curPublisher);
                        } else {

                        }
                    }
		   

  	            //dd2 = ((Element)curItem).getElementsByTagName("related_items_Version");
	            dd2 = getAllNodesXPath(".//related_items_Version", curItem);
		            for (int k=0; k< dd2.getLength(); k++){
		            Element curPV = (Element)dd2.item(k);
			            String e1 = curPV.getAttribute("end");
                        Node curRI = ((Node) curPV).getFirstChild();
                        
                        if (e1.compareTo(ct) > 0){
                            //System.out.println(e1+ " is less than " + ct + ": " + ((Element)curRI).getAttribute("related_items_id"));
                            curItem.appendChild(curRI);
                            
                        } else {

                        }
                    }


                  // Remove all old author_items elements (they are empty; content already moved)
	              //dd1 = ((Element) curItem).getElementsByTagName("authors");
	              dd1 = getAllNodesXPath(".//authors", curItem);
		          for (int i=0; i< dd1.getLength(); i++){
			        Node curA = dd1.item(i);
                    for(Node childNode = curA.getFirstChild(); childNode!=null;){
                        Node nextChild = childNode.getNextSibling();
                        if (childNode.getNodeName().equals("author_RepItem")){
                            curA.removeChild(childNode);
                        }
                        childNode = nextChild;
                    }
                  }

                  // Remove all old publisher_items and related_items_items
                    for(Node childNode = curItem.getFirstChild(); childNode!=null;){
                        Node nextChild = childNode.getNextSibling();
                        if (childNode.getNodeName().equals("publisher_RepItem")){
                            curItem.removeChild(childNode);
                        } else if (childNode.getNodeName().equals("related_items_RepItem")){
                            curItem.removeChild(childNode);
                        }
                        childNode = nextChild;
                    }


                //Node n = slice.importNode((Node)curItem, true);
                root1.appendChild(curItem);
              }

        }

        return slice;
}







/**
 * HELPER FUNCTION
 * Just takes the current "active" items and outputs them
 */
private static void shredXmlSlice(Document d, String version) {
	try {
		BufferedWriter outAuthors 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/author."+version+".csv", false));
		BufferedWriter outItems 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item."+version+".csv", false));
		BufferedWriter outPublishers= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/publisher."+version+".csv", false));
		BufferedWriter outRelated 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/related_item."+version+".csv", false));
		BufferedWriter outItemAuth 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_author."+version+".csv", false));
		BufferedWriter outItemPublisher	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_publisher."+version+".csv", false));

		Document slice = getXmlSlice(d, version);
		
///////////////////////////////////////////////////////////////////////
	    NodeList dd = slice.getElementsByTagName("author");
		for (int j=0; j< dd.getLength(); j++){
		  Element cur = (Element)dd.item(j);
		  Element itemParent = (Element)cur.getParentNode().getParentNode();
		  String itemID = itemParent.getAttribute("id");
		  
		  String c0 = cur.getAttribute("author_id");
		  String c1 = getElementValue(cur, "first_name");
		  String c2 = getElementValue(cur, "middle_name");
		  String c3 = getElementValue(cur, "last_name");
		  String c4 = getElementValue(cur, "date_of_birth");
		  String c5 = getElementValue(cur, "biography");
		  String c6 = getElementValue(cur, "street_address");
		  String c7 = getElementValue(cur, "name_of_city");
		  String c8 = getElementValue(cur, "name_of_state");
		  String c9 = getElementValue(cur, "zip_code");
		  String c10 = getElementValue(cur, "name_of_country");
		  String c11 = getElementValue(cur, "phone_number");
		  String c12 = getElementValue(cur, "email_address");

          String format = buildString(13);	
		  String tuple = java.lang.String.format(format, 
				  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12);
		  
          String format2 = buildString(2);	
		  String tuple2 = java.lang.String.format(format2, itemID, c0);

		  outAuthors.write(tuple);
		  outItemAuth.write(tuple2);
		}		
	    outAuthors.close();
	    outItemAuth.close();
		
///////////////////////////////////////////////////////////////////////
	    dd = slice.getElementsByTagName("publisher");
		for (int j=0; j< dd.getLength(); j++){
			  Element cur = (Element)dd.item(j);
			  
			  Element itemParent = (Element)cur.getParentNode();
			  String itemID = itemParent.getAttribute("id");

			  String c0 = cur.getAttribute("publisher_id");
			  String c1 = getElementValue(cur, "name");
			  String c2 = getElementValue(cur, "street_address");
			  String c3 = getElementValue(cur, "name_of_city");
			  String c4 = getElementValue(cur, "name_of_state");
			  String c5 = getElementValue(cur, "zip_code");
			  String c6 = getElementValue(cur, "exchange_rate");
			  String c7 = getElementValue(cur, "currency");
			  String c8 = getElementValue(cur, "phone_number");
			  String c9 = getElementValue(cur, "web_site");
			  String c10 = getElementValue(cur, "FAX_number");
	
              String format = buildString(11);	
			  String tuple = java.lang.String.format(format, 
					  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10);
			  
              String format2 = buildString(2);	
			  String tuple2 = java.lang.String.format(format2, itemID, c0);
	
			  outPublishers.write(tuple);
			  outItemPublisher.write(tuple2);
		}
	    outPublishers.close();
	    outItemPublisher.close();
	    
///////////////////////////////////////////////////////////////////////
		NodeList r = slice.getElementsByTagName("item");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  
			  String c0 = cur.getAttribute("id");
			  String c1 = getElementValue(cur, "ISBN");
			  String c2 = getElementValue(cur, "title");
			  String c3 = getElementValue(cur, "subject");
			  String c4 = getElementValue(cur, "number_of_pages");
			  String c5 = getElementValue(cur, "type_of_book");
			  String c6 = getElementValue(cur, "length");
			  String c7 = getElementValue(cur, "width");
			  String c8 = getElementValue(cur, "height");
			  String c9 = getElementValue(cur, "suggested_retail_price");
			  String c10 = getElementValue(cur, "cost");
			  String c11 = getElementValue(cur, "when_is_available");
			  String c12 = getElementValue(cur, "quantity_in_stock");
			  String c13 = getElementValue(cur, "date_of_release");
			  String c14 = getElementValue(cur, "description");
			  
	          String format = buildString(15);	
			  String tuple = java.lang.String.format(format, 
					  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14);

			  outItems.write(tuple);  
		  }		
		outItems.close();
		
///////////////////////////////////////////////////////////////////////		
		r = slice.getElementsByTagName("related_item");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
		 
			  Node parent = cur.getParentNode().getParentNode();
			  String c0 = ((Element) parent).getAttribute("id");
			  String c1 = getElementValue(cur, "item_id");

			  String format = buildString(2);	
			  String tuple = java.lang.String.format(format, c0, c1);

			  outRelated.write(tuple);
		}		
		outRelated.close();

		
	} catch (IOException e1) {System.err.println(e1);}
}








/**
 * HELPER FUNCTION
 * Just takes the current "active" items and outputs them
 */
private static void shredXmlSlice2(Document d, String version) {
	try {
		BufferedWriter outAuthors 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/author."+version+".csv", false));
		BufferedWriter outItems 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item."+version+".csv", false));
		BufferedWriter outPublishers= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/publisher."+version+".csv", false));
		BufferedWriter outRelated 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/related_item."+version+".csv", false));
		BufferedWriter outItemAuth 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_author."+version+".csv", false));
		BufferedWriter outItemPublisher	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_publisher."+version+".csv", false));

///////////////////////////////////////////////////////////////////////
		// Loop through
	    NodeList dd = d.getElementsByTagName("author_RepItem");
		for (int j=0; j< dd.getLength(); j++){
			Element curItem = (Element)dd.item(j);
			NodeList r = curItem.getElementsByTagName("author_Version");
			for (int i=0; i< r.getLength(); i++){
				  Element cur = (Element)r.item(i);
				  Element itemParent = (Element)cur.getParentNode().getParentNode().getParentNode();
				  String itemID = itemParent.getAttribute("id");
				  
				  String b = cur.getAttribute("begin");
				  String e = cur.getAttribute("end");
				  
				  if (e.compareTo(version) > 0){
				  
					  Element ve = (Element)cur.getFirstChild();
					  
					  //String c0 = java.lang.String.format("%d", counter);
					  String c0 = ve.getAttribute("author_id");
					  String c1 = getElementValue(ve, "first_name");
					  String c2 = getElementValue(ve, "middle_name");
					  String c3 = getElementValue(ve, "last_name");
					  String c4 = getElementValue(ve, "date_of_birth");
					  String c5 = getElementValue(ve, "biography");
					  String c6 = getElementValue(ve, "street_address");
					  String c7 = getElementValue(ve, "name_of_city");
					  String c8 = getElementValue(ve, "name_of_state");
					  String c9 = getElementValue(ve, "zip_code");
					  String c10 = getElementValue(ve, "name_of_country");
					  String c11 = getElementValue(ve, "phone_number");
					  String c12 = getElementValue(ve, "email_address");
	
		              String format = buildString(13);	
					  String tuple = java.lang.String.format(format, 
							  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12);
					  
		              String format2 = buildString(2);	
					  String tuple2 = java.lang.String.format(format2, itemID, c0);
			
					outAuthors.write(tuple);
					outItemAuth.write(tuple2);
				  }
			  }		
		}
	    outAuthors.close();
	    outItemAuth.close();
		
		
		
		
		
///////////////////////////////////////////////////////////////////////
		// Loop through
	    dd = d.getElementsByTagName("publisher_RepItem");
		for (int j=0; j< dd.getLength(); j++){
			Element curItem = (Element)dd.item(j);
			NodeList r = curItem.getElementsByTagName("publisher_Version");
			for (int i=0; i< r.getLength(); i++){
				  Element cur = (Element)r.item(i);
				  
				  Element itemParent = (Element)cur.getParentNode().getParentNode();
				  String itemID = itemParent.getAttribute("id");
				  
				  String b = cur.getAttribute("begin");
				  String e = cur.getAttribute("end");
				  
				  if (e.compareTo(version) > 0){
				  
					  Element ve = (Element)cur.getFirstChild();
					  
					  //String c0 = java.lang.String.format("%d", counter);
					  String c0 = ve.getAttribute("publisher_id");
					  String c1 = getElementValue(ve, "name");
					  String c2 = getElementValue(ve, "street_address");
					  String c3 = getElementValue(ve, "name_of_city");
					  String c4 = getElementValue(ve, "name_of_state");
					  String c5 = getElementValue(ve, "zip_code");
					  String c6 = getElementValue(ve, "exchange_rate");
					  String c7 = getElementValue(ve, "currency");
					  String c8 = getElementValue(ve, "phone_number");
					  String c9 = getElementValue(ve, "web_site");
					  String c10 = getElementValue(ve, "FAX_number");
			
		              String format = buildString(11);	
					  String tuple = java.lang.String.format(format, 
							  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10);
					  
		              String format2 = buildString(2);	
					  String tuple2 = java.lang.String.format(format2, itemID, c0);
			
					  outPublishers.write(tuple);
					  outItemPublisher.write(tuple2);
				  }
			  }
		}
	    outPublishers.close();
	    outItemPublisher.close();
		
	    
///////////////////////////////////////////////////////////////////////
		// Loop through
		NodeList r = d.getElementsByTagName("item_Version");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  
			  String b = cur.getAttribute("begin");
			  String e = cur.getAttribute("end");
			  
			  if (e.compareTo(version) > 0){
			  
				  Element ve = (Element)cur.getFirstChild();
				  
				  String c0 = ve.getAttribute("id");
				  String c1 = getElementValue(ve, "ISBN");
				  String c2 = getElementValue(ve, "title");
				  String c3 = getElementValue(ve, "subject");
				  String c4 = getElementValue(ve, "number_of_pages");
				  String c5 = getElementValue(ve, "type_of_book");
				  String c6 = getElementValue(ve, "length");
				  String c7 = getElementValue(ve, "width");
				  String c8 = getElementValue(ve, "height");
				  String c9 = getElementValue(ve, "suggested_retail_price");
				  String c10 = getElementValue(ve, "cost");
				  String c11 = getElementValue(ve, "when_is_available");
				  String c12 = getElementValue(ve, "quantity_in_stock");
				  String c13 = getElementValue(ve, "date_of_release");
				  String c14 = getElementValue(ve, "description");
				  //String c15 = getElementValue(ve, "publisher_id"); //java.lang.String.format("%d", counter++);
	
		          String format = buildString(15);	
				  String tuple = java.lang.String.format(format, 
						  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14);
	
				  outItems.write(tuple);
			  }
		  }		
		outItems.close();
		
		
		
		
		r = d.getElementsByTagName("related_items_Version");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  
			  // Check to see if there are any related items at all
			  if (cur.getFirstChild().getFirstChild() != null){
				  Element itemParent = (Element)cur.getParentNode().getParentNode();
			  
			  	String b = cur.getAttribute("begin");
			  	String e = cur.getAttribute("end");
			  	
			  	if (e.compareTo(version) > 0){
			  	
				  	NodeList r2 = cur.getElementsByTagName("related_item");
				  	for (int k=0; k< r2.getLength(); k++){
						  Element cur2 = (Element)r2.item(k);
		 
						  String c0 = itemParent.getAttribute("id");
						  String c1 = getElementValue(cur2, "item_id");
	
						  String format = buildString(2);	
						  String tuple = java.lang.String.format(format, c0, c1);
	
						  outRelated.write(tuple);
				  	}
			  	}
			  }
		  }		
		outRelated.close();
	    
	    
	    
	    
		
		
	} catch (IOException e1) {}
	
}

/**
 * HELPER FUNCTION
 */
private static void shredXmlFile(Document d, String version) {
	
	// Need to output 6 csv files
	
try {
	BufferedWriter outAuthors 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/author."+version+".csv", false));
	BufferedWriter outItems 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item."+version+".csv", false));
	BufferedWriter outPublishers= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/publisher."+version+".csv", false));
	BufferedWriter outRelated 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/related_item."+version+".csv", false));
	BufferedWriter outItemAuth 	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_author."+version+".csv", false));
	BufferedWriter outItemPublisher	= new BufferedWriter(new FileWriter(OUTPUT_DIR+"/item_publisher."+version+".csv", false));

	///////////////////////////////////////////////////////////////////////
	// Loop through
    NodeList dd = d.getElementsByTagName("author_RepItem");
	for (int j=0; j< dd.getLength(); j++){
		Element curItem = (Element)dd.item(j);
		NodeList r = curItem.getElementsByTagName("author_Version");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  Element itemParent = (Element)cur.getParentNode().getParentNode().getParentNode();
			  String itemID = itemParent.getAttribute("id");
			  
			  String b = cur.getAttribute("begin");
			  String e = cur.getAttribute("end");
			  
			  Element ve = (Element)cur.getFirstChild();
			  
			  //String c0 = java.lang.String.format("%d", counter);
			  String c0 = ve.getAttribute("author_id");
			  String c1 = getElementValue(ve, "first_name");
			  String c2 = getElementValue(ve, "middle_name");
			  String c3 = getElementValue(ve, "last_name");
			  String c4 = getElementValue(ve, "date_of_birth");
			  String c5 = getElementValue(ve, "biography");
			  String c6 = getElementValue(ve, "street_address");
			  String c7 = getElementValue(ve, "name_of_city");
			  String c8 = getElementValue(ve, "name_of_state");
			  String c9 = getElementValue(ve, "zip_code");
			  String c10 = getElementValue(ve, "name_of_country");
			  String c11 = getElementValue(ve, "phone_number");
			  String c12 = getElementValue(ve, "email_address");

              String format = buildString(15);	
			  String tuple = java.lang.String.format(format, 
					  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, b, e);
			  
              String format2 = buildString(4);	
			  String tuple2 = java.lang.String.format(format2, itemID, c0, b, e);
	
			outAuthors.write(tuple);
			outItemAuth.write(tuple2);
		  }		
	}
    outAuthors.close();
    outItemAuth.close();
    
    
    ///////////////////////////////////////////////////////////////////////
	// Loop through
    dd = d.getElementsByTagName("publisher_RepItem");
	for (int j=0; j< dd.getLength(); j++){
		Element curItem = (Element)dd.item(j);
		NodeList r = curItem.getElementsByTagName("publisher_Version");
		for (int i=0; i< r.getLength(); i++){
			  Element cur = (Element)r.item(i);
			  
			  Element itemParent = (Element)cur.getParentNode().getParentNode();
			  String itemID = itemParent.getAttribute("id");
			  
			  String b = cur.getAttribute("begin");
			  String e = cur.getAttribute("end");
			  
			  Element ve = (Element)cur.getFirstChild();
			  
			  //String c0 = java.lang.String.format("%d", counter);
			  String c0 = ve.getAttribute("publisher_id");
			  String c1 = getElementValue(ve, "name");
			  String c2 = getElementValue(ve, "street_address");
			  String c3 = getElementValue(ve, "name_of_city");
			  String c4 = getElementValue(ve, "name_of_state");
			  String c5 = getElementValue(ve, "zip_code");
			  String c6 = getElementValue(ve, "exchange_rate");
			  String c7 = getElementValue(ve, "currency");
			  String c8 = getElementValue(ve, "phone_number");
			  String c9 = getElementValue(ve, "web_site");
			  String c10 = getElementValue(ve, "FAX_number");
	
              String format = buildString(13);	
			  String tuple = java.lang.String.format(format, 
					  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, b, e);
			  
              String format2 = buildString(4);	
			  String tuple2 = java.lang.String.format(format2, itemID, c0, b, e);
	
			outPublishers.write(tuple);
			outItemPublisher.write(tuple2);
		  }
	}
    outPublishers.close();
    outItemPublisher.close();
    
    
///////////////////////////////////////////////////////////////////////
	// Loop through
	NodeList r = d.getElementsByTagName("item_Version");
	for (int i=0; i< r.getLength(); i++){
		  Element cur = (Element)r.item(i);
		  
		  String b = cur.getAttribute("begin");
		  String e = cur.getAttribute("end");
		  
		  Element ve = (Element)cur.getFirstChild();
		  
		  String c0 = ve.getAttribute("id");
		  String c1 = getElementValue(ve, "ISBN");
		  String c2 = getElementValue(ve, "title");
		  String c3 = getElementValue(ve, "subject");
		  String c4 = getElementValue(ve, "number_of_pages");
		  String c5 = getElementValue(ve, "type_of_book");
		  String c6 = getElementValue(ve, "length");
		  String c7 = getElementValue(ve, "width");
		  String c8 = getElementValue(ve, "height");
		  String c9 = getElementValue(ve, "suggested_retail_price");
		  String c10 = getElementValue(ve, "cost");
		  String c11 = getElementValue(ve, "when_is_available");
		  String c12 = getElementValue(ve, "quantity_in_stock");
		  String c13 = getElementValue(ve, "date_of_release");
		  String c14 = getElementValue(ve, "description");
		  //String c15 = getElementValue(ve, "publisher_id"); //java.lang.String.format("%d", counter++);

          String format = buildString(17);	
		  String tuple = java.lang.String.format(format, 
				  c0, c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, b, e);

		outItems.write(tuple);
	  }		
	outItems.close();
	
	
	
	
	r = d.getElementsByTagName("related_items_Version");
	for (int i=0; i< r.getLength(); i++){
		  Element cur = (Element)r.item(i);
		  
		  // Check to see if there are any related items at all
		  if (cur.getFirstChild().getFirstChild() != null){
			  Element itemParent = (Element)cur.getParentNode().getParentNode();
		  
		  	String b = cur.getAttribute("begin");
		  	String e = cur.getAttribute("end");
		  	
		  	NodeList r2 = cur.getElementsByTagName("related_item");
		  	for (int k=0; k< r2.getLength(); k++){
				  Element cur2 = (Element)r2.item(k);
 
				  String c0 = itemParent.getAttribute("id");
				  String c1 = getElementValue(cur2, "item_id");

				  String format = buildString(4);	
				  String tuple = java.lang.String.format(format, 
						  c0, c1, b, e);

				  outRelated.write(tuple);
		  	}
		  }
	  }		
	outRelated.close();
    
	} catch (IOException e1) {}
}

private static String getYearFromString(String s) {
    return s.substring(0, 4);
}

/**
 * HELPER FUNCTION
 */
private static String buildString(int num) {
    String result = "%s";
    for (int i=1; i< num; ++i){
        result += "&%s";
    }
    return result+"\n";
}

private static Document copyDocument(Document d1){
		DocumentBuilderFactory factory;
		DocumentBuilder loader;
		Document d2 = null;
		try {
		      factory = DocumentBuilderFactory.newInstance();
		      loader  = factory.newDocumentBuilder();
		      d2 = loader.newDocument();
              d2.appendChild(d2.importNode((Node)d1.getDocumentElement(), true));
        }catch(Exception e){}
        return d2;

}

private static void printNode(Node node){
 try
    {
      // Set up the output transformer
      TransformerFactory transfac = TransformerFactory.newInstance();
      Transformer trans = transfac.newTransformer();
      trans.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
      trans.setOutputProperty(OutputKeys.INDENT, "yes");
      // Print the DOM node
      StringWriter sw = new StringWriter();
      StreamResult result = new StreamResult(sw);
      DOMSource source = new DOMSource(node);
      trans.transform(source, result);
      String xmlString = sw.toString();

      System.out.println(xmlString);
    }
    catch (TransformerException e)
    {
      e.printStackTrace();
    }
}

/**
 * HELPER FUNCTION
 */
private static String getElementValue(Element ve, String string) {
	Node a = ve.getElementsByTagName(string).item(0);
	if (a != null){
		return a.getFirstChild().getNodeValue();
	}	else return NULL_VALUE;
}

/**
 * HELPER FUNCTION
 */
private static void readSettings(String file) {
try{
    FileInputStream fstream = new FileInputStream(file);
    DataInputStream in      = new DataInputStream(fstream);
    BufferedReader br       = new BufferedReader(new InputStreamReader(in));
    String strLine;
    while ((strLine = br.readLine()) != null)   {
    	
        if (strLine.matches("^ *$") || strLine.matches("#.*")){
    	    //just a blank line
        } else {
            String setting = strLine.split(" +")[0];
            String value   = strLine.split(" +")[1];
            setSetting(setting, value); 
        }
    }

    in.close();
    }catch (Exception e){
      System.err.println("tGenerator: Reading input error: " + e.getMessage());
      System.exit(1);
    }

}

private static void setSetting(String setting, String value) {

	      if (setting.equals("INITIAL_PERCENTAGE"))    { INITIAL_PERCENTAGE    = Double.parseDouble(value); }
	      else if (setting.equals("OUTPUT_XML_SLICES"))     { OUTPUT_XML_SLICES     = Integer.parseInt(value); }
	      else if (setting.equals("OUTPUT_RELATION_SLICES")){ OUTPUT_RELATION_SLICES= Integer.parseInt(value); }
	      else if (setting.equals("OUTPUT_XML"))     	   { OUTPUT_XML     	   = Integer.parseInt(value); }
	      else if (setting.equals("OUTPUT_RELATIONS"))      { OUTPUT_RELATIONS      = Integer.parseInt(value); }
	      else if (setting.equals("TIME_STEP"))             { TIME_STEP             = Integer.parseInt(value); }
	      else if (setting.equals("NUMBER_SLICES"))         { NUMBER_SLICES         = Integer.parseInt(value); }
	      else if (setting.equals("REQUIRED_CHANGES"))      { REQUIRED_CHANGES      = Integer.parseInt(value); }
	      else if (setting.equals("INPUT_FILE"))            { INPUT_FILE            = value; }
	      else if (setting.equals("OUTPUT_DIR"))            { OUTPUT_DIR            = value; }
	      else if (setting.equals("START_TIME"))            { START_TIME            = value; }
	      else if (setting.equals("FOREVER_TIME"))          { FOREVER_TIME          = value; }
	      else if (setting.equals("SELECTION_TYPE"))        { SELECTION_TYPE        = value; }
	      else if (setting.equals("SELECTION_STDDEV"))      { SELECTION_STDDEV      = Double.parseDouble(value); }
	      else if (setting.equals("NULL_VALUE"))       	    { NULL_VALUE            = value.replaceAll("\"", ""); }
	      else if (setting.equals("DEBUG_OUTPUT"))          { DEBUG_OUTPUT          = Integer.parseInt(value); }
	      else if (setting.equals("ITEM_PER_TIME_STEP_INSERTS")) { ITEM_PER_TIME_STEP[0] = Integer.parseInt(value); }
	      else if (setting.equals("ITEM_PER_TIME_STEP_DELETES")) { ITEM_PER_TIME_STEP[1] = Integer.parseInt(value); }
	      else if (setting.equals("ITEM_PER_TIME_STEP_UPDATES")) { ITEM_PER_TIME_STEP[2] = Integer.parseInt(value); }
	      else if (setting.equals("AUTHOR_PER_TIME_STEP_INSERTS")) { AUTHOR_PER_TIME_STEP[0] = Integer.parseInt(value); }
	      else if (setting.equals("AUTHOR_PER_TIME_STEP_DELETES")) { AUTHOR_PER_TIME_STEP[1] = Integer.parseInt(value); }
	      else if (setting.equals("AUTHOR_PER_TIME_STEP_UPDATES")) { AUTHOR_PER_TIME_STEP[2] = Integer.parseInt(value); }
	      //else if (setting.equals("PUBLISHER_PER_TIME_STEP_INSERTS")) { PUBLISHER_PER_TIME_STEP[0] = Integer.parseInt(value); }
	      //else if (setting.equals("PUBLISHER_PER_TIME_STEP_DELETES")) { PUBLISHER_PER_TIME_STEP[1] = Integer.parseInt(value); }
	      else if (setting.equals("PUBLISHER_PER_TIME_STEP_DELETES_INSERTS")) { PUBLISHER_PER_TIME_STEP[3] = Integer.parseInt(value); }
	      else if (setting.equals("PUBLISHER_PER_TIME_STEP_UPDATES")) { PUBLISHER_PER_TIME_STEP[2] = Integer.parseInt(value); }
	      //else if (setting.equals("RELATED_PER_TIME_STEP_INSERTS")) { RELATED_PER_TIME_STEP[0] = Integer.parseInt(value); }
	      //else if (setting.equals("RELATED_PER_TIME_STEP_DELETES")) { RELATED_PER_TIME_STEP[1]  = Integer.parseInt(value); }
	      else if (setting.equals("RELATED_PER_TIME_STEP_DELETES_INSERTS")) { RELATED_PER_TIME_STEP[3]  = Integer.parseInt(value); }
	      else if (setting.equals("RELATED_PER_TIME_STEP_UPDATES")) { RELATED_PER_TIME_STEP[2]  = Integer.parseInt(value); }
	      else {
	    	  System.err.println("tGenerator: Unknown parameter-pair: "+ setting);
	      }
}

private static void printItem(Node n, String sep){
	
	String name = n.getNodeName();
	if (!(name.contains("item") || name.contains("author") || name.contains("publisher") || name.contains("related_item"))){
		
	} else {
		
		if (n.getNodeType() == Node.ELEMENT_NODE){
			System.out.print(sep + n.getNodeName() + ": ");
			Element e = (Element )n;
			NamedNodeMap nnm = e.getAttributes();
			for (int i=0; i< nnm.getLength(); ++i){
				System.out.print(nnm.item(i) + " ");
			}
			if (name.equals("item_id")){
				System.out.println(n.getTextContent());
			}
			System.out.println("");
		}
	}

	Node child = n.getFirstChild();
	if (child != null){
		printItem(child, sep + "  ");
	}
	Node next = n.getNextSibling();
	if (next != null){
		printItem(next, sep);
	}
}

private static NodeList getAllNodesXPath(String path, Object p) {
	try{
      NodeList e = (NodeList) xpath.evaluate(path, p, XPathConstants.NODESET);
      return e;
    } catch (Exception e){System.out.println(e);};
    return null;
}


private static Node getItemFromXPath(String path, Document p) {
	try
    {
      NodeList e = (NodeList) xpath.evaluate(path, p, XPathConstants.NODESET);
      return (Node)e.item(0);
    } catch (Exception e){System.out.println(e);};
    return null;
}


private static void printNodeXPath(String path, Document p){
	try
    {
      NodeList e = (NodeList) xpath.evaluate(path, p, XPathConstants.NODESET);
      
      System.out.println("xpath \"" + path + " found " + e.getLength() + " items:");
      for (int i = 0; i<e.getLength();++i){
    	  Node n = e.item(i);
    	  System.out.println((i+1)+" of "+e.getLength()+":");
    	  printItem(n, "");
      }
    } catch (Exception e){System.out.println(e);};
}

}
