/**
 * 
 */
package cs.arizona.util;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import cs.arizona.tau.xml.TXSchemaConstants;
 
/**
 * @author spjoshi
 * TODO
 * 1) Following element types are supported
 * 				<xs:complexContent>
				<xs:complexType>   
				<xs:element>           
				<xs:group>           
				<xs:sequence>       
				<xs:choice>       
				<xs:attribute>       
				<xs:attributeGroup>
 * 2) Following element types are not yet supported. But could be supported easily with small changes.
 * 				<xs:any>               
				<xs:anyAttribute>   
				<xs:extension>       
				<xs:restriction>
 * 3) More testing of function "getTopLevelElement" is needed. It returns the top level element (which has not been referred by any other element.)
 * 4) Currently function getXMLSchemaName is not required. If not required, should be removed in future.				
 */
public class SchemaPathEvaluator {

	Document schemaDoc;			//Stores the DOM tree for Normalized Snapshot Schema
	String xmlSchemaName = TXSchemaConstants.SCHEMA_NAME_PREFIX; 		//The variable stores the name given to W3C specification schema "http://www.w3.org/2001/XMLSchema"
	Hashtable elementTable;
	Hashtable groupTable;
	Hashtable complexTypeTable;
	Hashtable attributeGroupTable;
	
	
	
	/**
	 * @schemaDoc - is the Document tree for the schema for which SchemaPath expressions will be evaluated. 
	 */
	public SchemaPathEvaluator(Document schemaDoc){
		this.schemaDoc = schemaDoc;
		fillHashTables();
	}
	
	/**
	 * The function will validate the given SchemaPath using the sample document created internally for given Snapshot Schema
	 * @param path
	 * @return
	 */
	public Element validateSchemaPath(String path){
		Element topLevelElement = getTopLevelElement();
		Element targetElement = validatePath(topLevelElement,path);
		//System.out.println("");
		return targetElement;
	}
	
	private Element validatePath(Element currentElement, String path){
		String elementName = currentElement.getLocalName();
		NodeList nl;
		Element targetElement = null;
		String remainingPath;
		//If element local name is "element"
		if (elementName == "element"){
			//If the element is referencing some other element, fetch that element from elementTable call function recursively
			if (currentElement.getAttribute("ref")!=""){
				targetElement = validatePath((Element)elementTable.get(currentElement.getAttribute("ref")),path);
			}else{
				//If the element is not referencing any other element
				String name = currentElement.getAttribute("name");
				//At this stage if current element local name satisfies the path
				if (path.startsWith("/"+name)){
					//if the element is the last element in the path
					if (path.equals("/"+name)){
						targetElement = currentElement;
					}else{
						//if element is not the last element in the path
						//get remaining path
						remainingPath = getRemainingPath(path);
						//If the element is of type some comlexType
						if (currentElement.getAttribute("type")!=""){
							//Fetch corresponding element from the complexTypeTable and call function validatePath on that node
							targetElement = validatePath((Element)complexTypeTable.get(currentElement.getAttribute("type")),remainingPath);	
						}else{
							//Get list of all child elements.
							nl=currentElement.getChildNodes();
							//For every child element check
							for (int i=0 ; i<nl.getLength(); i++){
								//Check if that element is of type ELEMENT_NODE
								if (nl.item(i).getNodeType() == Node.ELEMENT_NODE ){
									Element element = (Element)nl.item(i);
									//Recursively call function path
									targetElement= validatePath(element,remainingPath);
									//If returned element is not NULL, then return it. 
									if (targetElement != null){
										 break;
									}
								}
							}
						}
					}
				}else{
					//If current element does not satisfy the path 
					targetElement=null;	
				}
			}
		}else if (elementName == "group"){		//If element local name is "group" 
			if (currentElement.getAttribute("ref")!=""){		//If it is referencing any other group
				Element groupElement = (Element)groupTable.get(currentElement.getAttribute("ref"));
				//Call function validatePath on the corresponding element from groupTable 
				targetElement = validatePath(groupElement,path);
				return targetElement;
			}else{
				//Get list of all child elements.
				nl=currentElement.getChildNodes();
				//For every child element check
				for (int i=0 ; i<nl.getLength(); i++){
					//Check if that element is of type ELEMENT_NODE
					if (nl.item(i).getNodeType() == Node.ELEMENT_NODE ){
						Element element = (Element)nl.item(i);
						//Recursively call function path
						targetElement= validatePath(element,path);
						//If returned element is not NULL, then return it. 
						if (targetElement != null){
							 break;
						}
					}
				}
			}
		}else{
			//If element is of some other type (e.g. complexContent, sequence, choice etc.)
			//Get list of all child elements.
			nl=currentElement.getChildNodes();
			//For every child element check
			for (int i=0 ; i<nl.getLength(); i++){
				//Check if that element is of type ELEMENT_NODE
				if (nl.item(i).getNodeType() == Node.ELEMENT_NODE ){
					Element element = (Element)nl.item(i);
					//Recursively call function path
					targetElement= validatePath(element,path);
					//If returned element is not NULL, then return it. 
					if (targetElement != null){
						 break;
					}
				}
			}
		}
		return targetElement;
	}

	// TODO: Add functionality here.
	public boolean validateField(Element element, String fieldPath){
		if (fieldPath.equals("/text")){
			return true;
		}else
		if (fieldPath.startsWith("/@")){
			NodeList nl = element.getElementsByTagName(xmlSchemaName + "attribute");
			for (int i=0 ; i<nl.getLength() ; i++){
				if (fieldPath.substring(2).equals(((Element)nl.item(i)).getAttribute("name"))){
					return true;
				}
			}
			TauLogger.logger.error("Target path \"" + fieldPath + "\" was not found.");
			return false;
		}else{
			TauLogger.logger.error("Target path \"" + fieldPath + "\" does not start wtih \"/@\".");
			return false;
		}
	}
	
	/**
	 * The function returns the top element in the document.
	 * @return
	 */
	public Element getTopLevelElement(){
		Hashtable tempElementTable = new Hashtable();
		Enumeration elementKeys = elementTable.keys();

		//Replicate elementTable into tempElementTable
		while (elementKeys.hasMoreElements()){
			String key = (String)elementKeys.nextElement();
			tempElementTable.put(key,elementTable.get(key));
		}
		Element schemaElement = schemaDoc.getDocumentElement();
		Element topLevelElement = null;
		NodeList nl = schemaElement.getElementsByTagName(xmlSchemaName + "element");
		for (int i=0 ; i<nl.getLength() ; i++){
			if (nl.item(i).getNodeType() == Node.ELEMENT_NODE ){
				Element currentElement = (Element)nl.item(i);
				
				//Want to remove "element" elements that are actually just refs
				if (currentElement.getAttribute("ref")!=""){
					tempElementTable.remove(currentElement.getAttribute("ref"));
				}
			}
		}
		
		Enumeration enumeration = tempElementTable.elements();
		while (enumeration.hasMoreElements()){
			topLevelElement = (Element)enumeration.nextElement();	
		}
		return topLevelElement;
	}
	
	/* The function returns the child attribute elements for the given element */
	public Hashtable getChildAttributes(String path){

		Hashtable attributeTable = new Hashtable();
		Element e = validateSchemaPath(path);
		
		NodeList nl = e.getElementsByTagName(xmlSchemaName + "attributeGroup");
		
		for (int i=0 ; i<nl.getLength() ; i++){
			
			Element attrGroupElement = (Element)nl.item(i);
			attrGroupElement = (Element)attributeGroupTable.get(attrGroupElement.getAttribute("ref"));

			NodeList nl1 = attrGroupElement.getElementsByTagName(xmlSchemaName + "attribute");
			for (int j=0 ; j<nl.getLength() ; j++){
				Element attrElement = (Element)nl1.item(i);
				attributeTable.put(attrElement.getAttribute("name"), attrElement);
			}
		}
		
		nl = e.getElementsByTagName(xmlSchemaName + "attribute");

		for (int i=0 ; i<nl.getLength() ; i++){
			Element attrElement = (Element)nl.item(i);
			attributeTable.put(attrElement.getAttribute("name"), attrElement);
		}
		return attributeTable;
	}
	
	/* The function returns the immediate child elements of the given element
	 * TODO: The function right now supports only <sequence> element.
	 * 		 Either need to be modified or need to write a new function that would return 
	 * 		 elements enclosed in <choice> and <all>
	 */
	public ArrayList getChildElements(String path){
		ArrayList childElementList = new ArrayList();
		Element e = validateSchemaPath(path);
		
		if (null != e.getAttribute("type")){
			e = (Element)complexTypeTable.get(e.getAttribute("type"));
		}

		addGroupElements(childElementList, e);
		
		NodeList nl = e.getElementsByTagName(xmlSchemaName + "element"); 
		for (int i=0 ; i<nl.getLength() ; i++){
			Element childElement = (Element)nl.item(i);
			childElementList.add(childElement);
		}
		
		return childElementList;
	}
	

	/* Given a element, the function adds all the elements in all the nested groups recursively to the childElementList */ 
	private void addGroupElements(ArrayList childElementList, Element e){
		NodeList nl = e.getElementsByTagName(xmlSchemaName + "group"); 
		
		for (int i=0 ; i<nl.getLength() ; i++){
			
			Element groupElement = (Element)nl.item(i);
			groupElement = (Element)groupTable.get(groupElement.getAttribute("ref"));
			
			addGroupElements(childElementList, groupElement);

			NodeList nl1 = groupElement.getElementsByTagName(xmlSchemaName + "element");
			for (int j=0 ; j<nl.getLength() ; j++){
				Element childElement = (Element)nl1.item(i);
				childElementList.add(childElement);
			}
		}
	}
	
	/**
	 * The function stores the name given to W3C specification schema "http://www.w3.org/2001/XMLSchema" in xmlSchemaName.
	 * 
	 */
	private void getXMLSchemaName(){
		//Get attributes of topmost element (which is schema in this case)	
		NamedNodeMap attributes = schemaDoc.getDocumentElement().getAttributes();
		for (int i=0 ; i<attributes.getLength() ; i++){
			Attr attr = (Attr)attributes.item(i);
			String attrName = attr.getName();
			String attrValue = attr.getValue();
			//If attribute's name contains xmlns, check whether its value is "http://www.w3.org/2001/XMLSchema". If yes, store the name in 
			//variable xmlSchemaName
			if (attrName.indexOf("xmlns") != -1){
				if (attrValue.equals("http://www.w3.org/2001/XMLSchema")){
					xmlSchemaName=attrName.substring(attrName.indexOf(":")+1) + ":";
				}
			}
		}
	}
	
	private void fillHashTables(){
		elementTable = new Hashtable();
		groupTable = new Hashtable();
		complexTypeTable = new Hashtable();
		attributeGroupTable = new Hashtable();
		Element schemaElement= schemaDoc.getDocumentElement();
		NodeList nl= schemaElement.getChildNodes();
		for (int i=0 ; i<nl.getLength() ; i++){
			if (nl.item(i).getNodeType() == Node.ELEMENT_NODE ){
				Element element = (Element)nl.item(i);
				if ("element".equals(element.getLocalName())){
					elementTable.put(element.getAttribute("name"),element);
				}else
					if ("group".equals(element.getLocalName())){
						groupTable.put(element.getAttribute("name"),element);
					}else
						if ("complexType".equals(element.getLocalName())){
							complexTypeTable.put(element.getAttribute("name"),element);
						}else
							if ("attributeGroup".equals(element.getLocalName())){
								attributeGroupTable.put(element.getAttribute("name"),element);
							}
			}
		}
	}
	
	private String getRemainingPath(String path){
		return path.substring(path.indexOf("/",1));
	}

        public String printTable(char c) {
               switch(c) {
 		 case 'e': return elementTable.toString();
                 case 'g': return groupTable.toString();
		 case 'c': return complexTypeTable.toString();
		 case 'a': return attributeGroupTable.toString();
                 default: return "";
               }
        }
}
