package cs.arizona.tau.xml.temporalconstraint;

import java.io.File;
import java.net.URL;

import java.util.*;

import org.w3c.dom.*;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSParser;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.DefaultHandler2;
import org.xml.sax.helpers.DefaultHandler;

import cs.arizona.tau.docs.ConstantSchemaTemporalDocument;
import cs.arizona.tau.docs.TemporalSchema;
import cs.arizona.tau.time.ITimePeriod;
import cs.arizona.tau.xml.Common;
import cs.arizona.tau.xml.DecomposedRepresentationFactory;
import cs.arizona.tau.xml.DoUnSquashing;
import cs.arizona.tau.xml.IDoUnSquashing;
import cs.arizona.tau.xml.IRepresentationFactory;
import cs.arizona.tau.xml.LogicalAnnotationValidator;
import cs.arizona.tau.xml.PhysicalAnnotationValidator;
import cs.arizona.tau.xml.TXSchemaConstants;
import cs.arizona.util.ConventionalParser;
import cs.arizona.util.TargetIdentifier;
import cs.arizona.util.TauLogger;
import cs.arizona.util.ValidatorProperties;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

// This class is a hack to ignore an annoying problem, temporarily.
// It should be replaced by the real solution to the problem.
// The problem is when validate() is called, it says
// "Cannot find the declaration of element 'catalog'" but the catalog
// element is right there.
class MyErrorHandler implements ErrorHandler {
  public void fatalError(SAXParseException up) throws SAXException {
    System.out.println(up.toString());
    throw up;
  }

  
  public void error(SAXParseException up) throws SAXException {
    String message = up.getMessage();
    if (message.contains(
            "Cannot find the declaration of element")) {
      // just ignore this unsolvable mystery some other human programmer
      // created for a small subset of the programmers in the rest of the world.
    } else if (message.contains("The content of element")) {
    	
    } else {
      System.err.println(message);
      //throw up;
    }
  }
  
  
  public void warning(SAXParseException ex) throws SAXException {
    System.out.println(ex.toString());
  }
}


/**
 * Validating the temporal constraints of a temporal document
 * @author ruizhang
 *
 */
public class TemporalConstraintValidator {
  private Document temporal_doc_;
  private Document annotation_doc_;
  // the conventional schema used by referential constraint 
  private Document schema_doc_;
  private HashMap<String, Vector<Constraint> > constraint_map_;
  // the applicability implied by the lifetime of the temporal document
  // this is used when the "applicability" attribute is not specified
  private Applicability default_applicability_; 

  public static String NONSEQKEY = "nonSeqKey";
  public static String NONSEQUNIQUE = "nonSeqUnique";
  public static String UNIQUENULLRESTRICTED = "uniqueNullRestricted";
  public static String SEQCARDINALITY = "seqCardinality";
  public static String NONSEQCARDINALITY = "nonSeqCardinality";
  public static String TRANSITION = "transitionConstraint";
  public static String NONSEQKEYREF = "nonSeqKeyref";
  
  
  /**
   * Constructing the validator instance.
   * @param temporal_doc the temporal document to be validated
   * @param temporal_schema_name the name of the temporal schema
   */
  public TemporalConstraintValidator(Document temporal_doc,
                                     String temporal_schema_name) {
	ValidatorProperties vp = ValidatorProperties.getInstance();
	ConventionalParser cp = ConventionalParser.getInstance();
	TemporalSchema ts = new TemporalSchema(vp, cp, temporal_schema_name);
	annotation_doc_ = ts.getAnnotationDoc();
	schema_doc_ = ts.getConventionalSchemaDoc();
	TargetIdentifier ti = new TargetIdentifier(schema_doc_);
    Document defaultAnnotationDoc = Common.createDefaultAnnotationDoc(
        cp, Common.getTopLevelSchemaElementName(ti));
    LogicalAnnotationValidator lav = new LogicalAnnotationValidator(
    		annotation_doc_,schema_doc_,ti,true);
    PhysicalAnnotationValidator defaultPav = new PhysicalAnnotationValidator(
        defaultAnnotationDoc,schema_doc_,ti,true);
    ConstantSchemaTemporalDocument internal_temporal_document =
        new ConstantSchemaTemporalDocument(ti, lav, defaultPav, temporal_doc);
    temporal_doc_ = internal_temporal_document.getInternalTemporalDocument();
    Element root_element = temporal_doc_.getDocumentElement();
    default_applicability_ =
        new Applicability(root_element.getAttribute("begin"),
                          root_element.getAttribute("end"));
  }
  
  
  public Document GetTemporalDocument() {
    return temporal_doc_;
  }
  
  
  public Document GetAnnotationDocument() {
    return annotation_doc_;
  }
  
  
  public Document GetSchemaDocument() {
    return schema_doc_;
  }
  
  
  /**
   * Parsing the annotation document and extracting the defined constraints
   * for validation.
   * @param annotation_doc the annotation document where the constraints are
   * defined
   */
  public void ExtractConstraint(Document annotation_doc) {
    constraint_map_ = new HashMap<String, Vector<Constraint> >();
    Element annotation_root = annotation_doc.getDocumentElement();
    Vector<Constraint> nonseqkey_fields = new Vector<Constraint>();
    Vector<Constraint> nonsequnique_fields = new Vector<Constraint>();
    Vector<Constraint> uniquenullrestricted_fields = new Vector<Constraint>();
    Vector<Constraint> seqidentity_fields = new Vector<Constraint>();
    Vector<Constraint> seqcard_fields = new Vector<Constraint>();
    Vector<Constraint> nonseqcard_fields = new Vector<Constraint>();
    Vector<Constraint> transition_fields = new Vector<Constraint>();
    Vector<Constraint> nonseqkeyref_fields = new Vector<Constraint>();
    Vector<Constraint> keyref_fields = new Vector<Constraint>();
    // 6.1
    ExtractConstraints(annotation_root, NONSEQKEY, nonseqkey_fields);
    constraint_map_.put(NONSEQKEY, nonseqkey_fields);
    ExtractConstraints(annotation_root, NONSEQUNIQUE, nonsequnique_fields);
    constraint_map_.put(NONSEQUNIQUE, nonsequnique_fields);
    ExtractConstraints(annotation_root, UNIQUENULLRESTRICTED,
                       uniquenullrestricted_fields);
    /*
    constraint_map_.put(NONSEQUNIQUE, uniquenullrestricted_fields);
    ExtractConstraints(annotation_root, SEQIDENTITY,
                       seqidentity_fields);
    constraint_map_.put(SEQIDENTITY, seqidentity_fields);
    */
    // 6.2
    ExtractConstraints(annotation_root, NONSEQKEYREF, nonseqkeyref_fields);
    constraint_map_.put(NONSEQKEYREF, nonseqkeyref_fields);
    // 6.3
    ExtractConstraints(annotation_root, SEQCARDINALITY, seqcard_fields);
    constraint_map_.put(SEQCARDINALITY, seqcard_fields);
    ExtractConstraints(annotation_root, NONSEQCARDINALITY, nonseqcard_fields);
    constraint_map_.put(NONSEQCARDINALITY, nonseqcard_fields);
    // 6.4
    ExtractConstraints(annotation_root, TRANSITION, transition_fields);
    constraint_map_.put(TRANSITION, transition_fields);
    TauLogger.logger.info("All Constraints");
    System.out.println("*********************************");
    TauLogger.logger.info("<nonseqkey>");
    for (Constraint s : nonseqkey_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<nonsequnique>");
    for (Constraint s : nonsequnique_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<uniquenullrestricted>");
    for (Constraint s : uniquenullrestricted_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<seqidentity>");
    for (Constraint s : seqidentity_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<keyref>");
    for (Constraint s : keyref_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<nonseqkeyref>");
    for (Constraint s : nonseqkeyref_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("<seqcardinality>");
    for (Constraint s : seqcard_fields) {
      TauLogger.logger.info(s.toString() +
                         " -- " + ((CardinalityConstraint)s).getMaxOccur());
    }
    TauLogger.logger.info("<nonseqcardinality>");
    for (Constraint s : nonseqcard_fields) {
      TauLogger.logger.info(s.toString() +
                         " -- " + ((CardinalityConstraint)s).getMaxOccur());
    }
    TauLogger.logger.info("<transition>");
    for (Constraint s : transition_fields) {
      TauLogger.logger.info(s.toString());
    }
    TauLogger.logger.info("*********************************");
  }
  
  
  /**
   * Use the conventional indetifier/constraint to look for
   * the defined identifer/constraint in the snapshot schema
   * @param conv_id_name the conventional identifier
   * @param schema_doc the snapshot schema
   * @return a string vector with the elements being
   * selector, field and refer, in case of parsing "keyref"
   */
  public static Vector<String> ConvertConventionalIdentifier(
      String conv_id_name, Document schema_doc) {
    Vector<String> ret_vec = new Vector<String>();
    int identifier = -1; // 0 unique
                         // 1 key
                         // 2 keyref
    try {
      NodeList keyref_def = schema_doc.getElementsByTagName(
          TXSchemaConstants.SCHEMA_NAME_PREFIX + "unique");
      Vector<Node> def_vec = new Vector<Node>();
      for (int i = 0; i < keyref_def.getLength(); ++i) {
        Node tmp_node = keyref_def.item(i);
        if (((Element)tmp_node).getAttribute("name").equals(conv_id_name)) {
          def_vec.add(tmp_node);
        }
        if (def_vec.size() > 0) {
          identifier = 0;
        }
      }
      if (def_vec.size() == 0) {
        keyref_def = schema_doc.getElementsByTagName(
            TXSchemaConstants.SCHEMA_NAME_PREFIX + "key");
        def_vec = new Vector<Node>();
        for (int i = 0; i < keyref_def.getLength(); ++i) {
          Node tmp_node = keyref_def.item(i);
          if (((Element)tmp_node).getAttribute("name").equals(conv_id_name)) {
            def_vec.add(tmp_node);
          }
        }
        if (def_vec.size() > 0) {
          identifier = 1;
        }
      }
      if (def_vec.size() == 0) {
        keyref_def = schema_doc.getElementsByTagName(
            TXSchemaConstants.SCHEMA_NAME_PREFIX + "keyref");
        // in this case, the "refer" attribute is returned as the 3rd element 
        def_vec = new Vector<Node>();
        for (int i = 0; i < keyref_def.getLength(); ++i) {
          Node tmp_node = keyref_def.item(i);
          if (((Element)tmp_node).getAttribute("name").equals(conv_id_name)) {
            def_vec.add(tmp_node);
          }
        }
        if (def_vec.size() > 0) {
          identifier = 2;
        }
      }
      
      if (def_vec.size() != 1) {
        System.err.println(
            "conventionalIdetifier " + conv_id_name +
            " not found or not uniquely declared. Def found " +
            def_vec.size());
        return null;
      }
      Element keyref_element = (Element)def_vec.get(0);
      ret_vec.add(((Element)keyref_element.getElementsByTagName(
                      TXSchemaConstants.SCHEMA_NAME_PREFIX + "selector").
                          item(0)).getAttribute("xpath"));
      ret_vec.add(((Element)keyref_element.getElementsByTagName(
                      TXSchemaConstants.SCHEMA_NAME_PREFIX + "field").
                          item(0)).getAttribute("xpath"));
      if (identifier == 2) {
        ret_vec.add(keyref_element.getAttribute("refer"));
      }
    } catch (Exception ex) {
      ex.printStackTrace();
      return null;
    }
    return ret_vec;
  }
  
  
  /**
   * Parsing annotation document and extracting the defined constraints.
   * @param root the root element of the annotation document
   * @param target the type name of the constraint 
   * @param results the result vector holding all the found constraint
   */
  public void ExtractConstraints(Element anno_root, String target,
                                 Vector<Constraint> results) {
    NodeList defined_items = null;
    String item_target = null;
    if (anno_root.getNodeName().equals("logical") &&
            anno_root.getNodeType() == Node.ELEMENT_NODE) {
      defined_items = anno_root.getElementsByTagName("constraint");
      for (int i = 0; i < defined_items.getLength(); ++i) {
        String item_identifier = null;
        Element item_node = (Element)defined_items.item(i);
        item_target = item_node.getAttribute("target");
        NodeList item_children = item_node.getChildNodes();
        for (int j = 0; j < item_children.getLength(); ++j) {
    	  Node item_child_node = item_children.item(j);
          if (item_child_node.getNodeName().equals("itemIdentifier") &&
                  (item_child_node.getNodeType() == Node.ELEMENT_NODE)) {
            item_identifier =
                ((Element)(((Element)item_child_node).
                    getElementsByTagName("field").item(0))).
                        getAttribute("path");
          }
        }
        for (int j = 0; j < item_children.getLength(); ++j) {
      	  Node root = item_children.item(j);
          if (root.getNodeName().equals(target) &&
                  (root.getNodeType() == Node.ELEMENT_NODE)) {
            String name = ((Element)root).getAttribute("name");
            String dimension = ((Element)root).getAttribute("dimension");
            if (dimension == null || dimension.equals("")) {
              dimension = "validTime";
            }
            String selector = null;
            Element selector_element =
                (Element)(((Element)root).getElementsByTagName(
                    "selector").item(0));
            if (selector_element != null) {
              selector = selector_element.getAttribute("xpath");
            }
            String field = null;
            Element field_element =
                (Element)(((Element)root).getElementsByTagName(
                    "field").item(0));
            if (field_element != null) {
              field = field_element.getAttribute("xpath");
            }
            String evaluation_window_attr =
            	((Element)root).getAttribute("evaluationWindow");
            int evaluation_window = -1;
            if (evaluation_window_attr != null &&
                    !evaluation_window_attr.equals("")) {
              evaluation_window = Integer.parseInt(evaluation_window_attr);
            }
            String slide_size_attr = ((Element)root).getAttribute("slideSize");
            int slide_size = 1;
            if (slide_size_attr != null &&
                !slide_size_attr.equals("")) {
              slide_size = Integer.parseInt(slide_size_attr);
            }

            Applicability applicability = null;
            Element app_element =
                (Element)((((Element)root).getElementsByTagName(
                    "applicability")).item(0));
            if (app_element != null) {
              applicability = new Applicability(
                  app_element.getAttribute("begin"),
                  app_element.getAttribute("end"));
            } else {
              applicability = default_applicability_;
            }
    	      
            Constraint new_constraint = null;
            if (target.equals(NONSEQUNIQUE) ||
                target.equals(NONSEQKEY) ||
                target.equals(UNIQUENULLRESTRICTED)) {
              String scope = ((Element)root).getAttribute("scope");
              int null_count_min = -1;
              int null_count_max = -1;
              String null_cnt_min_attr =
            	  ((Element)root).getAttribute("nullCountMin");
              if (null_cnt_min_attr != null && !null_cnt_min_attr.equals("")) {
                null_count_min = Integer.parseInt(null_cnt_min_attr);
              }
              String null_cnt_max_attr =
            	  ((Element)root).getAttribute("nullCountMax");
              if (null_cnt_max_attr != null && !null_cnt_max_attr.equals("")) {
                null_count_max = Integer.parseInt(null_cnt_max_attr);
              }
              String conventional_identifier =
            	  ((Element)root).getAttribute("conventionalIdentifier");
              if (conventional_identifier == null ||
                  conventional_identifier.equals("")) {
                conventional_identifier = null;
              } else {
                Vector<String> selector_field =
                    ConvertConventionalIdentifier(conventional_identifier,
                                                  schema_doc_);
                if (selector_field == null) {
                  System.exit(3);
                }
                selector = selector_field.get(0);
                field = selector_field.get(1);
              }
              new_constraint = new IdentityConstraint(
                  name, item_target, item_identifier,
                  dimension, selector, field, evaluation_window, slide_size,
                  applicability, scope, conventional_identifier,
                  null_count_min, null_count_max);
            } else if (target.equals(SEQCARDINALITY) ||
                           target.equals(NONSEQCARDINALITY)) {
              String maxoccur_attr = ((Element)root).getAttribute("maxOccurs");
              int max_occur = -1;
              if (maxoccur_attr != null && !maxoccur_attr.equals("")) {
          	    max_occur = Integer.parseInt(maxoccur_attr);
              }
              String minoccur_attr = ((Element)root).getAttribute("minOccurs");
              int min_occur = -1;
              if (minoccur_attr != null && !minoccur_attr.equals("")) {
                min_occur = Integer.parseInt(minoccur_attr);
              }
              NodeList grp_node = ((Element)root).getElementsByTagName("group");
              String group = null;
              if (grp_node != null && grp_node.getLength() > 0) {
                group = ((Element)grp_node.item(0)).getAttribute("xpath");
              }
              String new_only_attr = ((Element)root).getAttribute("newOnly");
              boolean new_only = false;
              if (new_only_attr != null && new_only_attr.equals("true")) {
                new_only = true;
              }
              new_constraint = new CardinalityConstraint(
                  name, item_target, item_identifier,
                  dimension, selector, field,
                  group,
                  evaluation_window, slide_size,
                  applicability, new_only, max_occur, min_occur);
            } else if (target.equals(TRANSITION)) {
              NodeList ve_node =
            	  ((Element)root).getElementsByTagName("valueEvolution");
              if (ve_node != null && ve_node.getLength() > 0) {
                String direction =
                    ((Element)ve_node.item(0)).getAttribute("direction");
                new_constraint = new TransitionConstraint(
                    name, item_target, item_identifier,
                    dimension, selector, field, evaluation_window, slide_size,
                    applicability,  direction);
              } else {
                NodeList vp_nodes =
                	((Element)root).getElementsByTagName("valuePair");
                Vector<ValuePair> value_pairs = new Vector<ValuePair>();
                for (int k = 0; k < vp_nodes.getLength(); ++k) {
                  Element tmp_element = (Element)vp_nodes.item(k);
                  ValuePair new_vp = new ValuePair(
                      tmp_element.getElementsByTagName("old").item(0).
                          getTextContent(),
                      tmp_element.getElementsByTagName("new").item(0).
                          getTextContent());
                  value_pairs.add(new_vp);
                }
                new_constraint = new TransitionConstraint(
                    name, item_target, item_identifier,
                    dimension, selector, field, evaluation_window, slide_size,
                    applicability,  value_pairs);
              }
            } else if (target.equals(NONSEQKEYREF)) {
              String ref_selector = null;
              String ref_field = null;
              String refer = ((Element)root).getAttribute("refer");
              String conventional_constraint =
            	  ((Element)root).getAttribute("conventionalConstraint");
              if (conventional_constraint == null ||
          	          conventional_constraint.equals("")) {
                conventional_constraint = null;
                if (refer == null || refer.equals("")) {
                  System.err.println("nonseqkeyref definition of " +
                                     name +
                		             "doesn't have " +
                                     "conventionalIdentifier and refer");
                  System.exit(4);
                }
                Vector<String> selector_field =
                	  ConvertConventionalIdentifier(refer, schema_doc_);
                if (selector_field == null) {
                  System.exit(3);
                }
                ref_selector = selector_field.get(0);
                ref_field = selector_field.get(1);
              } else {
                Vector<String> selector_field =
            	    ConvertConventionalIdentifier(conventional_constraint,
                                                  schema_doc_);
                if (selector_field == null) {
                  System.exit(3);
                }
                selector = selector_field.get(0);
                field = selector_field.get(1);
                String tmp_ref = selector_field.get(2);
                Vector<String> ref_selector_field =
                    ConvertConventionalIdentifier(tmp_ref, schema_doc_);
                ref_selector = ref_selector_field.get(0);
                ref_field = ref_selector_field.get(1);
              }
              new_constraint = new ReferentialIntegrityConstraint(
                  name, item_target, item_identifier,
                  dimension, selector, field, evaluation_window, slide_size,
                  applicability, conventional_constraint,
                  refer, ref_selector, ref_field);
            }
            results.add(new_constraint);
          }
        }
      }
      return;
    }
    NodeList child_nodes = anno_root.getChildNodes();
    for (int i = 0; i < child_nodes.getLength(); ++i) {
      Node temp_node = child_nodes.item(i);
      if (temp_node.getNodeType() == Node.ELEMENT_NODE) {
        ExtractConstraints((Element)temp_node, target, results);
      }
    }
  }
  
  
  /**
   * Converting the orignal selector and field into an internally used
   * format.
   */
  public static String PrepareXPathString(String selector, String field) {
    if (selector == null || selector.equals("")) {
      return field;
    } else {
      return selector + "/" + field;
    }
  }
  
  
  /**
   * Using XPath query to retrieve the candidate nodes to be examined for
   * the temporal constraints from the temporal document. The applicability
   * may apply if defined
   * @param temporal_doc the temporal document where the candidate nodes are
   * requested from
   * @param xpath_query the XPath query used to identify the candidate nodes
   * @param applicability the time period where the candidate nodes should be
   * retrieved from
   * @return the vector of candidate nodes
   */
  public static Vector<String> GetCandidateNodes(Document temporal_doc,
                                                 String xpath_query,
                                                 String target,
                                                 String identifier,
                                                 Applicability applicability) {
    Vector<String> result_nodes = new Vector<String>();
    if (xpath_query.equals(".")) {
      result_nodes.add(
          ((Node)temporal_doc.getDocumentElement()).getNodeValue());
      return result_nodes; 
    }
    try {
      // case 1: target is empty. This should only happen in refkey check,
      // as to find all the valid keys.
      if (target == null) {
        XPath query_xpath = XPathFactory.newInstance().newXPath();
        XPathExpression expr_query = query_xpath.compile(xpath_query);
        NodeList all_nodes = (NodeList)expr_query.evaluate(
            temporal_doc, XPathConstants.NODESET);
        for (int i = 0; i < all_nodes.getLength(); ++i) {
          Node tmp_node = all_nodes.item(i);
          if (tmp_node.getNodeType() == Node.ATTRIBUTE_NODE) {
            result_nodes.add(tmp_node.getNodeValue());
          } else if (tmp_node.getNodeType() == Node.ELEMENT_NODE) {
            result_nodes.add(tmp_node.getTextContent());
          }
        }
        return result_nodes;
      }
      // case 2: The identifier is empty. This should only happen when
      // unique/key constraint is on the same element/attribute as the
      // original key. In such a case, the versions are grouped by
      // the _RepItem element.
      if (identifier == null) {
        // the first step is the retrieve all the repitems as the true
        // identifiers
        String all_id_query = "//" + target + "_RepItem";
        XPath all_id_xpath = XPathFactory.newInstance().newXPath();
        XPathExpression expr_all_ids = all_id_xpath.compile(all_id_query);
        //long st = System.currentTimeMillis();
        NodeList all_id_nodes = (NodeList)expr_all_ids.evaluate(
            temporal_doc, XPathConstants.NODESET);
        //long et = System.currentTimeMillis();
        //System.out.println("Get all ids: " + (et - st) + " msec.");
        Map<Node, Vector<String>> id_node_map =
            new HashMap<Node, Vector<String>>();
        for (int i = 0; i < all_id_nodes.getLength(); ++i) {
          id_node_map.put(all_id_nodes.item(i), new Vector<String>());
        }
        // the second step is then find all the needed values and back-track
        // to the repitem to be grouped
        String all_item_query = "//" + target + "/" + xpath_query;
        XPath all_item_xpath = XPathFactory.newInstance().newXPath();
        XPathExpression expr_all_items = all_item_xpath.compile(all_item_query);
        //long st2 = System.currentTimeMillis();
        NodeList all_item_nodes = (NodeList)expr_all_items.evaluate(
            temporal_doc, XPathConstants.NODESET);
        //long et2 = System.currentTimeMillis();
        //System.out.println("Get all nodes: " + (et2 - st2) + " msec.");
        for (int i = 0; i < all_item_nodes.getLength(); ++i) {
      	  Node can_node = all_item_nodes.item(i);
          Node tmp_can_node = can_node;
          String value = "";
          if (can_node.getNodeType() == Node.ATTRIBUTE_NODE) {
            value = can_node.getNodeValue();
          } else if (can_node.getNodeType() == Node.ELEMENT_NODE) {
            value = can_node.getTextContent();
          }
          Node par_node = null;
          String start_time = null;
          String end_time = null;
          // loop to get to the repitem node and timestamps
          // if applicability applies
          do {
            if (tmp_can_node.getNodeType() == Node.ELEMENT_NODE) {
              par_node = tmp_can_node.getParentNode();
            } else if (tmp_can_node.getNodeType() == Node.ATTRIBUTE_NODE) {
              par_node = ((Attr)tmp_can_node).getOwnerElement();
            } else if (tmp_can_node.getNodeType() == Node.DOCUMENT_NODE){
          	  break;
            } else {
              System.err.println("Invalid type of node.");
              System.exit(1);
            }
            if (par_node.getNodeName().endsWith(
                    TXSchemaConstants.VERSION_SUFFIX)) {
              start_time = ((Element)par_node).getAttribute("begin");
              end_time = ((Element)par_node).getAttribute("end");
            } else if (par_node.getNodeName().equals(target + "_RepItem")) {
              if (applicability != null) {
                if (start_time == null || end_time == null) {
                  System.err.println("start time or end time is invalid " +
                                     "when applicability is present");
                }
                if (applicability.IsIntersect(start_time, end_time)) {
                  Vector<String> values = id_node_map.get(par_node);
                  if (values == null) {
                    System.err.println(
                        "Serious problem: parent node is not a valid repitem");
                    System.exit(1);
                  }
                  values.add(value);
                }
              } else {
                Vector<String> values = id_node_map.get(par_node);
                if (values == null) {
                  System.err.println(
                      "Serious problem: parent node is not a valid repitem");
                  System.exit(1);
                }
                values.add(value);
              }
            }
            tmp_can_node = par_node;
          } while (par_node != null);
          if (par_node == null) {
            System.err.println("Can't ever identify the timestamp");
            System.exit(2);
          }
        }
        Set<Node> key_nodes = id_node_map.keySet();
        int num_keys = 0;
        for (Node n : key_nodes) {
          Vector<String> tmp_values = id_node_map.get(n);
          if (tmp_values == null) {
            System.err.println("Can't fetch proper result for node");
            System.exit(1);
          }
          for (String value : tmp_values) {
            result_nodes.add(num_keys + TXSchemaConstants.DELIMITER + value);
          }
          ++num_keys;
        }
      } else {
        // case 3: The normal case.
        String all_ids_query = "//" + target + "_RepItem";
        XPath all_ids_xpath = XPathFactory.newInstance().newXPath();
        XPathExpression expr_all_ids =
            all_ids_xpath.compile(all_ids_query);
        NodeList all_ids_nodes = (NodeList)expr_all_ids.evaluate(
            temporal_doc, XPathConstants.NODESET);
        // each RepItem should correspond to one identifier
        Map<String, Map<String, Vector<String>>> repitem_result_map =
        	new HashMap<String, Map<String, Vector<String>>>();
        for (int i = 0; i < all_ids_nodes.getLength(); ++i) {
          repitem_result_map.put(
              all_ids_nodes.item(i).hashCode() + "",
              new TreeMap<String, Vector<String>>());
        }
        XPath identifier_xpath = XPathFactory.newInstance().newXPath();
        XPathExpression expr_identifier =
            identifier_xpath.compile("//" + target + "/" + identifier);
        NodeList identifier_nodes = (NodeList)expr_identifier.evaluate(
            temporal_doc, XPathConstants.NODESET);
        Map<String, String> keynode_value_map = new HashMap<String, String>();
        for (int i = 0; i < identifier_nodes.getLength(); ++i) {
          Node can_id_node = identifier_nodes.item(i);
          Node tmp_can_node = can_id_node;
          String key_value = "";
          Node key_node = null;
          if (can_id_node.getNodeType() == Node.ATTRIBUTE_NODE) {
            key_value = can_id_node.getNodeValue();
          } else if (can_id_node.getNodeType() == Node.ELEMENT_NODE) {
        	key_value = can_id_node.getTextContent();  
          }
          Node par_node = null;
          do {
            if (tmp_can_node.getNodeType() == Node.ELEMENT_NODE) {
              par_node = tmp_can_node.getParentNode();
            } else if (tmp_can_node.getNodeType() == Node.ATTRIBUTE_NODE) {
              par_node = ((Attr)tmp_can_node).getOwnerElement();
            } else if (tmp_can_node.getNodeType() == Node.DOCUMENT_NODE){
          	  break;
            } else {
              System.err.println("Invalid type of node.");
              System.exit(1);
            }
            if (par_node.getNodeName().equals(target)) {
              key_node = par_node;
            } else if (par_node.getNodeName().equals(target + "_RepItem")) {
              if (!repitem_result_map.containsKey(par_node.hashCode() + "")) {
            	System.err.println(
                    "Serious problem: parent node is not a valid repitem");
                System.exit(1);
              }
              Map<String, Vector<String>> tmp_map =
            	  repitem_result_map.get(par_node.hashCode() + "");
              if (!tmp_map.containsKey(key_node.hashCode() + "")) {
            	tmp_map.put(
                    key_node.hashCode() + "",
                    new Vector<String>());
              }
              if (!keynode_value_map.containsKey(key_node.hashCode() + "")) {
                keynode_value_map.put(key_node.hashCode() + "", key_value);
              }
            }
            tmp_can_node = par_node;
          } while (par_node != null);
          if (par_node == null) {
            System.err.println("Can't ever identify the timestamp");
            System.exit(2);
          }
        }

        XPath result_xpath = XPathFactory.newInstance().newXPath();
        String result_xpath_query = xpath_query;
        if (!xpath_query.startsWith("//" + target)) {
          result_xpath_query = "//" + target + "//" + xpath_query;
        }
        XPathExpression expr_result =
            result_xpath.compile(result_xpath_query);
        NodeList all_result_nodes = (NodeList)expr_result.evaluate(
                temporal_doc, XPathConstants.NODESET);
        for (int i = 0; i < all_result_nodes.getLength(); ++i) {
          Node can_node = all_result_nodes.item(i);
          Node tmp_can_node = can_node;
          String value = "";
          if (can_node.getNodeType() == Node.ATTRIBUTE_NODE) {
            value = can_node.getNodeValue();
          } else if (can_node.getNodeType() == Node.ELEMENT_NODE) {
            value = can_node.getTextContent();
          }
          Node par_node = null;
          String start_time = null;
          String end_time = null;
          Node key_node = null;
          // loop to get to the repitem node and timestamps
          // if applicability applies
          do {
            if (tmp_can_node.getNodeType() == Node.ELEMENT_NODE) {
              par_node = tmp_can_node.getParentNode();
            } else if (tmp_can_node.getNodeType() == Node.ATTRIBUTE_NODE) {
              par_node = ((Attr)tmp_can_node).getOwnerElement();
            } else if (tmp_can_node.getNodeType() == Node.DOCUMENT_NODE){
          	  break;
            } else {
              System.err.println("Invalid type of node.");
              System.exit(1);
            }
            if (par_node.getNodeName().endsWith(
                    TXSchemaConstants.VERSION_SUFFIX)) {
              start_time = ((Element)par_node).getAttribute("begin");
              end_time = ((Element)par_node).getAttribute("end");
            } else if (par_node.getNodeName().equals(target)) {
              key_node = par_node;
            } else if (par_node.getNodeName().equals(target + "_RepItem")) {
              if (applicability != null) {
                if (start_time == null || end_time == null) {
                  System.err.println("start time or end time is invalid " +
                                     "when applicability is present");
                }
                if (applicability.IsIntersect(start_time, end_time)) {
                  Map<String, Vector<String>> target_map =
                	  repitem_result_map.get(par_node.hashCode() + "");
                  if (target_map == null) {
                    System.err.println(
                        "Serious problem: parent node is not a valid repitem");
                    System.exit(1);
                  }
                  Vector<String> values =
                	  target_map.get(key_node.hashCode() + "");
                  values.add(value);
                }
              } else {
            	Map<String, Vector<String>> target_map =
                 repitem_result_map.get(par_node.hashCode() + "");
                if (target_map == null) {
                  System.err.println(
                      "Serious problem: parent node is not a valid repitem");
                  System.exit(1);
                }
                Vector<String> values =
                	target_map.get(key_node.hashCode() + "");
                values.add(value);
              }
            }
            tmp_can_node = par_node;
          } while (par_node != null);
          if (par_node == null) {
            System.err.println("Can't ever identify the timestamp");
            System.exit(2);
          }
        }
        Set<String> key_nodes = repitem_result_map.keySet();
        for (String n : key_nodes) {
          Map<String, Vector<String>> tmp_result_map = 
        	  repitem_result_map.get(n);
          Set<String> id_nodes = tmp_result_map.keySet();
          for (String m : id_nodes) {
            if (!keynode_value_map.containsKey(m)) {
              System.err.println("can't find value for key " + m);
              System.exit(1);
            }
            String key = keynode_value_map.get(m);
            Vector<String> values = tmp_result_map.get(m);
            for (String value : values) {
              result_nodes.add(
                  key + TXSchemaConstants.DELIMITER + value); 	
            }
          }
        }
      }
      return result_nodes;
    } catch (XPathExpressionException e) {
      e.printStackTrace();
      return null;
    }
  }
  
  
  // wrapper of the identify constraints (6.1) 
  protected boolean CheckIdentityConstraint(Document temporal_doc) {
    long start_time = System.currentTimeMillis();
    boolean nonseq_unique = IdentityConstraint.CheckNonSeqUnique(
        temporal_doc, constraint_map_.get(NONSEQUNIQUE));
    boolean nonseq_key = IdentityConstraint.CheckNonSeqKey(
        temporal_doc, constraint_map_.get(NONSEQKEY));
    boolean unique_null_restricted =
        IdentityConstraint.CheckUniqueNullRestricted(
             temporal_doc, constraint_map_.get(UNIQUENULLRESTRICTED));
    long end_time = System.currentTimeMillis();
    System.out.println("Checking Identity Constraints: " +
                       ((end_time - start_time)/1000.0) + " sec.");
    return nonseq_unique && nonseq_key && unique_null_restricted;
  }
  
  
  // wrapper of the cardinality constraints (6.3)
  protected boolean CheckCardinalityConstraint(Document temporal_doc) {
    long start_time = System.currentTimeMillis();
    /*
    boolean seq_val = CardinalityConstraint.CheckSeqCardinality(
        temporal_doc, constraint_map_.get(SEQCARDINALITY));
        */
    boolean nonseq_val = CardinalityConstraint.CheckNonSeqCardinality(
        temporal_doc, constraint_map_.get(NONSEQCARDINALITY)); 
    long end_time = System.currentTimeMillis();
    System.out.println("Checking Cardinality Constraints: " +
                       ((end_time - start_time)/1000.0) + " sec.");
    return nonseq_val;
  }
  
  
  protected boolean CheckRefIntegrityConstraint(Document temporal_doc) {
    long start_time = System.currentTimeMillis();
    boolean ref_val = ReferentialIntegrityConstraint.CheckNonSeqKeyRef(
        temporal_doc, schema_doc_, constraint_map_.get(NONSEQKEYREF));
    long end_time = System.currentTimeMillis();
    System.out.println("Checking ReferentialIntegrity Constraints: " +
                       ((end_time - start_time)/1000.0) + " sec.");
    return ref_val;
  }
  
  
  protected boolean CheckTransitionConstraint(Document temporal_doc) {
    long start_time = System.currentTimeMillis();
    boolean tran_val = TransitionConstraint.CheckTransitionConstraint(
        temporal_doc, constraint_map_.get(TRANSITION));
    long end_time = System.currentTimeMillis();
    System.out.println("Checking Datatype Constraints: " +
                       ((end_time - start_time)/1000.0) + " sec.");
    return tran_val;
  }
  
  
  public boolean CheckSequencedConstraints(
      String temporal_data_dir, String temporal_document_name,
      String temporalSchema, String conventional_schema) {
    return SequencedConstraintChecker.CheckSequencedConstraints(
        temporal_data_dir, temporal_document_name,
    	temporalSchema, conventional_schema, schema_doc_);
  }

  
  /**
   * Perform the validation
   * @return <code>TRUE</code> if validated; <code>FALSE</code> otherwise  
   */
  public boolean PerformValidation(
	  String temporalDataDir, String temporalDataFile,
      String temporalSchema, String conventionalSchema) {
    ExtractConstraint(annotation_doc_);
    boolean seq_val = CheckSequencedConstraints(
        temporalDataDir, temporalDataFile,
    	temporalSchema, conventionalSchema);
	boolean identity_val = CheckIdentityConstraint(temporal_doc_);
	boolean cardinality_val = CheckCardinalityConstraint(temporal_doc_);
	boolean trans_val = CheckTransitionConstraint(temporal_doc_);
	boolean refkey_val = CheckRefIntegrityConstraint(temporal_doc_);
    return seq_val && identity_val && cardinality_val && trans_val &&
        refkey_val;
  }
}
