package cs.arizona.tau.xml.temporalconstraint;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import cs.arizona.tau.xml.TXSchemaConstants;

/**
 * The <code>CardinalityConstraint</code> sub-class extending
 * <code>Constraint</code>
 * @author ruizhang
 *
 */
public class CardinalityConstraint extends Constraint {
  protected int max_occur_;
  protected int min_occur_ = 0;
  // only used for nonSeqCardinality
  protected boolean new_only_ = false;
  protected String group_;

  /**
   * Constructing the <code>CardinalityConstraint</code> instance with its
   * specific attributes, which are <code>new_only</code>,
   * <code>max_occur</code> and <code>min_occur</code>.
   * @param name
   * @param dimension
   * @param selector
   * @param field
   * @param group
   * @param evaluation_window
   * @param slide_size
   * @param applicability
   * @param new_only
   * @param max_occur
   * @param min_occur
   */
  public CardinalityConstraint(String name, String item_target,
                               String item_identifier,
                               String dimension, String selector,
                               String field, String group,
                               int evaluation_window, int slide_size,
                               Applicability applicability,
                               boolean new_only,
                               int max_occur, int min_occur) {
    super(name, item_target, item_identifier,
    	  dimension, selector, field, evaluation_window, slide_size,
          applicability);
    max_occur_ = max_occur;
    min_occur_ = min_occur;
    new_only_ = new_only;
    group_ = group;
  }
    
  
  public int getMaxOccur() {
    return max_occur_;
  }
  
  
  public int getMinOccur() {
    return min_occur_;
  }
  
  
  public String getGroup() {
    return group_;
  }
  
  
  public boolean getNewOnly() {
    return new_only_;
  }
  
  
  /**
   * Checking constraint 6.3 (1) the <code>NonSeqCardinality</code>
   * constraint
   * @param temporal_doc the temporal document
   * @param constraints the list of defined <code>NonSeqCardinality</code>
   * constraints
   */
  public static boolean CheckNonSeqCardinality(
      Document temporal_doc, Vector<Constraint> constraints) {
    if (constraints == null) {
      return true;
    }
    for (Constraint c : constraints) {
      for (Applicability app : c.eval_windows_) {
        boolean use_group = false;
        if (((CardinalityConstraint)c).getGroup() != null) {
          use_group = true;
        } else {
          use_group = false;
        }
        String xpath_query = "";
        if (use_group) {
          xpath_query = TemporalConstraintValidator.PrepareXPathString(
              c.getSelector() + "//" + ((CardinalityConstraint)c).getGroup(),
              c.getField());
        } else {
          xpath_query = TemporalConstraintValidator.PrepareXPathString(
              c.getSelector(), c.getField());
        }
        //xpath_query += "|" +  c.getItemIdentifier();
        Vector<String> result_nodes =
            TemporalConstraintValidator.GetCandidateNodes(
                temporal_doc, xpath_query, c.getItemTarget(),
                c.getItemIdentifier(), app);
        if (result_nodes == null) {
          return false;
        }
        // regular case
        if (!((CardinalityConstraint)c).getNewOnly()) {
          Map<String, Set<String>> value_map =
              new HashMap<String, Set<String>>();
          for (String temp_node : result_nodes) {
            String[] key_value = temp_node.split(TXSchemaConstants.DELIMITER);
            if (value_map.containsKey(key_value[0])) {
              Set<String> temp_distinct_items = value_map.get(key_value[0]);
              temp_distinct_items.add(key_value[1]);
            } else {
              Set<String> distinct_items = new TreeSet<String>();
              distinct_items.add(key_value[1]);
              value_map.put(key_value[0], distinct_items);
            }
          }
          Set<String> key_set = value_map.keySet();
          for (String key : key_set) {
            Set<String> distinct_items = value_map.get(key);
            //System.out.println("key " + key + " val " + distinct_items.size());
            if (distinct_items.size() >
                ((CardinalityConstraint)c).getMaxOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() +  "] non-seq cardinality violation" +
                  " for item [" + key + "]: " +
                  "max is " + ((CardinalityConstraint)c).getMaxOccur() +
                  " actual is " + distinct_items.size() + "[" +
                  distinct_items.toString() + "]");
              return false;
            } else if (distinct_items.size() <
                       ((CardinalityConstraint)c).getMinOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() +  "] non-seq cardinality violation" +
                  " for item [" + key + "]: " +
                  "min is " + ((CardinalityConstraint)c).getMinOccur() +
                  " actual is " + distinct_items.size() + "[" +
                  distinct_items.toString() + "]");
              return false;
            }
          }
        } else {
          // newOnly case
          Map<String, Vector<String>> value_map =
              new HashMap<String, Vector<String>>();
          for (String temp_node : result_nodes) {
            String[] key_value = temp_node.split(TXSchemaConstants.DELIMITER);
            if (value_map.containsKey(key_value[0])) {
              Vector<String> temp_items = value_map.get(key_value[0]);
              temp_items.add(key_value[1]);
            } else {
              Vector<String> all_items = new Vector<String>();
              all_items.add(key_value[1]);
              value_map.put(key_value[0], all_items);
            }
          }
          Set<String> key_set = value_map.keySet();
          for (String key : key_set) {
            Vector<String> all_items = value_map.get(key);        	
            int changed_times = 0;
            String prev_node_name = "";
            for (int i = 0; i < all_items.size(); ++i) {
              String curr_node_name = all_items.get(i);
              if (i == 0) {
                prev_node_name = curr_node_name;
                continue;
              }
              if (!curr_node_name.equals(prev_node_name)) {
                ++changed_times;
                prev_node_name = curr_node_name;
              }
            }
            if (changed_times > ((CardinalityConstraint)c).getMaxOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() +  "] non-seq cardinality violation " +
                  "(new only): max is " +
                  ((CardinalityConstraint)c).getMaxOccur() +
                  " actual is " + changed_times);
                return false;
            } else if (changed_times <
                       ((CardinalityConstraint)c).getMinOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() +  "] non-seq cardinality violation " +
                  "(new only): min is " +
                  ((CardinalityConstraint)c).getMinOccur() +
                  " actual is " + changed_times);
              return false;
            }
          }
        }
      }
    }
    return true;
  }
  
  
  /**
   * Checking constraint 6.3 (2) the <code>SeqCardinality</code>
   * constraint
   * @param temporal_doc the temporal document
   * @param constraints the list of defined <code>SeqCardinality</code>
   * constraints
   */
  public static boolean CheckSeqCardinality(
      Document temporal_doc, Vector<Constraint> constraints) {
    if (constraints == null) {
      return true;
    }
    XPath xpath = XPathFactory.newInstance().newXPath();
    for (Constraint c : constraints) {
      for (Applicability app : c.eval_windows_) {
        boolean use_group = false;
        String xpath_query = null;
        if (((CardinalityConstraint)c).getGroup() != null) {
          use_group = true;
        } else {
          use_group = false;
        }
        if (!use_group) {
          xpath_query = TemporalConstraintValidator.PrepareXPathString(
              c.getSelector(), c.getField());
        } else {
          xpath_query = TemporalConstraintValidator.PrepareXPathString(
              c.getSelector() + "/" + ((CardinalityConstraint)c).getGroup(),
              c.getField());
        }
        //xpath_query += "|" + c.getItemIdentifier();
        Vector<String> result_nodes =
            TemporalConstraintValidator.GetCandidateNodes(
                temporal_doc, xpath_query,
                c.getItemTarget(), c.getItemIdentifier(), app);
        Map<String, Vector<String>> value_map =
            new HashMap<String, Vector<String>>();
        for (String tmp_element : result_nodes) {
          String[] key_value = tmp_element.split(TXSchemaConstants.DELIMITER);
          if (value_map.containsKey(key_value[0])) {
            Vector<String> temp_items = value_map.get(key_value[0]);
            temp_items.add(key_value[1]);
          } else {
            Vector<String> all_items = new Vector<String>();
            all_items.add(key_value[1]);
            value_map.put(key_value[0], all_items);
          }
        }
        Set<String> key_set = value_map.keySet();
        for (String key : key_set) {
          if (!use_group) {
            Set<String> value_set = new TreeSet<String>(value_map.get(key));
            if (value_set.size() >
                ((CardinalityConstraint)c).getMaxOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() + "] seq cardinality violation: " +
                  "max is " + ((CardinalityConstraint)c).getMaxOccur() +
                  " actual is " + value_set.size() + "[" +
                  value_set.toString() + "]");
              return false;
            } else if (value_set.size() <
                       ((CardinalityConstraint)c).getMinOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() + "] non-seq cardinality violation: " +
                  "min is " + ((CardinalityConstraint)c).getMinOccur() +
                  " actual is " + value_set.size() + "[" +
                  value_set.toString() + "]");
              return false;
            }
          } else {
            Vector<String> value_set = value_map.get(key);
            if (value_set.size() >
                ((CardinalityConstraint)c).getMaxOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() + "] seq cardinality violation: " +
                  "max is " + ((CardinalityConstraint)c).getMaxOccur() +
                  " actual is " + value_set.size() + "[" +
                  value_set.toString() + "]");
              return false;
            } else if (value_set.size() <
                       ((CardinalityConstraint)c).getMinOccur()) {
              System.err.println(
                  "In period " + app.toString() +
                  " [" + c.getName() + "] non-seq cardinality violation: " +
                  "min is " + ((CardinalityConstraint)c).getMinOccur() +
                  " actual is " + value_set.size() + "[" +
                  value_set.toString() + "]");
              return false;
            }
          }
        }
      }
    }
    return true;
  }
}
