/**
 * 
 */
package cs.arizona.tau.time;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.text.SimpleDateFormat;
import java.text.ParseException;

import org.w3c.dom.Element;
import org.w3c.dom.Document;

import cs.arizona.tau.time.ITimePeriod;
import cs.arizona.util.ValidatorProperties;
//import cs.arizona.tau.xml.Common;

/**
 * @author spjoshi
 *
 */
public class TimePeriod implements ITimePeriod {

	/**
	 * ValidatorProperties
	 */
	private Date beginDate;
	private Date endDate;
	
	/**
	 * 
	 */
	public TimePeriod() {
		super();
	}

	//TODO: entire method copied from Common class (fix circular dependency).
	public String formatDate(Date a){
		String result = String.format("%04d%02d%02d",   (a.getYear() + 1900), (a.getMonth() + 1), a.getDate());
		return result;
	}
	
	
	public String toString(){
		return formatDate(beginDate) + " to " +  formatDate(endDate);
	}

	/**
	 * 
	 */
	public TimePeriod(Date beginDate, Date endDate) {
		super();
		this.beginDate = beginDate;
		this.endDate = endDate;
	}
	
	public TimePeriod(String strBeginDate, String strEndDate){
		try{
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
			this.beginDate = sdf.parse(strBeginDate);
			this.endDate = sdf.parse(strEndDate);
		}catch (ParseException pe){
			throw new RuntimeException(pe);
		}
	}

	/**
	 * 
	 */
	public TimePeriod(ITimePeriod tp) {
		super();
		this.beginDate = tp.getBeginDate();
		this.endDate = tp.getEndDate();
	}
	
	public TimePeriod(Element xmlPeriodElement){
		try{
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
			beginDate = sdf.parse(xmlPeriodElement.getAttribute("begin"));
			endDate = sdf.parse(xmlPeriodElement.getAttribute("end"));
		}catch (ParseException pe){
			throw new RuntimeException(pe);
		}
	}
	

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#setBeginDate(java.util.Date)
	 */
	public void setBeginDate(Date beginDate) {
		this.beginDate = beginDate;	
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#setEndDate(java.util.Date)
	 */
	public void setEndDate(Date endDate) {
		this.endDate = endDate;	
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#getBeginDate()
	 */
	public Date getBeginDate() {
		return beginDate;
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#getEndDate()
	 */
	public Date getEndDate() {
		return endDate;
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#toXML()
	 */
	public Element toXML(int timeDimension, int timeRepresentation, Document doc) {
		Element xmlElement;
		ValidatorProperties vp = ValidatorProperties.getInstance();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
		if (timeDimension == ITimePeriod.VALID_TIME){
			if (timeRepresentation == ITimePeriod.EXTENT_REP){
				xmlElement = doc.createElement(vp.getProperty("TVSchemaNameAlias") + "timestamp_ValidExtent");
			}else{
				xmlElement = doc.createElement(vp.getProperty("TVSchemaNameAlias") + "timestamp_ValidStep");
			}
		}else{
			if (timeRepresentation == ITimePeriod.EXTENT_REP){
				xmlElement = doc.createElement(vp.getProperty("TVSchemaNameAlias") + "timestamp_TransExtent");
			}else{
				xmlElement = doc.createElement(vp.getProperty("TVSchemaNameAlias") + "timestamp_TransStep");
			}
		}
		xmlElement.setAttribute("begin",sdf.format(beginDate));
		xmlElement.setAttribute("end",sdf.format(endDate));
		return xmlElement;
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#getRelationship(cs.arizona.tau.time.ITimePeriod)
	 */
	public int getRelationship(ITime tp1) {
		/*
		 * |------------------|
		 * Refer conditions from Dr. Snodgrass' book. Page number - 92
		 * |------------------|
		 */
		ITimePeriod tp = (ITimePeriod)tp1;
		Date b1,b2,e1,e2;
		b1=this.beginDate; b2=tp.getBeginDate() ; e1=this.endDate; e2=tp.getEndDate();
		
		if ((b1.equals(b2)) && (e1.equals(e2))){
			return A_EQUALS_B;
		}				

		if ((b1.before(b2)) && (e1.before(b2))){
			return A_BEFORE_B; 
		}		
	
		if ((b2.before(b1)) && (e2.before(b1))){
			return B_BEFORE_A; 
		}		

		if ((b1.before(b2)) && (e1.equals(b2))){
			return A_MEETS_B; 
		}		
		if ((b2.before(b1)) && (e2.equals(b1))){
			return B_MEETS_A; 
		}		
		
		if ((b1.before(b2)) && (e1.before(e2)) && (e1.after(b2))){
			return A_OVERLAPS_B; 
		}		

		if ((b2.before(b1)) && (e2.before(e1)) && (e2.after(b1))){
			return B_OVERLAPS_A;
		}
		
		if ((b1.equals(b2)) && (e1.before(e2))){
			return A_STARTS_B;
		}
		
		if ((b2.equals(b1)) && (e2.before(e1))){
			return B_STARTS_A;
		}
		
		if ((b2.before(b1)) && (e1.equals(e2))){
			return A_FINISHES_B;
		}
		
		if ((b1.before(b2)) && (e1.equals(e2))){
			return B_FINISHES_A;
		}
		
		if ((b2.before(b1)) && (b1.before(e2)) && (e1.before(e2))){
			return A_DURING_B;
		}

		if ((b1.before(b2)) && (b2.before(e1)) && (e2.before(e1))){
			return B_DURING_A;
		}
		return NO_OVERLAP;
	}

	/* (non-Javadoc)
	 * @see cs.arizona.tau.time.ITimePeriod#split(cs.arizona.tau.time.ITimePeriod)
	 */
	public Iterator split(ITimePeriod tp) {
		ArrayList temporalElement = new ArrayList();
		Date s1,s2,e1,e2;
		s1=this.beginDate; s2=tp.getBeginDate() ; e1=this.endDate; e2=tp.getEndDate();
		ITimePeriod tp1,tp2,tp3;
		switch (getRelationship(tp)){
		case A_OVERLAPS_B	:	
					tp1 = new TimePeriod(s1,s2);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s2,e1);
				 	temporalElement.add(tp2);
				 	tp3 = new TimePeriod(e1,e2);
				 	temporalElement.add(tp3);
				 	break;
		case B_OVERLAPS_A	:			
					tp1 = new TimePeriod(s2,s1);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s1,e2);
				 	temporalElement.add(tp2);
				 	tp3 = new TimePeriod(e2,e1);
				 	temporalElement.add(tp3);
				 	break;
		case A_EQUALS_B	:	
					tp1 = new TimePeriod(s1,e1);
					temporalElement.add(tp1);
					break;
		case A_STARTS_B	:	
					tp1 = new TimePeriod(s1,e1);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(e1,e2);
				 	temporalElement.add(tp2);
				 	break;
		case B_STARTS_A	:
					tp1 = new TimePeriod(s2,e2);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(e2,e1);
				 	temporalElement.add(tp2);
				 	break;
		case A_FINISHES_B	:
					tp1 = new TimePeriod(s2,s1);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s1,e1);
				 	temporalElement.add(tp2);
				 	break;
		case B_FINISHES_A	:
					tp1 = new TimePeriod(s1,s2);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s2,e2);
				 	temporalElement.add(tp2);
				 	break;
		case A_DURING_B	:
					tp1 = new TimePeriod(s2,s1);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s1,e1);
				 	temporalElement.add(tp2);
				 	tp3 = new TimePeriod(e1,e2);
				 	temporalElement.add(tp3);
				 	break;
		case B_DURING_A	:
					tp1 = new TimePeriod(s1,s2);
				 	temporalElement.add(tp1);
				 	tp2 = new TimePeriod(s2,e2);
				 	temporalElement.add(tp2);
				 	tp3 = new TimePeriod(e2,e1);
				 	temporalElement.add(tp3);
				 	break;
		}
		return temporalElement.iterator();
	}

	public String getInterfaceName(){
		return "ITimePeriod";
	}
	
	public ITime cloneTime(){
		ITimePeriod tp = new TimePeriod();
		tp.setBeginDate((Date)this.beginDate.clone());
		tp.setEndDate((Date)this.endDate.clone());
		return tp;
	}
}
